#' Useful precomputations to quickly update GPC mean and variance
#'
#' @param model an object of class \code{\link[GPCsign]{gpcm}}.
#' @param integration.points \code{p*d} matrix of fixed integration points in the \code{X} space.
#'
#' @return a list with components:
#'  \item{\code{c.K}}{Vector equal to \eqn{t(c)K^{-1}} where \eqn{K^{-1}} is the inverse covariance matrix \code{invK} returned by \code{object} and \code{c} the unconditional covariance matrix  between \code{newdata} and design points \code{Xf}}
#'  \item{\code{lambda.intpoints}}{Vector equal to \eqn{K^{-1}c} where \eqn{K^{-1}} is the inverse covariance matrix \code{invK} returned by \code{object} and \code{c} the unconditional covariance matrix  between \code{newdata} and design points \code{Xf}}
#'  \item{\code{pn.intpoints}}{the (averaged) probability of class 1 at \code{integration.points}.}
#'  \item{\code{intpoints.oldmean}}{conditional mean matrix of the latent GP at \code{integration.points}}
#'  \item{\code{intpoints.oldmean}}{conditional variance vector of the latent GP at \code{integration.points}}
#' 
#' @import GPCsign
#' @export
#'
#' @references Menz, M., Munoz-Zuniga, M., Sinoquet, D. Estimation of simulation failure set with active learning based on Gaussian Process classifiers and random set theory (2023). \url{https://hal.science/hal-03848238}.
#' @references Bachoc, F., Helbert, C. & Picheny, V. Gaussian process optimization with failures: classification and convergence proof. \emph{J Glob Optim} \bold{78}, 483–506 (2020). \doi{10.1007/s10898-020-00920-0}.
#'
#' @author Morgane MENZ, Delphine SINOQUET, Miguel MUNOZ-ZUNIGA. Contributors: Naoual SERRAJI.
#'
#' @examples
#' #-------------------------------------------------------------------
#' #------------------- precomputeUpdateData ----------------------------
#' #-------------------------------------------------------------------
#'
#' ## 20-points DoE, and the corresponding response
#' d <- 2
#' nb_PX <- 20
#' x <- matrix(c(0.205293785978832, 0.0159983370750337,
#'               0.684774733109666, 0.125251417595962,
#'               0.787208786290006, 0.700475706055049,
#'               0.480507717105934, 0.359730889653793,
#'               0.543665267336735, 0.565974761807069,
#'               0.303412043992361, 0.471502352650857,
#'               0.839505250127309, 0.504914690245002,
#'               0.573294917143728, 0.784444726564573,
#'               0.291681289223421, 0.255053812451938,
#'               0.87233450888786, 0.947168337730927,
#'               0.648262257638515, 0.973264712407035,
#'               0.421877310273815, 0.0686662506387988,
#'               0.190976166753807, 0.810964668176754,
#'               0.918527262507395, 0.161973686467513,
#'               0.0188128700859558, 0.43522031347403,
#'               0.99902788789426, 0.655561821513544,
#'               0.741113863862512, 0.321050086076934,
#'               0.112003007565305, 0.616551317575545,
#'               0.383511473487687, 0.886611679106771,
#'               0.0749211435982952, 0.205805968972305),
#'             byrow = TRUE, ncol = d)
#' require(DiceKriging)
#' fx <- apply(x, 1, branin)
#' f <- ifelse(fx < 14, -1, 1)
#' Xf <- as.matrix(x)
#'
#'
#' ## gpcm object
#' require(GPCsign)
#' require(randtoolbox)
#' model <- gpcm(f, Xf, coef.m = -1.25, coef.cov = c(1.17,0.89))
#' 
#' nb.integration <- d * 100
#' integration.points <- sobol(n = nb.integration, dim = d, scrambling = 0)
#' integration.points <- scale(x = integration.points, center = model@X.mean, scale = model@X.std)
#' 
#' precalc.data <-  precomputeUpdateData(model = model, integration.points = integration.points)
#' 
precomputeUpdateData <- function(model, integration.points){

    colnames(integration.points) <- colnames(model@X)


    gpc_int <- predict(object=model, newdata=integration.points)
    intpoints.c <- gpc_int$c
    pn <- gpc_int$prob

    precalc.data <- list(5)
    precalc.data$intpoints.oldmean <- gpc_int$Zsimu_mean
    precalc.data$intpoints.oldsd <- sqrt(gpc_int$Zsimu_var)
    precalc.data$c.K <- crossprod(intpoints.c, model@invK)
    precalc.data$lambda.intpoints <- gpc_int$lambda
    precalc.data$pn.intpoints <- pn


    return(precalc.data)
}

