#' Fit enrollment model
#'
#' @description
#' Fits an exponential model to enrollment interarrival times and returns
#' posterior draws of the enrollment rate \eqn{\mu}.
#'
#' @details
#' Let \eqn{t_i} denote the interarrival time between consecutive enrollments.
#' The enrollment process is modeled using an exponential distribution with
#' rate parameter \eqn{\mu}, so that
#' \deqn{t_i \sim \mathrm{Exponential}(\mu),}
#' where \eqn{\mu > 0} represents the average enrollment rate. The corresponding
#' density is
#' \deqn{f(t) = \mu e^{-\mu t}, \quad t \ge 0.}
#'
#' Administrative censoring of enrollment is handled through the indicator
#' \code{status_enroll}, which follows the convention:
#' \itemize{
#'   \item \code{1}: the interarrival time is fully observed at \code{t_enroll}
#'     (density contribution);
#'   \item \code{0}: the interarrival time is administratively censored at
#'     \code{t_enroll} (survival contribution).
#' }
#'
#' @param status_enroll
#' Integer vector with values \code{0} or \code{1} indicating whether
#' the interarrival time is observed (\code{1}) or administratively censored (\code{0}).
#'
#' @param t_enroll
#' Numeric vector giving observed or censored interarrival times.
#' Must be non-negative and finite.
#'
#' @param hyperparams_enroll
#' A named list of prior hyperparameters for the enrollment model.
#' If it is empty (\code{list()}), default values
#' are assigned internally.
#' \itemize{
#'   \item \code{alpha_mu}: Shape parameter of the Gamma prior for the
#'     enrollment rate \eqn{\mu} (default: \eqn{0.1}).
#'   \item \code{beta_mu}: Rate parameter of the Gamma prior for the
#'     enrollment rate \eqn{\mu} (default: \eqn{0.1}).
#' }
#' No other hyperparameters are allowed.
#'
#' @param chains
#' Number of Markov chain Monte Carlo (MCMC) chains. Defaults to \code{4}.
#'
#' @param iter
#' Number of iterations per chain (including warmup). Defaults to \code{4000}.
#'
#' @param mc.cores Integer. Number of CPU cores to use when executing Markov
#'   chains in parallel via \code{\link[rstan]{sampling}}. Defaults to \code{1}.
#'   We recommend setting \code{mc.cores} to the maximum number of processors
#'   supported by the available hardware and memory, up to the number of
#'   chains.
#'
#' @param seed
#' Optional integer seed passed to \code{\link[rstan]{sampling}} for reproducibility. If \code{NULL},
#' 'Stan' generates a seed internally and results may differ across runs.
#' Defaults to \code{1}.
#'
#' @param refresh
#' Frequency of progress updates from \code{\link[rstan]{sampling}}. Set to \code{0} (default) to suppress output.
#'
#' @param warmup
#' Number of warmup (burn-in) iterations per chain. Must be strictly smaller
#' than \code{iter}. Defaults to \code{floor(iter / 2)}.
#'
#' @param control
#' A named list of sampler control parameters passed to
#' \code{\link[rstan]{sampling}}. Examples include \code{adapt_delta} and
#' \code{max_treedepth}.
#' Defaults to \code{list(adapt_delta = 0.95)}.
#'
#' @param return_fit
#' Logical; if \code{TRUE}, also return the underlying 'rstan' \code{stanfit}
#' object. Defaults to \code{FALSE}.
#'
#' @param quiet
#' Logical. If \code{TRUE} (default), suppress messages and diagnostic warnings
#' from \code{\link[rstan]{sampling}} during model fitting. Useful for large simulation studies.
#'
#' @return
#' A list with components:
#' \itemize{
#'   \item \code{mu}: posterior draws of \eqn{\mu},
#'   \item \code{fit}: The 'rstan' \code{stanfit} fit object (only if \code{return_fit = TRUE}).
#' }
#'
#' @examplesIf requireNamespace("rstan", quietly = TRUE)
#' \donttest{
#' data(data_example)
#' example_enroll<-data_example$example_enroll
#'
#' out <- fit_enroll(
#'   status_enroll = example_enroll$enrollstatus,
#'   t_enroll = example_enroll$interarrivaltime,
#'   seed = 1, return_fit = TRUE,
#'   quiet = FALSE
#' )
#'
#' summary(out$mu)
#' print(out$fit)
#' }
#'
#' @seealso
#' Other BayesPET model fitting: \code{\link{fit_censor}}, \code{\link{fit_event_blind}},
#' \code{\link{fit_event_unblind}},
#' \code{\link{fit_models}}, \cr \code{\link{print.BayesPET_fit}}
#'
#' @export
fit_enroll <- function(status_enroll, t_enroll, hyperparams_enroll = list(),
                       chains = 4, iter = 4000, seed = 1, refresh = 0,
                       warmup = floor(iter/2), mc.cores = 1,
                       control = list(adapt_delta = 0.95), return_fit = FALSE,
                       quiet = TRUE) {
  if (!exists("stanmodels", inherits = TRUE) || is.null(stanmodels$Enroll)) {
    stop("Internal `Stan` model `Enroll` is not available. Reinstall the package.", call. = FALSE)
  }

  # ---- input checks ----
  if (length(status_enroll) != length(t_enroll)) {
    stop("`status_enroll` and `t_enroll` must have the same length.", call. = FALSE)
  }
  if (!is.numeric(t_enroll)) stop("`t_enroll` must be numeric.", call. = FALSE)
  if (length(t_enroll) < 1L) stop("Inputs must have length >= 1.", call. = FALSE)
  if (any(!is.finite(t_enroll))) stop("`t_enroll` must be finite (no NA/NaN/Inf).", call. = FALSE)
  if (any(t_enroll < 0)) stop("`t_enroll` must be nonnegative.", call. = FALSE)
  if (any(is.na(status_enroll))) stop("`status_enroll` cannot contain NA.", call. = FALSE)
  if (!all(status_enroll %in% c(0L, 1L, 0, 1))) {
    stop("`status_enroll` must contain only 0/1 values.", call. = FALSE)
  }
  status_enroll <- as.integer(status_enroll)

  # ---- Check hyperparameters ----
  if (!is.list(hyperparams_enroll)) stop("`hyperparams_enroll` must be a list.", call. = FALSE)
  .allowed_hp <- c("alpha_mu", "beta_mu")
  .extra_hp <- setdiff(names(hyperparams_enroll), .allowed_hp)
  if (length(.extra_hp) > 0L) {
    stop(sprintf(
      "Unknown hyperparameter(s) in `hyperparams_enroll`: %s",
      paste(.extra_hp, collapse = ", ")
    ), call. = FALSE)
  }

  defaults <- list(alpha_mu = 0.1, beta_mu = 0.1)
  hp <- utils::modifyList(defaults, hyperparams_enroll)

  .check_scalar_pos <- function(x, name) {
    if (!is.numeric(x) || length(x) != 1L || !is.finite(x) || x <= 0) {
      stop(sprintf("Hyperparameter `%s` in `hyperparams_enroll` must be a finite, positive numeric scalar.", name), call. = FALSE)
    }
  }

  .check_scalar_pos(hp$alpha_mu, "alpha_mu")
  .check_scalar_pos(hp$beta_mu,  "beta_mu")

  # ---- Check stan parameters ----
  if (!is.null(seed)) {
    if (!is.numeric(seed) || length(seed) != 1L || !is.finite(seed) || seed != as.integer(seed)) {
      stop("`seed` must be NULL or a single finite integer.", call. = FALSE)
    }
    seed <- as.integer(seed)
  }
  if (!is.list(control)) {
    stop("`control` must be a named list of Stan sampler control parameters.", call. = FALSE)
  }
  if (!is.null(control) && length(control) > 0L &&
      (is.null(names(control)) || any(names(control) == ""))) {
    warning("`control` should be a *named* list (e.g., list(adapt_delta = 0.95)).",
            call. = FALSE)
  }

  .check_pos_int(chains, "chains")
  .check_pos_int(iter,   "iter")
  .check_nonneg_int(refresh, "refresh")
  .check_nonneg_int(warmup, "warmup")
  .check_pos_int(mc.cores,'mc.cores')
  if (warmup >= iter) {
    stop("`warmup` must be strictly smaller than `iter`.", call. = FALSE)
  }

  if (!is.logical(return_fit) || length(return_fit) != 1L || is.na(return_fit)) {
    stop("`return_fit` must be a single TRUE/FALSE value.", call. = FALSE)
  }
  if (!is.logical(quiet) || length(quiet) != 1L || is.na(quiet)) {
    stop("`quiet` must be a single TRUE/FALSE value.", call. = FALSE)
  }


  # ---- Stan data ----
  standata <- list(
    N = length(status_enroll),
    t = t_enroll,
    status = status_enroll,
    alpha = hp$alpha_mu,
    beta  = hp$beta_mu
  )
  wrap <- if (!isTRUE(quiet)) identity else suppressWarnings
  if(!is.null(seed)){
    fit <- wrap(rstan::sampling(
      stanmodels$Enroll,
      data = standata,
      chains = chains,
      iter = iter,
      cores = mc.cores,
      seed = seed,
      refresh = refresh,
      warmup = warmup,
      control = control
    ))
  }
  else{
    fit <- wrap(rstan::sampling(
      stanmodels$Enroll,
      data = standata,
      chains = chains,
      iter = iter,
      cores = mc.cores,
      refresh = refresh,
      warmup = warmup,
      control = control
    ))
  }

  ext <- rstan::extract(fit, pars = "mu", permuted = FALSE)

  mu_vec<-ext_to_draws(ext = ext, pars = 'mu')
  if (return_fit)
  {mu_vec$fit <- fit}
  stopifnot(is.numeric(mu_vec$mu), is.null(dim(mu_vec$mu)))
  mu_vec

}




#' Fit a Weibull model for random censoring times
#'
#' @description
#' Fits a Weibull proportional hazards model for random censoring times and returns
#' posterior draws of the Weibull shape parameter \eqn{\rho_c>0}, the baseline
#' scale parameter \eqn{\lambda_{0c}>0}, and, when covariates are included,
#' the covariate log hazard ratios \eqn{\boldsymbol{\beta}_c}.
#'
#' @details
#' Let \eqn{T_{ci}} denote the censoring time for subject \eqn{i}.
#' Let \eqn{\boldsymbol{Z}_i} denote the covariate vector for subject \eqn{i}
#' (the \eqn{i}-th row of \code{cov} when provided).
#' The parameterization has baseline density function
#' \deqn{
#'   f_{0c}(t) = \lambda_{0c}\,\rho_c\, t^{\rho_c-1}\exp(-\lambda_{0c}t^{\rho_c}), \quad t \ge 0.
#' }
#' The model assumes the survival function
#' \deqn{
#'   S_c(t \mid \boldsymbol{Z}_i) = \exp\{-\lambda_{0c} \, t^{\rho_c} \exp(\boldsymbol{Z}_i^\top \boldsymbol{\beta}_c)\}, \quad t \ge 0.
#' }
#' The corresponding hazard function is
#' \deqn{
#'   h_c(t \mid \boldsymbol{Z}_i) = \lambda_{0c}\,\rho_c\,t^{\rho_c-1}\exp(\boldsymbol{Z}_i^\top \boldsymbol{\beta}_c), \quad t \ge 0.
#' }
#' Note: In 'Stan', the Weibull distribution is parameterized by shape parameter \eqn{\rho_c} and
#' scale parameter \eqn{\sigma}, with baseline density function
#' \deqn{f_{0c}(t) = (\rho_c/\sigma)\, (t/\sigma)^{\rho_c-1}\, \exp(-(t/\sigma)^{\rho_c}), \quad t \ge 0.}
#'
#' To match the hazard-scale parameterization above, we set
#' \deqn{
#'   \sigma_i = \{\lambda_{0c}\exp(\boldsymbol{Z}_i^\top \boldsymbol{\beta}_c)\}^{-1/\rho_c}.
#' }
#' The random censoring indicator \code{status_censor} follows the convention:
#' \itemize{
#'   \item \code{0}: random censoring time observed at \code{t_obs} (density contribution),
#'   \item \code{1}: administratively right-censored at \code{t_obs} (survival contribution).
#' }
#'
#' @param t_obs
#' Numeric vector of observed times \eqn{t^*_{ci}}. Must be \code{>0}.
#'
#' @param status_censor
#' Integer vector of random censoring indicators \eqn{\delta_{ci}} with values \code{0} or \code{1}.
#' A value of \code{1} indicates the random censoring time is observed, while
#' \code{0} indicates administrative censoring or event at \code{t_obs}.
#'
#' @param cov
#' Optional matrix or data frame of covariates \eqn{\boldsymbol{Z}}.
#' Each row corresponds to a subject and each column to a covariate.
#' Covariates must be numeric and are treated as linear effects in the model.
#' Only binary (0/1) and continuous covariates are supported.
#' Multilevel or categorical variables are not expanded into dummy variables;
#' if provided as numeric (e.g., factor codes), they are treated as continuous.
#' If \code{NULL} (default), a no-covariate Weibull model is fitted.
#'
#' @param hyperparams_censor
#' A named list of prior hyperparameters for the censoring model.
#' If it is empty (\code{list()}), default values
#' are assigned internally.
#' \itemize{
#'   \item \code{alpha_c, beta_c}: Shape and rate parameters of the Gamma prior for
#'     the baseline scale parameter \eqn{\lambda_{0c}}
#'     (defaults: \eqn{0.1}, \eqn{0.1}).
#'   \item \code{alpha_rc, beta_rc}: Shape and rate parameters of the Gamma prior for
#'     the baseline shape parameter \eqn{\rho_c}
#'     (defaults: \eqn{0.1}, \eqn{0.1}).
#'   \item \code{mu_bc, sigma_bc}: Mean and standard deviation of the Normal
#'     prior for the covariate effects \eqn{\beta_{cj}}
#'     (defaults: \eqn{0}, \eqn{\sqrt{10}}).
#' }
#' No other hyperparameters are allowed.
#'
#' @param chains
#' Number of Markov chain Monte Carlo (MCMC) chains. Defaults to \code{4}.
#'
#' @param iter
#' Number of iterations per chain (including warmup). Defaults to \code{4000}.
#'
#' @param mc.cores Integer. Number of CPU cores to use when executing Markov
#'   chains in parallel via \code{\link[rstan]{sampling}}. Defaults to \code{1}.
#'   We recommend setting \code{mc.cores} to the maximum number of processors
#'   supported by the available hardware and memory, up to the number of
#'   chains.
#'
#' @param seed
#' Optional integer seed passed to \code{\link[rstan]{sampling}} for reproducibility. If \code{NULL},
#' 'Stan' generates a seed internally and results may differ across runs.
#' Defaults to \code{2}.
#'
#' @param refresh
#' Frequency of progress updates from \code{\link[rstan]{sampling}}. Set to \code{0} (default) to suppress output.
#'
#' @param warmup
#' Number of warmup (burn-in) iterations per chain. Must be strictly smaller
#' than \code{iter}. Defaults to \code{floor(iter / 2)}.
#'
#' @param control
#' A named list of sampler control parameters passed to
#' \code{\link[rstan]{sampling}}. Examples include \code{adapt_delta} and
#' \code{max_treedepth}.
#' Defaults to \code{list(adapt_delta = 0.95)}.
#'
#' @param return_fit
#' Logical; if \code{TRUE}, also return the underlying 'rstan' \code{stanfit} object.
#' Defaults to \code{FALSE}.
#'
#' @param quiet
#' Logical. If \code{TRUE} (default), suppress messages and diagnostic warnings
#' from 'Stan' during model fitting. Useful for large simulation studies.
#'
#' @return
#' A list with the following components:
#' \itemize{
#'   \item \code{rho_c}: Posterior draws of the Weibull shape parameter \eqn{\rho_c}.
#'   \item \code{lambda_0c}: Posterior draws of the baseline Weibull scale parameter \eqn{\lambda_{0c}}.
#'   \item \code{beta_c}: Posterior draws of the covariate log hazard ratios \eqn{\boldsymbol{\beta}_c},
#'    or \code{NULL} if no covariates are included.
#'   \item \code{fit}: The 'rstan' \code{stanfit} object (only if \code{return_fit = TRUE}).
#' }
#'
#' @examplesIf requireNamespace("rstan", quietly = TRUE)
#' \donttest{
#' data(data_example)
#' example_eventcensor<-data_example$example_eventcensor
#'
#' ## ---- fit censoring model ----
#' ## Reduced number of chains and iterations compared to defaults
#' ## to keep the example computationally manageable.
#' fit <- fit_censor(
#'   t_obs = example_eventcensor$time,
#'   status_censor = example_eventcensor$censorstatus,
#'   cov = example_eventcensor[,6:7],
#'   chains = 2,
#'   iter = 2000, quiet = FALSE,
#'   seed = 2, return_fit = TRUE
#' )
#'
#' summary(fit$rho_c)
#' summary(fit$lambda_0c)
#' summary(fit$beta_c)
#' print(fit$fit)
#' }
#'
#' @seealso
#' Other BayesPET model fitting: \code{\link{fit_enroll}}, \code{\link{fit_event_blind}},
#' \code{\link{fit_event_unblind}},
#' \code{\link{fit_models}}, \cr \code{\link{print.BayesPET_fit}}
#'
#' @export
fit_censor <- function(t_obs, status_censor, cov = NULL, hyperparams_censor = list(),
                       chains = 4, iter = 4000, seed = 2, refresh = 0, mc.cores = 1,
                       warmup = floor(iter/2), control = list(adapt_delta = 0.95), return_fit = FALSE,
                       quiet = TRUE) {

  if (!exists("stanmodels", inherits = TRUE)) {
    stop("Internal object `stanmodels` is not available. Reinstall the package.", call. = FALSE)
  }
  if (is.null(stanmodels$Censor) || is.null(stanmodels$Censor_nocov)) {
    stop("Internal `Stan` models `Censor`/`Censor_nocov` are not available. Reinstall the package.",
         call. = FALSE)
  }

  # ---- basic checks ----
  if (missing(t_obs) || missing(status_censor)) {
    stop("`t_obs` and `status_censor` must be provided.", call. = FALSE)
  }

  if (!is.numeric(t_obs)) stop("`t_obs` must be numeric.", call. = FALSE)
  if (length(t_obs) < 1L) stop("Inputs must have length >= 1.", call. = FALSE)
  if (any(!is.finite(t_obs))) stop("`t_obs` must be finite (no NA/NaN/Inf).", call. = FALSE)
  if (any(t_obs <= 0)) stop("`t_obs` must be > 0.", call. = FALSE)

  if (length(t_obs) != length(status_censor)) {
    stop("`t_obs` and `status_censor` must have the same length.", call. = FALSE)
  }

  if (any(is.na(status_censor))) stop("`status_censor` cannot contain NA.", call. = FALSE)
  if (!all(status_censor %in% c(0L, 1L, 0, 1))) {
    stop("`status_censor` must be coded as 0/1 (1 = censoring observed, 0 = right-censored).",
         call. = FALSE)
  }
  status_censor <- as.integer(1-status_censor) # match Stan model definition

  # ---- Hyperparameters ----
  if (!is.list(hyperparams_censor)) stop("`hyperparams_censor` must be a list.", call. = FALSE)
  .allowed_hp <- c("alpha_c","beta_c","alpha_rc","beta_rc","mu_bc","sigma_bc")
  .hp_names <- names(hyperparams_censor)
  if (length(hyperparams_censor) > 0L && is.null(.hp_names)) {
    stop("`hyperparams_censor` must be a *named* list (e.g., list(alpha_c = 0.1, beta_c = 0.1)).",
         call. = FALSE)
  }
  if (!is.null(.hp_names) && any(.hp_names == "")) {
    stop("`hyperparams_censor` must not contain empty names.", call. = FALSE)
  }
  if (is.null(.hp_names)) .hp_names <- character(0)

  .extra_hp <- setdiff(.hp_names, .allowed_hp)
  if (length(.extra_hp) > 0L) {
    stop(sprintf("Unknown hyperparameter(s): %s in `hyperparams_censor`", paste(.extra_hp, collapse = ", ")),
         call. = FALSE)
  }
  defaults <- list(
    alpha_c  = 0.1, beta_c  = 0.1,
    alpha_rc = 0.1, beta_rc = 0.1,
    mu_bc    = 0,   sigma_bc = sqrt(10)
  )
  hp <- utils::modifyList(defaults, hyperparams_censor)

  pos_names <- c("alpha_c", "beta_c", "alpha_rc", "beta_rc", "sigma_bc")
  for (nm in pos_names) {
    if (!is.numeric(hp[[nm]]) || length(hp[[nm]]) != 1 || !is.finite(hp[[nm]])) {
      stop(sprintf("Hyperparameter `%s` in `hyperparams_censor` must be a finite numeric scalar.", nm), call. = FALSE)
    }
  }
  if (hp$alpha_c <= 0 || hp$beta_c <= 0 || hp$alpha_rc <= 0 || hp$beta_rc <= 0) {
    stop("Gamma prior hyperparameters (alpha_*, beta_*) in `hyperparams_censor` must be > 0.", call. = FALSE)
  }
  if (hp$sigma_bc <= 0) stop("`sigma_bc` must be > 0.", call. = FALSE)
  if (!is.numeric(hp$mu_bc) || length(hp$mu_bc) != 1L || !is.finite(hp$mu_bc)) {
    stop("Hyperparameter `mu_bc` in `hyperparams_censor` must be a finite numeric scalar.", call. = FALSE)
  }

  # ---- covariates handling ----
  has_cov <- !is.null(cov)
  if (has_cov) {
    cov <- as.matrix(cov)
    if (nrow(cov) != length(t_obs)) {
      stop("If provided, `cov` must have nrow(cov) == length(t_obs).", call. = FALSE)
    }
    if (ncol(cov) < 1L) stop("`cov` must have at least one column.", call. = FALSE)
    if (!is.numeric(cov)) {
      stop("`cov` must be coercible to a numeric matrix (no factors/characters).", call. = FALSE)
    }
    if (any(!is.finite(cov))) stop("`cov` must be finite (no NA/NaN/Inf).", call. = FALSE)
  }


  # ---- STAN parameters check ----
  if (!is.null(seed)) {
    if (!is.numeric(seed) || length(seed) != 1L || !is.finite(seed) || seed != as.integer(seed)) {
      stop("`seed` must be NULL or a single finite integer.", call. = FALSE)
    }
    seed <- as.integer(seed)
  }
  .check_pos_int(chains, "chains")
  .check_pos_int(iter,   "iter")
  .check_nonneg_int(refresh, "refresh")
  .check_nonneg_int(warmup, "warmup")
  .check_pos_int(mc.cores,'mc.cores')
  if (warmup >= iter) {
    stop("`warmup` must be strictly smaller than `iter`.", call. = FALSE)
  }
  if (!is.logical(return_fit) || length(return_fit) != 1L || is.na(return_fit)) {
    stop("`return_fit` must be a single TRUE/FALSE value.", call. = FALSE)
  }
  if (!is.list(control)) {
    stop("`control` must be a list of `Stan` sampler control parameters.", call. = FALSE)
  }
  if (length(control) > 0L &&
      (is.null(names(control)) || any(names(control) == ""))) {
    warning("`control` should be a *named* list (e.g., list(adapt_delta = 0.95)).",
            call. = FALSE)
  }
  if (!is.logical(quiet) || length(quiet) != 1L || is.na(quiet)) {
    stop("`quiet` must be a single TRUE/FALSE value.", call. = FALSE)
  }


  # ---- build stan data & pick model ----
  if (!has_cov) {
    standata <- list(
      N = length(t_obs),
      t = t_obs,
      status = status_censor,
      a = hp$alpha_c,  b = hp$beta_c,
      c = hp$alpha_rc, d = hp$beta_rc
    )
    sm <- stanmodels$Censor_nocov
  } else {
    standata <- list(
      N = length(t_obs),
      p = ncol(cov),
      t = t_obs,
      status = status_censor,
      a = hp$alpha_c,  b = hp$beta_c,
      c = hp$alpha_rc, d = hp$beta_rc,
      m1 = hp$mu_bc, se1 = hp$sigma_bc,
      cov = cov
    )
    sm <- stanmodels$Censor
  }

  # ---- fit ----
  wrap <- if (!isTRUE(quiet)) identity else suppressWarnings
  if(is.null(seed)){
    fit <- wrap(rstan::sampling(
      object  = sm,
      data    = standata,
      chains  = chains,
      iter    = iter,
      cores   = mc.cores,
      warmup  = warmup,
      refresh = refresh,
      control = control
    ))
  }
  else{
    fit <- wrap(rstan::sampling(
      object  = sm,
      data    = standata,
      chains  = chains,
      iter    = iter,
      cores   = mc.cores,
      seed    = seed,
      warmup  = warmup,
      refresh = refresh,
      control = control
    ))
    }


  # ---- extract only what exists ----
  if (has_cov) {
    ext <- rstan::extract(fit, pars = c("rho", "lambda", "beta"), permuted = FALSE)
    ext <- ext_to_draws(ext = ext, pars = c("rho", "lambda", "beta"))
    beta_out <- ext$beta
  } else {
    ext <- rstan::extract(fit, pars = c("rho", "lambda"), permuted = FALSE)
    ext <- ext_to_draws(ext = ext, pars = c("rho", "lambda"))
    beta_out <- NULL
  }

  if(isTRUE(ncol(cov) == 1)){
    beta_out<-matrix(beta_out, ncol = 1)
  }
  if (!is.null(beta_out) && !is.null(colnames(cov))) {
    colnames(beta_out) <- colnames(cov)
  }



  out <- list(
    rho_c = ext$rho,
    lambda_0c = ext$lambda,
    beta_c = beta_out
  )
  if (return_fit) out$fit <- fit
  return(out)
}




#' Fit a Weibull event-time model with known treatment assignments
#'
#' @description
#' Fits a Weibull event time model in which treatment assignments are observed.
#' The function returns posterior draws of the Weibull shape parameter
#' \eqn{\rho_e > 0}, the baseline scale parameter \eqn{\lambda_{0e} > 0}, the
#' treatment effect coefficient (log hazard ratio) \eqn{\eta},
#' and when covariates are included, the covariate log hazard ratios
#' \eqn{\boldsymbol{\beta}_e}.
#'
#' @details
#' Let \eqn{T_{ei}} denote the event time for subject \eqn{i}. Let
#' \eqn{\boldsymbol{Z}_i} denote the corresponding covariate vector, which is the
#' \eqn{i} th row of \code{cov} when provided. Treatment assignment is represented
#' by a known indicator \eqn{x_i}, where for example \eqn{x_i = 0} denotes control
#' and \eqn{x_i = 1} denotes experimental treatment.
#' The treatment effect parameter \eqn{\eta} represents the log hazard ratio
#' comparing experimental treatment to control, and
#' \eqn{\boldsymbol{\beta}_e} denotes the vector of covariate regression
#' coefficients (log hazard ratios) in the proportional hazards model.
#'
#' The baseline event time distribution follows a Weibull model with density
#' \deqn{
#'   f_{0e}(t) = \lambda_{0e}\,\rho_e\, t^{\rho_e - 1}
#'   \exp(-\lambda_{0e} t^{\rho_e}), \quad t \ge 0.
#' }
#'
#' Conditional on treatment and covariates, the hazard function is
#' \deqn{
#'   h_e(t \mid x_i, \boldsymbol{Z}_i)
#'   = \lambda_{0e}\,\rho_e\, t^{\rho_e - 1}
#'   \exp(\eta x_i + \boldsymbol{Z}_i^\top \boldsymbol{\beta}_e), \quad t \ge 0,
#' }
#' and the corresponding survival function is
#' \deqn{
#'   S_e(t \mid x_i, \boldsymbol{Z}_i)
#'   = \exp\{-\lambda_{0e}\, t^{\rho_e}
#'     \exp(\eta x_i + \boldsymbol{Z}_i^\top \boldsymbol{\beta}_e)\}, \quad t \ge 0.
#' }
#'
#' For reference, 'Stan' uses an equivalent Weibull representation based on a shape
#' parameter \eqn{\rho_e} and a scale parameter \eqn{\sigma}, with baseline density
#' \deqn{
#'   f_{0e}(t) = (\rho_e / \sigma)\, (t / \sigma)^{\rho_e - 1}
#'   \exp\{-(t / \sigma)^{\rho_e}\}, \quad t \ge 0.
#' }
#' The two formulations are related through
#' \deqn{
#'   \sigma_i =
#'   \{\lambda_{0e}\exp(\eta x_i + \boldsymbol{Z}_i^\top \boldsymbol{\beta}_e)\}^{-1/\rho_e}, \quad t \ge 0.
#' }
#'
#' @param t_event Numeric observed time to event \eqn{t^*_{ei}}. Must be \code{>0}.
#'
#' @param status_event
#' Integer vector of event indicators \eqn{\delta_{ei}} with values \code{0} or \code{1}
#' (\code{1} = event observed, \code{0} = right-censored at \code{t_event}).
#'
#' @param treatment_ind
#' Integer vector of treatment assignments \eqn{x_i} with values \code{0} or \code{1}
#' (\code{1} = treated, \code{0} = control).
#'
#' @param cov
#' Optional matrix or data frame of covariates \eqn{\boldsymbol{Z}}.
#' Each row corresponds to a subject and each column to a covariate.
#' Covariates must be numeric and are treated as linear effects in the model.
#' Only binary (0/1) and continuous covariates are supported.
#' Multilevel or categorical variables are not expanded into dummy variables;
#' if provided as numeric (e.g., factor codes), they are treated as continuous.
#' If \code{NULL} (default), a no-covariate Weibull model is fitted.
#'
#' @param hyperparams_event
#' A named list of prior hyperparameters for the event-time model.
#' If it is empty (\code{list()}), default values
#' are assigned internally.
#' \itemize{
#'   \item \code{alpha_e}, \code{beta_e}: Shape and rate parameters of the Gamma prior for
#'     the baseline scale parameter \eqn{\lambda_{0e}}
#'     (defaults: \eqn{0.1}, \eqn{0.1}).
#'   \item \code{alpha_re}, \code{beta_re}: Shape and rate parameters of the Gamma prior for
#'     the baseline shape parameter \eqn{\rho_e}
#'     (defaults: \eqn{0.1}, \eqn{0.1}).
#'   \item \code{mu_eta}, \code{sigma_eta}: Mean and standard deviation of the
#'     Normal prior for the treatment effect (log hazard ratio) \eqn{\eta}
#'     (defaults: \eqn{0}, \eqn{\sqrt{2}}).
#'   \item \code{mu_be}, \code{sigma_be}: Mean and standard deviation of the Normal
#'     prior for the covariate effects \eqn{\beta_{ej}}
#'     (defaults: \eqn{0}, \eqn{\sqrt{10}}).
#' }
#' No other hyperparameters are allowed.
#'
#' @param chains
#' Number of Markov chain Monte Carlo (MCMC) chains. Defaults to \code{4}.
#'
#' @param iter
#' Number of iterations per chain (including warmup). Defaults to \code{4000}.
#'
#' @param mc.cores Integer. Number of CPU cores to use when executing Markov
#'   chains in parallel via \code{\link[rstan]{sampling}}. Defaults to \code{1}.
#'   We recommend setting \code{mc.cores} to the maximum number of processors
#'   supported by the available hardware and memory, up to the number of
#'   chains.
#'
#' @param seed
#' Optional integer seed passed to \code{\link[rstan]{sampling}} for reproducibility. If \code{NULL},
#' 'Stan' generates a seed internally and results may differ across runs.
#' Defaults to \code{123}.
#'
#' @param refresh
#' Frequency of progress updates from \code{\link[rstan]{sampling}}. Set to \code{0} (default) to suppress output.
#'
#' @param warmup
#' Number of warmup (burn-in) iterations per chain. Must be strictly smaller
#' than \code{iter}. Defaults to \code{floor(iter / 2)}.
#'
#' @param control
#' A named list of sampler control parameters passed to
#' \code{\link[rstan]{sampling}}. Examples include \code{adapt_delta} and
#' \code{max_treedepth}.
#' Defaults to \code{list(adapt_delta = 0.95)}.
#'
#' @param return_fit
#' Logical; if \code{TRUE}, also return the underlying 'rstan' \code{stanfit} object.
#' Defaults to \code{FALSE}.
#'
#' @param quiet
#' Logical. If \code{TRUE} (default), suppress messages and diagnostic warnings
#' from 'Stan' during model fitting. Useful for large simulation studies.
#'
#' @return
#' A list with the following components:
#' \itemize{
#'   \item \code{eta}: Posterior draws of the treatment log hazard ratio \eqn{\eta}.
#'   \item \code{rho_e}: Posterior draws of the Weibull shape parameter \eqn{\rho_e}.
#'   \item \code{lambda_0e}: Posterior draws of the baseline Weibull scale parameter
#'   \eqn{\lambda_{0e}}.
#'   \item \code{beta_e}: Posterior draws of the covariate log hazard ratios \eqn{\boldsymbol{\beta}_e},
#'          or \code{NULL} if no covariates are included.
#'   \item \code{fit}: The 'rstan' \code{stanfit}  object (only if \code{return_fit = TRUE}).
#' }
#'
#' @examplesIf requireNamespace("rstan", quietly = TRUE)
#' \donttest{
#' data(data_example)
#' example_eventcensor<-data_example$example_eventcensor
#' # Use chains = 1 here to reduce runtime for the example;
#' # use more chains in real analyses.
#' fit_e_unblind <- fit_event_unblind(
#'   t_event = example_eventcensor$time,
#'   status_event = example_eventcensor$eventstatus,
#'   treatment_ind = example_eventcensor$trt,
#'   cov = example_eventcensor[,6:7],
#'   chains = 1, iter = 2000, seed = 123,
#'   return_fit = TRUE
#' )
#'
#' summary(fit_e_unblind$eta)
#' print(fit_e_unblind$fit)
#' }
#'
#' @seealso
#' Other BayesPET model fitting: \code{\link{fit_censor}}, \code{\link{fit_enroll}},
#' \code{\link{fit_event_blind}},
#' \code{\link{fit_models}}, \cr \code{\link{print.BayesPET_fit}}
#'
#' @export
fit_event_unblind <- function(t_event, status_event, treatment_ind, cov = NULL,
                              hyperparams_event = list(), chains = 4, iter = 4000,
                              seed = 123, refresh = 0, warmup = floor(iter/2), mc.cores = 1,
                              control = list(adapt_delta = 0.95), return_fit = FALSE, quiet = TRUE) {
  if (!exists("stanmodels", inherits = TRUE)) {
    stop("Internal object `stanmodels` is not available. Reinstall the package.", call. = FALSE)
  }
  if (is.null(stanmodels$Unblind) || is.null(stanmodels$Unblind_nocov)) {
    stop("Internal `Stan` models `Unblind`/`Unblind_nocov` are not available. Reinstall the package.",
         call. = FALSE)
  }


  # ---- Parameter check ----
  if (missing(t_event) || missing(status_event) || missing(treatment_ind)) {
    stop("`t_event`, `status_event`, and `treatment_ind` must be provided.", call. = FALSE)
  }

  if (!is.numeric(t_event)) stop("`t_event` must be numeric.", call. = FALSE)
  if (length(t_event) < 1L) stop("Inputs must have length >= 1.", call. = FALSE)
  if (any(!is.finite(t_event))) stop("`t_event` must be finite (no NA/NaN/Inf).", call. = FALSE)
  if (any(t_event <= 0)) stop("`t_event` must be > 0.", call. = FALSE)

  if (length(t_event) != length(status_event)) {
    stop("`t_event` and `status_event` must have the same length.", call. = FALSE)
  }
  if (length(t_event) != length(treatment_ind)) {
    stop("`t_event` and `treatment_ind` must have the same length.", call. = FALSE)
  }

  if (any(is.na(status_event)) || !all(status_event %in% c(0L, 1L, 0, 1))) {
    stop("`status_event` must be coded as 0/1 (1 = event, 0 = right-censored).", call. = FALSE)
  }
  status_event <- as.integer(status_event)

  if (any(is.na(treatment_ind)) || !all(treatment_ind %in% c(0L, 1L, 0, 1))) {
    stop("`treatment_ind` must be coded as 0/1 (1 = treated, 0 = control).", call. = FALSE)
  }
  treatment_ind <- as.integer(treatment_ind)


  # ---- Hyperparameters ----
  if (!is.list(hyperparams_event)) stop("`hyperparams_event` must be a list.", call. = FALSE)

  .allowed_hp <- c("alpha_e","beta_e","alpha_re","beta_re","mu_eta","sigma_eta","mu_be","sigma_be")
  .hp_names <- names(hyperparams_event)

  if (length(hyperparams_event) > 0L && is.null(.hp_names)) {
    stop("`hyperparams_event` must be a *named* list (e.g., list(alpha_e = 0.1, beta_e = 0.1)).",
         call. = FALSE)
  }
  if (!is.null(.hp_names) && any(.hp_names == "")) {
    stop("`hyperparams_event` must not contain empty names.", call. = FALSE)
  }
  if (is.null(.hp_names)) .hp_names <- character(0)

  .extra_hp <- setdiff(.hp_names, .allowed_hp)
  if (length(.extra_hp) > 0L) {
    stop(sprintf("Unknown hyperparameter(s): %s in hyperparams_event", paste(.extra_hp, collapse = ", ")),
         call. = FALSE)
  }

  defaults <- list(
    alpha_e = 0.1, beta_e = 0.1,
    alpha_re = 0.1, beta_re = 0.1,
    mu_eta = 0, sigma_eta = sqrt(2),
    mu_be = 0, sigma_be = sqrt(10)
  )
  hp <- utils::modifyList(defaults, hyperparams_event)

  pos_names <- c("alpha_e","beta_e","alpha_re","beta_re","sigma_eta","sigma_be")
  for (nm in pos_names) {
    if (!is.numeric(hp[[nm]]) || length(hp[[nm]]) != 1L || !is.finite(hp[[nm]])) {
      stop(sprintf("Hyperparameter `%s` must be a finite numeric scalar.", nm), call. = FALSE)
    }
  }
  if (hp$alpha_e <= 0 || hp$beta_e <= 0 || hp$alpha_re <= 0 || hp$beta_re <= 0) {
    stop("Gamma prior hyperparameters (alpha_*, beta_*) must be > 0.", call. = FALSE)
  }
  if (hp$sigma_eta <= 0) stop("`sigma_eta` must be > 0.", call. = FALSE)
  if (hp$sigma_be <= 0) stop("`sigma_be` must be > 0.", call. = FALSE)

  for (nm in c("mu_eta","mu_be")) {
    if (!is.numeric(hp[[nm]]) || length(hp[[nm]]) != 1L || !is.finite(hp[[nm]])) {
      stop(sprintf("Hyperparameter `%s` must be a finite numeric scalar.", nm), call. = FALSE)
    }
  }

  # ---- STAN parameters check ----
  if (!is.null(seed)) {
    if (!is.numeric(seed) || length(seed) != 1L || !is.finite(seed) || seed != as.integer(seed)) {
      stop("`seed` must be NULL or a single finite integer.", call. = FALSE)
    }
    seed <- as.integer(seed)
  }
  .check_pos_int(chains, "chains")
  .check_pos_int(iter,   "iter")
  .check_nonneg_int(refresh, "refresh")
  .check_nonneg_int(warmup, "warmup")
  .check_pos_int(mc.cores, "mc.cores")
  if (warmup >= iter) {
    stop("`warmup` must be strictly smaller than `iter`.", call. = FALSE)
  }
  if (!is.logical(return_fit) || length(return_fit) != 1L || is.na(return_fit)) {
    stop("`return_fit` must be a single TRUE/FALSE value.", call. = FALSE)
  }
  if (!is.list(control)) {
    stop("`control` must be a list of `Stan` sampler control parameters.", call. = FALSE)
  }
  if (length(control) > 0L &&
      (is.null(names(control)) || any(names(control) == ""))) {
    warning("`control` should be a *named* list (e.g., list(adapt_delta = 0.95)).",
            call. = FALSE)
  }
  if (!is.logical(quiet) || length(quiet) != 1L || is.na(quiet)) {
    stop("`quiet` must be a single TRUE/FALSE value.", call. = FALSE)
  }

  # ---- Covariates ----
  has_cov <- !is.null(cov)
  if (has_cov) {
    cov <- as.matrix(cov)
    if (nrow(cov) != length(t_event)) {
      stop("If provided, `cov` must have nrow(cov) == length(t_event).", call. = FALSE)
    }
    if (ncol(cov) < 1L) stop("`cov` must have at least one column.", call. = FALSE)
    if (!is.numeric(cov)) {
      stop("`cov` must be coercible to a numeric matrix (no factors/characters).", call. = FALSE)
    }
    if (any(!is.finite(cov))) {
      stop("`cov` must be finite (no NA/NaN/Inf).", call. = FALSE)
    }
  }

  # ---- Stan models ----
  if(has_cov){
    standata <- list(
      N = length(t_event),
      p = ncol(cov),
      t = t_event,
      status = as.integer(status_event),
      treatment = as.integer(treatment_ind),
      a = hp$alpha_e, b = hp$beta_e, c = hp$alpha_re, d = hp$beta_re,
      mean_c = hp$mu_eta, se_c = hp$sigma_eta,
      mean_b = hp$mu_be, se_b = hp$sigma_be,
      cov = cov
    )} else{
      standata <- list(
        N = length(t_event),
        t = t_event,
        status = as.integer(status_event),
        treatment = as.integer(treatment_ind),
        a = hp$alpha_e, b = hp$beta_e, c = hp$alpha_re, d = hp$beta_re,
        mean_c = hp$mu_eta, se_c = hp$sigma_eta
      )
    }
  wrap <- if (!isTRUE(quiet)) identity else suppressWarnings
  if(!is.null(seed)){
    if(has_cov){fit <- wrap(rstan::sampling(stanmodels$Unblind, data = standata,
                                       chains = chains, iter = iter, seed=seed,
                                       refresh=refresh, control = control, cores = mc.cores,
                                       warmup = warmup))}
    else{fit<-wrap(rstan::sampling(stanmodels$Unblind_nocov, data = standata,
                              chains = chains, iter = iter, seed=seed, cores = mc.cores,
                              refresh=refresh, control = control, warmup = warmup))}
  }
  if(is.null(seed)){
    if(has_cov){fit<-wrap(rstan::sampling(stanmodels$Unblind, data = standata,
                                     chains = chains, iter = iter, cores = mc.cores,
                                     refresh=refresh, control = control,
                                     warmup = warmup))}
    else{fit<-wrap(rstan::sampling(stanmodels$Unblind_nocov, data = standata, cores = mc.cores,
                              chains = chains, iter = iter, refresh=refresh,
                              control = control, warmup = warmup))}
  }
  ext <- rstan::extract(fit, permuted = FALSE)
  if(has_cov){
    ext <- ext_to_draws(ext = ext, pars = c('eta','rho','lambda','beta'))
  }
  else{
    ext <- ext_to_draws(ext = ext, pars = c('eta','rho','lambda'))
  }

  out <- list(
    eta       = ext$eta,
    rho_e     = ext$rho,
    lambda_0e = ext$lambda
  )
  if(has_cov){beta_out<-ext$beta}
  else{beta_out <- NULL}

  if(!is.null(beta_out) && isTRUE(ncol(cov) == 1)){
    beta_out<-matrix(beta_out, ncol = 1)
  }
  if (!is.null(beta_out) && !is.null(colnames(cov))) {
    colnames(beta_out) <- colnames(cov)
  }
  out$beta_e<-beta_out

  if (return_fit) {out$fit <- fit}
  return(out)
}





#' Fit a Weibull event-time model with unknown treatment assignments
#'
#' @description
#' Fits a Weibull event-time model formulated as a mixture over unobserved
#' treatment assignments. The function returns posterior draws of the
#' Weibull shape parameter \eqn{\rho_e}, the baseline scale parameter
#' \eqn{\lambda_{0e}}, the treatment log hazard ratio \eqn{\eta},
#' and when covariates are included, the covariate log hazard ratios
#' \eqn{\boldsymbol{\beta}_e}.
#' Posterior draws of the latent treatment indicators \eqn{x_i} are also returned.
#'
#' @details
#' Let \eqn{T_{ei}} denote the event time for subject \eqn{i}. Let
#' \eqn{\boldsymbol{Z}_i} denote the corresponding covariate vector, which is the
#' \eqn{i} th row of \code{cov} when provided. Treatment assignment is represented
#' by a latent indicator \eqn{x_i}, where
#' \eqn{x_i = 0} denotes control and \eqn{x_i = 1} denotes experimental treatment.
#' The latent treatment indicators are jointly inferred with the model parameters
#' from the observed event time data.
#'
#' The treatment effect parameter \eqn{\eta} represents the log hazard ratio
#' comparing experimental treatment to control, and
#' \eqn{\boldsymbol{\beta}_e} denotes the vector of covariate regression
#' coefficients (log hazard ratios) in the proportional hazards model.
#'
#' The baseline event time distribution follows a Weibull model with density
#' \deqn{
#'   f_{0e}(t) = \lambda_{0e}\,\rho_e\, t^{\rho_e - 1}
#'   \exp(-\lambda_{0e} t^{\rho_e}), \quad t \ge 0.
#' }
#'
#' Conditional on treatment and covariates, the hazard function is
#' \deqn{
#'   h_e(t \mid x_i, \boldsymbol{Z}_i)
#'   = \lambda_{0e}\,\rho_e\, t^{\rho_e - 1}
#'   \exp(\eta x_i + \boldsymbol{Z}_i^\top \boldsymbol{\beta}_e), \quad t \ge 0,
#' }
#' and the corresponding survival function is
#' \deqn{
#'   S_e(t \mid x_i, \boldsymbol{Z}_i)
#'   = \exp\{-\lambda_{0e}\, t^{\rho_e}
#'     \exp(\eta x_i + \boldsymbol{Z}_i^\top \boldsymbol{\beta}_e)\}, \quad t \ge 0.
#' }
#'
#' For reference, 'Stan' uses an equivalent Weibull representation based on a shape
#' parameter \eqn{\rho_e} and a scale parameter \eqn{\sigma}, with baseline density
#' \deqn{
#'   f_{0e}(t) = (\rho_e / \sigma)\, (t / \sigma)^{\rho_e - 1}
#'   \exp\{-(t / \sigma)^{\rho_e}\}, \quad t \ge 0.
#' }
#' The two formulations are related through
#' \deqn{
#'   \sigma_i =
#'   \{\lambda_{0e}\exp(\eta x_i + \boldsymbol{Z}_i^\top \boldsymbol{\beta}_e)\}^{-1/\rho_e}.
#' }
#' Because \eqn{x_i} is not observed in blinded randomized trials, it is
#' treated as a latent variable, and the observed event-time data
#' marginally follow a mixture of two Weibull distributions corresponding
#' to the latent treatment groups.
#'
#' To avoid label switching in posterior inference, the treatment effect
#' parameter \eqn{\eta} is assigned a Normal prior truncated to
#' \eqn{[0, \infty)}, restricting \eqn{\eta} to be nonnegative.
#'
#' @param t_event Numeric observed time to event \eqn{t^*_{ei}}. Must be \code{>0}.
#'
#' @param status_event
#' Integer vector of event indicators \eqn{\delta_{ei}} with values \code{0} or \code{1}
#' (\code{1} = event observed, \code{0} = right-censored at \code{t_event}).
#'
#' @param p_trt
#' Scalar randomization probability to the experimental arm, \eqn{\gamma \in (0,1)}.
#'
#' @param cov
#' Optional matrix or data frame of covariates \eqn{\boldsymbol{Z}}.
#' Each row corresponds to a subject and each column to a covariate.
#' Covariates must be numeric and are treated as linear effects in the model.
#' Only binary (0/1) and continuous covariates are supported.
#' Multilevel or categorical variables are not expanded into dummy variables;
#' if provided as numeric (e.g., factor codes), they are treated as continuous.
#' If \code{NULL} (default), a no-covariate Weibull model is fitted.
#'
#' @param hyperparams_event
#' A named list of prior hyperparameters for the event-time model.
#' If it is empty (\code{list()}), default values
#' are assigned internally.
#' \itemize{
#'   \item \code{alpha_e}, \code{beta_e}: Shape and rate parameters of the Gamma prior for
#'     the baseline scale parameter \eqn{\lambda_{0e}}
#'     (defaults: \eqn{0.1}, \eqn{0.1}).
#'   \item \code{alpha_re}, \code{beta_re}: Shape and rate parameters of the Gamma prior for
#'     the baseline shape parameter \eqn{\rho_e}
#'     (defaults: \eqn{0.1}, \eqn{0.1}).
#'   \item \code{mu_eta}, \code{sigma_eta}: Mean and standard deviation of the Normal prior,
#'    truncated to \eqn{[0, \infty)},
#'    for the treatment effect (log hazard ratio) \eqn{\eta} (defaults: \eqn{0}, \eqn{\sqrt{2}}).

#'   \item \code{mu_be}, \code{sigma_be}: Mean and standard deviation of the Normal
#'     prior for the covariate effects \eqn{\beta_{ej}}
#'     (defaults: \eqn{0}, \eqn{\sqrt{10}}).
#' }
#' No other hyperparameters are allowed.
#'
#' @param chains
#' Number of Markov chain Monte Carlo (MCMC) chains. Defaults to \code{4}.
#'
#' @param iter
#' Number of iterations per chain (including warmup). Defaults to \code{4000}.
#'
#' @param mc.cores Integer. Number of CPU cores to use when executing Markov
#'   chains in parallel via \code{\link[rstan]{sampling}}. Defaults to \code{1}.
#'   We recommend setting \code{mc.cores} to the maximum number of processors
#'   supported by the available hardware and memory, up to the number of
#'   chains.
#'
#' @param seed
#' Optional integer seed passed to \code{\link[rstan]{sampling}} for reproducibility. If \code{NULL},
#' 'Stan' generates a seed internally and results may differ across runs.
#' Defaults to \code{123}.
#'
#' @param refresh
#' Frequency of progress updates from \code{\link[rstan]{sampling}}. Set to \code{0} (default) to suppress output.
#'
#' @param warmup
#' Number of warmup (burn-in) iterations per chain. Must be strictly smaller
#' than \code{iter}. Defaults to \code{floor(iter / 2)}.
#'
#' @param control
#' A named list of sampler control parameters passed to
#' \code{\link[rstan]{sampling}}. Examples include \code{adapt_delta} and
#' \code{max_treedepth}.
#' Defaults to \code{list(adapt_delta = 0.95)}.
#'
#' @param return_fit
#' Logical; if \code{TRUE}, also return the underlying 'rstan' \code{stanfit} object.
#' Defaults to \code{FALSE}.
#'
#' @param quiet
#' Logical. If \code{TRUE} (default), suppress messages and diagnostic warnings
#' from 'Stan' during model fitting. Useful for large simulation studies.
#'
#' @return
#' A list with the following components:
#' \itemize{
#'   \item \code{eta}: Posterior draws of the treatment log hazard ratio \eqn{\eta}.
#'   \item \code{rho_e}: Posterior draws of the Weibull shape parameter \eqn{\rho_e}.
#'   \item \code{lambda_0e}: Posterior draws of the baseline Weibull scale parameter \eqn{\lambda_{0e}}.
#'   \item \code{beta_e}: Posterior draws of the covariate log hazard ratios \eqn{\boldsymbol{\beta}_e}, or \code{NULL}
#'   if no covariates are included.
#'   \item \code{x}: Posterior draws of latent treatment indicators \eqn{x_i}.
#'   \item \code{fit}: The 'rstan' \code{stanfit} object (only if \code{return_fit = TRUE}).
#' }
#'
#' @examplesIf requireNamespace("rstan", quietly = TRUE)
#' \donttest{
#' data(data_example)
#' example_eventcensor<-data_example$example_eventcensor
#' # Use 2 chains and iter = 1000 here to reduce runtime for the example;
#' # use more chains in real analyses.
#' fit_e_blind <- fit_event_blind(
#'   t_event = example_eventcensor$time,
#'   status_event = example_eventcensor$eventstatus,
#'   cov = example_eventcensor[,6:7],
#'   p_trt = 0.5,
#'   chains = 2,
#'   iter = 1000, seed = 123,
#'   return_fit = TRUE
#' )
#'
#' summary(fit_e_blind$eta)
#' print(fit_e_blind$fit)
#' }
#'
#' @seealso
#' Other BayesPET model fitting: \code{\link{fit_censor}}, \code{\link{fit_enroll}},
#' \code{\link{fit_event_unblind}},
#' \code{\link{fit_models}}, \cr \code{\link{print.BayesPET_fit}}
#'
#' @export
fit_event_blind <- function(t_event, status_event,
                            p_trt, cov = NULL,
                            hyperparams_event = list(),
                            chains = 4, iter = 4000,
                            seed = 123, refresh = 0,
                            warmup = floor(iter/2), mc.cores = 1,
                            control = list(adapt_delta = 0.95),
                            return_fit = FALSE, quiet = TRUE) {

  if (!exists("stanmodels", inherits = TRUE)) {
    stop("Internal object `stanmodels` is not available. Reinstall the package.", call. = FALSE)
  }
  if (is.null(stanmodels$Blind) || is.null(stanmodels$Blind_nocov)) {
    stop("Internal `Stan` models `Blind`/`Blind_nocov` are not available. Reinstall the package.",
         call. = FALSE)
  }

  # ---- Parameter check ----
  if (missing(t_event) || missing(status_event) || missing(p_trt)) {
    stop("`t_event`, `status_event`, and `p_trt` must be provided.", call. = FALSE)
  }
  if (!is.numeric(t_event) || length(t_event) < 1L || any(!is.finite(t_event)) || any(t_event <= 0)) {
    stop("`t_event` must be a finite numeric vector with all values > 0.", call. = FALSE)
  }
  if (length(t_event) != length(status_event)) {
    stop("`t_event` and `status_event` must have the same length.", call. = FALSE)
  }
  if (any(is.na(status_event)) || !all(status_event %in% c(0L,1L,0,1))) {
    stop("`status_event` must be coded as 0/1 (1 = event, 0 = right-censored).", call. = FALSE)
  }
  status_event <- as.integer(status_event)

  if (!is.numeric(p_trt) || length(p_trt) != 1L || !is.finite(p_trt)) {
    stop("`p_trt` must be a single finite numeric value in [0, 1].", call. = FALSE)
  }
  if (p_trt < 0 || p_trt > 1) stop("`p_trt` must be in [0, 1].", call. = FALSE)
  if (p_trt <= 0 || p_trt >= 1) {
    warning("`p_trt` is 0 or 1; blinded mixture degenerates to a single arm.", call. = FALSE)
  }


  # ---- STAN parameters check ----
  if (!is.null(seed)) {
    if (!is.numeric(seed) || length(seed) != 1L || !is.finite(seed) || seed != as.integer(seed)) {
      stop("`seed` must be NULL or a single finite integer.", call. = FALSE)
    }
    seed <- as.integer(seed)
  }
  .check_pos_int(chains, "chains")
  .check_pos_int(iter,   "iter")
  .check_nonneg_int(refresh, "refresh")
  .check_nonneg_int(warmup, "warmup")
  .check_pos_int(mc.cores, "mc.cores")
  if (warmup >= iter) {
    stop("`warmup` must be strictly smaller than `iter`.", call. = FALSE)
  }
  if (!is.logical(return_fit) || length(return_fit) != 1L || is.na(return_fit)) {
    stop("`return_fit` must be a single TRUE/FALSE value.", call. = FALSE)
  }
  if (!is.list(control)) {
    stop("`control` must be a list of `Stan` sampler control parameters.", call. = FALSE)
  }
  if (length(control) > 0L &&
      (is.null(names(control)) || any(names(control) == ""))) {
    warning("`control` should be a *named* list (e.g., list(adapt_delta = 0.95)).",
            call. = FALSE)
  }
  if (!is.logical(quiet) || length(quiet) != 1L || is.na(quiet)) {
    stop("`quiet` must be a single TRUE/FALSE value.", call. = FALSE)
  }

  # ---- Hyperparameters ----
  if (!is.list(hyperparams_event)) stop("`hyperparams_event` must be a list.", call. = FALSE)

  .allowed_hp <- c("alpha_e","beta_e","alpha_re","beta_re","mu_eta","sigma_eta","mu_be","sigma_be")
  .hp_names <- names(hyperparams_event)

  if (length(hyperparams_event) > 0L && is.null(.hp_names)) {
    stop("`hyperparams_event` must be a *named* list (e.g., list(alpha_e = 0.1, beta_e = 0.1)).",
         call. = FALSE)
  }
  if (!is.null(.hp_names) && any(.hp_names == "")) {
    stop("`hyperparams_event` must not contain empty names.", call. = FALSE)
  }
  if (is.null(.hp_names)) .hp_names <- character(0)

  .extra_hp <- setdiff(.hp_names, .allowed_hp)
  if (length(.extra_hp) > 0L) {
    stop(sprintf("Unknown hyperparameter(s): %s in hyperparams_event", paste(.extra_hp, collapse = ", ")),
         call. = FALSE)
  }

  defaults <- list(
    alpha_e = 0.1, beta_e = 0.1,
    alpha_re = 0.1, beta_re = 0.1,
    mu_eta = 0, sigma_eta = sqrt(2),
    mu_be = 0, sigma_be = sqrt(10)
  )
  hp <- utils::modifyList(defaults, hyperparams_event)

  pos_names <- c("alpha_e","beta_e","alpha_re","beta_re","sigma_eta","sigma_be")
  for (nm in pos_names) {
    if (!is.numeric(hp[[nm]]) || length(hp[[nm]]) != 1L || !is.finite(hp[[nm]])) {
      stop(sprintf("Hyperparameter `%s` must be a finite numeric scalar.", nm), call. = FALSE)
    }
  }
  if (hp$alpha_e <= 0 || hp$beta_e <= 0 || hp$alpha_re <= 0 || hp$beta_re <= 0) {
    stop("Gamma prior hyperparameters (alpha_*, beta_*) must be > 0.", call. = FALSE)
  }
  if (hp$sigma_eta <= 0) stop("`sigma_eta` must be > 0.", call. = FALSE)
  if (hp$sigma_be <= 0) stop("`sigma_be` must be > 0.", call. = FALSE)

  for (nm in c("mu_eta","mu_be")) {
    if (!is.numeric(hp[[nm]]) || length(hp[[nm]]) != 1L || !is.finite(hp[[nm]])) {
      stop(sprintf("Hyperparameter `%s` must be a finite numeric scalar.", nm), call. = FALSE)
    }
  }
  if (!identical(hp$mu_eta, 0)) {
    warning(
      "`mu_eta` is not zero; the prior for `eta` is a Normal(mu_eta, sigma_eta^2) truncated to [0, Inf), rather than a half-normal.",
      call. = FALSE
    )

  }

  # ---- Covariates ----
  has_cov <- !is.null(cov)
  if (has_cov) {
    cov <- as.matrix(cov)
    if (nrow(cov) != length(t_event)) {
      stop("If provided, `cov` must have nrow(cov) == length(t_event).", call. = FALSE)
    }
    if (ncol(cov) < 1L) stop("`cov` must have at least one column.", call. = FALSE)
    if (!is.numeric(cov)) {
      stop("`cov` must be coercible to a numeric matrix (no factors/characters).", call. = FALSE)
    }
    if (any(!is.finite(cov))) {
      stop("`cov` must be finite (no NA/NaN/Inf).", call. = FALSE)
    }
  }

  if(has_cov){
    standata <- list(
      N = length(t_event),
      p = ncol(cov),
      t = t_event,
      status = as.integer(status_event),
      a = hp$alpha_e, b = hp$beta_e, c = hp$alpha_re, d = hp$beta_re,
      mean_c = hp$mu_eta, se_c = hp$sigma_eta,
      mean_b = hp$mu_be, se_b = hp$sigma_be,
      p_trt=p_trt, cov = cov
    )}
  else{
      standata <- list(
        N = length(t_event),
        t = t_event,
        status = as.integer(status_event),
        a = hp$alpha_e, b = hp$beta_e, c = hp$alpha_re, d = hp$beta_re,
        mean_c = hp$mu_eta, se_c = hp$sigma_eta,
        p_trt=p_trt
      )
    }

  wrap <- if (!isTRUE(quiet)) identity else suppressWarnings
  if(!is.null(seed)){
    if(has_cov){fit <- wrap(rstan::sampling(stanmodels$Blind, data = standata,
                                       chains = chains, iter = iter, seed=seed,
                                       refresh=refresh, warmup = warmup, cores = mc.cores,
                                       control=control))}
    else{fit <- wrap(rstan::sampling(stanmodels$Blind_nocov, data = standata,
                                chains = chains, iter = iter, seed=seed, cores = mc.cores,
                                refresh=refresh, warmup = warmup, control=control))}
  }
  if(is.null(seed)){
    if(has_cov){fit <- wrap(rstan::sampling(stanmodels$Blind, data = standata,
                                       chains = chains, iter = iter, warmup = warmup,
                                       cores = mc.cores, refresh=refresh,control=control))}
    else{fit <- wrap(rstan::sampling(stanmodels$Blind_nocov, data = standata,
                                chains = chains, iter = iter, refresh=refresh,
                                warmup = warmup, cores = mc.cores, control=control))}
  }
  ext <- rstan::extract(fit, permuted = FALSE)
  if(has_cov){
    ext <-ext_to_draws(ext = ext, pars = c('eta','rho','lambda','beta','x'))
  }
  else{
    ext <-ext_to_draws(ext = ext, pars = c('eta','rho','lambda','x'))
  }
  out <- list(eta=ext$eta, rho_e=ext$rho, lambda_0e=ext$lambda,
              x=ext$x)#, p_x1=ext$p_x1)

  if(has_cov){beta_out<-ext$beta}
  else{beta_out <- NULL}

  if (!is.null(beta_out) && isTRUE(ncol(cov) == 1)) {
    beta_out <- matrix(beta_out, ncol = 1)
  }
  if (!is.null(colnames(cov)) && !is.null(beta_out)) {
    colnames(beta_out) <- colnames(cov)
  }
  out$beta_e<-beta_out

  if(return_fit){out$fit <- fit}
  return(out)
}




#' Fit enrollment, event-time, and censoring models to clinical
#' trial data and return posterior draws model parameters
#'
#' @description
#' Fits the enrollment, event-time, and censoring models to trial data
#' and returns posterior draws of model parameters.
#'
#' @details
#' This function fits three submodels: an enrollment model,
#' a censoring model, and an event-time model conditional on the given trial data.
#' If treatment assignments are known, the event-time model is fit using
#' \code{\link{fit_event_unblind}}; otherwise, a blinded event-time model
#' is fit using \cr \code{\link{fit_event_blind}}.
#' Technical details of the likelihoods, priors and parameterizations are documented in
#' \code{\link{fit_enroll}}, \code{\link{fit_censor}},
#' \code{\link{fit_event_unblind}}, and \code{\link{fit_event_blind}}.
#'
#' @param data.enroll
#' A data frame of enrollment information up to the analysis time.
#' Must contain the columns:
#' \itemize{
#'   \item \code{interarrivaltime}: Numeric vector of interarrival times (> 0).
#'   \item \code{enrollstatus}: Integer vector coded \code{1} = enrolled,
#'     \code{0} = administratively censored.
#' }
#' Any additional columns are ignored.
#'
#' @param data.eventcensor
#' A data frame of observed event/censoring outcomes at the analysis time.
#' Must contain (at minimum) the following columns:
#' \itemize{
#'   \item \code{time}: observed follow-up time (event or censoring time);
#'      must be numeric and \eqn{> 0}.
#'   \item \code{eventstatus}: event indicators (\code{1} = event, \code{0} = right-censored).
#'   \item \code{censorstatus}: random censoring indicators \eqn{\delta_{ci}}
#'   (\code{1} = random censoring observed,
#'   \code{0} = no random censoring, including administrative censoring
#'   or event observed).
#' }
#'
#' If \code{blinded = FALSE}, \code{data.eventcensor} must also contain:
#' \itemize{
#'   \item \code{trt}: observed treatment assignment indicators coded \code{0/1}.
#' }
#'
#' The column \code{No} (representing a subject index) may be included but is not required.
#' Any additional columns (other than \code{No}, \code{trt}, \code{time}, \code{eventstatus},
#' and \code{censorstatus}) are treated as numeric baseline covariates and will be used if present.
#'
#' @param blinded Logical. If \code{TRUE} (default), the interim analysis is blinded and treatment assignments
#'   for current subjects are not observed in the data. If \code{FALSE}, the analysis
#'   is unblinded and observed treatment assignments are used.
#'
#' @param p_trt
#' Numeric scalar in \eqn{[0,1]} giving the prespecified randomization probability
#' of assignment to the experimental treatment arm. Required only if \code{blinded = TRUE};
#' ignored otherwise. Defaults to \code{NULL}.
#'
#' @param hyperparams_enroll
#' A named list of prior hyperparameters for the enrollment model.
#' If it is empty (\code{list()}), default values
#' are assigned internally.
#' \itemize{
#'   \item \code{alpha_mu}: Shape parameter of the Gamma prior for the
#'     enrollment rate \eqn{\mu} (default: \eqn{0.1}).
#'   \item \code{beta_mu}: Rate parameter of the Gamma prior for the
#'     enrollment rate \eqn{\mu} (default: \eqn{0.1}).
#' }
#' No other hyperparameters are allowed.
#'
#' @param hyperparams_censor
#' A named list of prior hyperparameters for the censoring model.
#' If it is empty (\code{list()}), default values
#' are assigned internally.
#' \itemize{
#'   \item \code{alpha_c, beta_c}: Shape and rate parameters of the Gamma prior for
#'     the baseline scale parameter \eqn{\lambda_{0c}}
#'     (defaults: \eqn{0.1}, \eqn{0.1}).
#'   \item \code{alpha_rc, beta_rc}: Shape and rate parameters of the Gamma prior for
#'     the baseline shape parameter \eqn{\rho_c}
#'     (defaults: \eqn{0.1}, \eqn{0.1}).
#'   \item \code{mu_bc, sigma_bc}: Mean and standard deviation of the Normal
#'     prior for the covariate effects \eqn{\beta_{cj}}
#'     (defaults: \eqn{0}, \eqn{\sqrt{10}}).
#' }
#' No other hyperparameters are allowed.
#'
#' @param hyperparams_event
#' A named list of prior hyperparameters for the event-time model.
#' If it is empty (\code{list()}), default values
#' are assigned internally.
#' \itemize{
#'   \item \code{alpha_e}, \code{beta_e}: Shape and rate parameters of the Gamma prior for
#'     the baseline scale parameter \eqn{\lambda_{0e}}
#'     (defaults: \eqn{0.1}, \eqn{0.1}).
#'   \item \code{alpha_re}, \code{beta_re}: Shape and rate parameters of the Gamma prior for
#'     the baseline shape parameter \eqn{\rho_e}
#'     (defaults: \eqn{0.1}, \eqn{0.1}).
#'   \item \code{mu_eta}, \code{sigma_eta}: Mean and standard deviation of the
#'     Normal prior for the treatment effect (log hazard ratio) \eqn{\eta}
#'     (defaults: \eqn{0}, \eqn{\sqrt{2}}).
#'   \item \code{mu_be}, \code{sigma_be}: Mean and standard deviation of the Normal
#'     prior for the covariate effects \eqn{\beta_{ej}}
#'     (defaults: \eqn{0}, \eqn{\sqrt{10}}).
#' }
#' When \code{blinded = TRUE}, the prior for the treatment effect \eqn{\eta}
#' is a truncated Normal distribution on \eqn{[0, \infty)} with parameters
#' \code{mu_eta} and \code{sigma_eta}. When \code{blinded = FALSE}, the prior
#' for \eqn{\eta} is an untruncated Normal distribution.
#' No other hyperparameters are allowed.
#'
#' @param chains
#' Number of Markov chain Monte Carlo (MCMC) chains. Defaults to \code{4}.
#'
#' @param iter
#' Number of iterations per chain (including warmup). Defaults to \code{4000}.
#'
#' @param mc.cores Integer. Number of CPU cores to use when executing Markov
#'   chains in parallel via \code{\link[rstan]{sampling}}. Defaults to \code{1}.
#'   We recommend setting \code{mc.cores} to the maximum number of processors
#'   supported by the available hardware and memory, up to the number of
#'   chains.
#'
#' @param seed
#' Optional random seed(s) passed to \code{\link[rstan]{sampling}} for reproducibility.
#' Can be specified as:
#' \itemize{
#'   \item a single integer or \code{NULL}, in which case the same seed
#'     is used for all three submodels, or
#'   \item a list of up to three integers or \code{NULL}s, recycled to length 3,
#'     corresponding to the enrollment, censoring, and event-time models,
#'     respectively.
#' }
#' Use \code{NULL} to allow 'Stan' to select a seed internally. Defaults to \code{list(123)}.
#'
#' @param refresh
#' Frequency of progress updates from \code{\link[rstan]{sampling}}. Set to \code{0} (default) to suppress output.
#'
#' @param warmup
#' Number of warmup (burn-in) iterations per chain. Must be strictly smaller
#' than \code{iter}. Defaults to \code{floor(iter / 2)}.
#'
#' @param control
#' Sampler control settings passed to \code{\link[rstan]{sampling}} for the three
#' submodels (enrollment, censoring, and event-time).
#' Can be specified as:
#' \itemize{
#'   \item a single named list of control parameters (shared across all
#'     three submodels), or
#'   \item a list of up to three named lists, recycled to length 3, giving
#'     separate control settings for the enrollment, censoring, and
#'     event-time models, respectively.
#' }
#' Typical entries include \code{adapt_delta} and \code{max_treedepth}.
#'
#' Defaults to \code{list(list(adapt_delta = 0.95))}.
#'
#' @param return_fit
#' Logical; if \code{TRUE}, also return the underlying 'rstan' \code{stanfit} objects for the
#' enrollment, censoring, and event models. Defaults to \code{FALSE}.
#'
#' @param quiet
#' Logical. If \code{TRUE} (default), suppress messages and diagnostic warnings
#' from 'Stan' during model fitting. Useful for large simulation studies.
#'
#' @return
#' An object of class \code{"BayesPET_fit"}, a named list containing
#' posterior draws and related information from the fitted models, with elements:
#' \itemize{
#'   \item \code{blinded}: Logical; indicates whether the analysis is blinded,
#'
#'   \item \code{mu}:
#'     Posterior draws of the enrollment rate \eqn{\mu}.
#'
#'   \item \code{rho_c}:
#'     Posterior draws of the censoring-model Weibull shape parameter \eqn{\rho_c}.
#'
#'   \item \code{lambda_0c}:
#'     Posterior draws of the censoring-model baseline Weibull scale parameter \eqn{\lambda_{0c}}.
#'
#'   \item \code{beta_c}:
#'     Posterior draws of the censoring-model covariate log hazard ratios
#'     \eqn{\boldsymbol{\beta}_c}, or \code{NULL} if no covariates are included.
#'
#'   \item \code{eta}:
#'     Posterior draws of the treatment log hazard ratio \eqn{\eta}.
#'
#'   \item \code{rho_e}:
#'     Posterior draws of the event-model Weibull shape parameter \eqn{\rho_e}.
#'
#'   \item \code{lambda_0e}:
#'     Posterior draws of the event-model baseline Weibull scale parameter \eqn{\lambda_{0e}}.
#'
#'   \item \code{beta_e}:
#'     Posterior draws of the event-model covariate log hazard ratios
#'     \eqn{\boldsymbol{\beta}_e}, or \code{NULL} if no covariates are included.
#'
#'   \item \code{treatment_ind}:
#'     Observed treatment assignments \eqn{x_i \in \{0,1\}} (only returned if
#'     \code{blinded = FALSE}).
#'
#'   \item \code{x}:
#'     Posterior draws of latent treatment assignments \eqn{x_i \in \{0,1\}}
#'     (only returned if \code{blinded = TRUE}).
#'
#'   \item \code{fit}:
#'     A list with components \code{enroll}, \code{censor}, and \code{event}
#'     containing the underlying 'rstan' \code{stanfit} objects (present only if \code{return_fit = TRUE}).
#' }
#' The default \code{\link{print}} method displays a concise overview.
#'
#' @seealso
#' Other BayesPET model fitting: \code{\link{fit_censor}}, \code{\link{fit_enroll}},
#' \code{\link{fit_event_blind}}, \code{\link{fit_event_unblind}}, \cr \code{\link{print.BayesPET_fit}}
#'
#' @examplesIf requireNamespace("rstan", quietly = TRUE)
#' \donttest{
#' data(data_example)
#' example_enroll <- data_example$example_enroll
#' example_eventcensor <- data_example$example_eventcensor
#'
#' # Unblinded analysis
#' ## Use 2 chains and iter = 2000 here to reduce runtime for the example;
#' ## use more chains in real analyses.
#' fit.unblind <- fit_models(
#'   data.enroll = example_enroll,
#'   data.eventcensor = example_eventcensor,
#'   blinded = FALSE,
#'   chains = 2, iter = 2000, seed = list(123),
#'   return_fit = TRUE, mc.cores = 1, quiet = FALSE
#' )
#'
#' # Blinded analysis
#' example_eventcensor.blind <- example_eventcensor
#' example_eventcensor.blind$trt <- NA
#' ## Use 2 chains and iter = 2000 here to reduce runtime for the example;
#' ## use more chains in real analyses.
#' fit.blind <- fit_models(
#'   data.enroll = example_enroll,
#'   data.eventcensor = example_eventcensor.blind,
#'   blinded = TRUE, p_trt = 0.5,
#'   chains = 2, iter = 2000, seed = list(123),
#'   return_fit = TRUE, mc.cores = 1, quiet = FALSE
#' )
#'
#' print(fit.unblind)
#' summary(fit.unblind$eta)
#'
#' print(fit.blind)
#' summary(fit.blind$eta)
#' }
#'
#' @export
#'
fit_models<-function(data.enroll, data.eventcensor, blinded = TRUE, p_trt = NULL,
                   hyperparams_enroll = list(), hyperparams_event = list(),
                   hyperparams_censor = list(),
                   chains = 4, iter = 4000, mc.cores = 1,
                   warmup = floor(iter/2), seed = list(123),
                   refresh = 0, control = list(list(adapt_delta = 0.95)),
                   return_fit = FALSE, quiet = TRUE){

  # ---- blinded / p_trt logic ----
  # normalize blinded to logical
  if (length(blinded) != 1L || is.na(blinded)) {
    stop("`blinded` must be a single TRUE/FALSE (or 1/0) value.", call. = FALSE)
  }

  blinded <- as.logical(blinded)

  if (is.na(blinded)) {
    stop("`blinded` must be TRUE/FALSE or 1/0.", call. = FALSE)
  }


  if (blinded) {
    if (missing(p_trt) || is.null(p_trt)) {
      stop("`p_trt` must not be NULL when `blinded = TRUE`.", call. = FALSE)
    }
    if (!is.numeric(p_trt) || length(p_trt) != 1L || !is.finite(p_trt)) {
      stop("`p_trt` must be a single finite numeric value in [0, 1].", call. = FALSE)
    }
    if (p_trt < 0 || p_trt > 1) {
      stop("`p_trt` must be in [0, 1].", call. = FALSE)
    }
    if (p_trt <= 0 || p_trt >= 1) {
      warning("`p_trt` is 0 or 1; blinded mixture degenerates to a single arm.", call. = FALSE)
    }
  } else {
    # unblinded: ignore p_trt if user supplied it
    if (!missing(p_trt) && !is.null(p_trt)) {
      warning("`p_trt` is ignored when `blinded = FALSE`.", call. = FALSE)
    }
  }

  # ---- data.enroll and data.eventcensor checks and cov ----
  .check_data.enroll(data.enroll)
  cov <- .check_data.eventcensor(data.eventcensor,blinded=blinded)
  n_enroll <- nrow(data.enroll)
  n_event  <- nrow(data.eventcensor)


  # ---- Hyperparameters are checked within each fit function ----

  # ---- Check STAN parameters (control and seed) ----
  seed    <- .check_seed_list3(seed)
  control <- .check_control_list3(control)

  # ---- Fit Models ----
  if(!isTRUE(quiet)){print('Fitting enrollment model...')}
  enrollment <- fit_enroll(status_enroll = as.integer(data.enroll$enrollstatus),
                         t_enroll = data.enroll$interarrivaltime,
                         hyperparams_enroll = hyperparams_enroll,
                         chains = chains, iter = iter, warmup = warmup,
                         seed = seed[[1]], refresh = refresh, mc.cores = mc.cores,
                         control = control[[1]], return_fit = return_fit)
  if(!isTRUE(quiet)){print('Done.')}
  if(!isTRUE(quiet)){print('Fitting censoring model...')}
  censor <- fit_censor(t_obs = data.eventcensor$time,
                     status_censor = data.eventcensor$censorstatus,
                     cov = cov, hyperparams_censor = hyperparams_censor, mc.cores = mc.cores,
                     chains = chains, iter = iter, warmup = warmup, seed = seed[[2]],
                     refresh = refresh, control = control [[2]], return_fit = return_fit)
  if(!isTRUE(quiet)){print('Done.')}
  if(!isTRUE(quiet)){print('Fitting event model...')}
  if(!blinded){
    event<-fit_event_unblind(t_event = data.eventcensor$time,
                             status_event = data.eventcensor$eventstatus,
                             treatment_ind = data.eventcensor$trt, mc.cores = mc.cores,
                             cov = cov, hyperparams_event = hyperparams_event,
                             chains = chains, iter = iter, warmup = warmup,
                             seed=seed[[3]], refresh=refresh, control = control[[3]],
                             return_fit= return_fit)
  } else{
    event<-fit_event_blind(t_event = data.eventcensor$time,
                           status_event = data.eventcensor$eventstatus,
                           cov = cov, p_trt = p_trt, hyperparams_event = hyperparams_event,
                           chains = chains, iter = iter, warmup = warmup, mc.cores = mc.cores,
                           seed=seed[[3]], refresh=refresh, control = control[[3]],
                           return_fit = return_fit)
  }
  if(!isTRUE(quiet)){print('Done.')}
  result<-list(
    blinded = blinded,
    mu = enrollment$mu,
    rho_c = censor$rho_c, lambda_0c = censor$lambda_0c, beta_c= censor$beta_c,
    eta = event$eta, rho_e = event$rho_e, lambda_0e = event$lambda_0e, beta_e = event$beta_e)
  if(blinded){
    result$x <- event$x
  } else{
    result$treatment_ind = data.eventcensor$trt
  }
  if (isTRUE(return_fit)) {
    result$fit <- list(
      enroll = enrollment$fit,
      censor = censor$fit,
      event  = event$fit
    )
  }
  result$call <- match.call()
  class(result) <- c("BayesPET_fit", "list")

  return(result)
}







#' Solve baseline survival parameters by matching the marginal median survival time
#'
#' @description
#' In practice, it is sometimes more convenient to work with a marginal
#' (population-level) median survival time. This function solves for exactly
#' one unknown baseline quantity - \code{shape}, \code{scale}, or \code{median} -
#' in a proportional hazards (PH) survival model with either a Weibull or
#' log-logistic baseline distribution, given the other two.
#'
#' When covariates are provided, \code{median} is interpreted as the marginal
#' (population-level) median survival time defined implicitly by
#' \deqn{\mathbb{E}_{\boldsymbol{Z}}\{S(\mathrm{median}\mid \boldsymbol{Z})\} = 0.5}
#' and the unknown quantity is obtained by Monte Carlo integration over the
#' covariate distribution.
#'
#' @details
#' The proportional hazards (PH) model assumes that, conditional on a covariate
#' vector \eqn{\boldsymbol{Z}}, the hazard function satisfies
#' \deqn{h(t \mid \boldsymbol{Z}) = h_0(t)\exp(\boldsymbol{Z}^\top \boldsymbol{\beta})}
#' for \eqn{t \ge 0}, where \eqn{h_0(t)} is the baseline hazard and
#' \eqn{\boldsymbol{\beta}} is a vector of regression coefficients.
#'
#' The baseline model is specified by \code{distribution} through the baseline
#' survival function \eqn{S_0(t)}:
#'
#' \itemize{
#'   \item \code{"Weibull"}: baseline survival is
#'     \deqn{S_0(t) = \exp\{-\lambda_0 t^{\rho}\}}
#'     for \eqn{t \ge 0}, where \eqn{\rho > 0} is the \code{shape} parameter and
#'     \eqn{\lambda_0 > 0} is the \code{scale} parameter. An equivalent
#'     representation used by several standard implementations is
#'     \deqn{S_0(t) = \exp\{-(t/\sigma_0)^{\rho}\}}
#'     for \eqn{t \ge 0}, where \eqn{\sigma_0 > 0} is a reparameterization satisfying
#'     \eqn{\lambda_0 = \sigma_0^{-\rho}}. This latter form is used by
#'     \code{\link[stats]{dweibull}},
#'     \code{\link[eha]{phreg}},
#'     \code{\link[flexsurv]{flexsurvreg}}, and \code{\link[rstan:stan_model]{rstan}}.
#'
#'   \item \code{"Loglogistic"}: baseline survival is
#'     \deqn{S_0(t) = \{1 + (t/b)^{a}\}^{-1}}
#'     for \eqn{t \ge 0}, where \eqn{a>0} is the \code{shape} parameter
#'      and \eqn{b>0} is the \code{scale} parameter.
#' }
#' Under the PH assumption, the conditional survival satisfies
#' \deqn{S(t \mid \boldsymbol{Z}) = S_0(t)^{\exp(\boldsymbol{Z}^\top \boldsymbol{\beta})}.}
#'
#' When no covariates are supplied (\code{cov_type = NULL}), the median survival time
#' corresponds to the baseline median and closed-form
#' solutions are available for some distributions.
#'
#' When covariates are supplied, the median survival time \eqn{m} is interpreted as
#' the marginal (population) median defined implicitly by
#' \deqn{\mathbb{E}_{\boldsymbol{Z}}\{S(m \mid \boldsymbol{Z})\} = 0.5,}
#' where the expectation is taken with respect to the covariate distribution implied
#' by \code{cov_type} and \code{cov_dist}. This expectation is approximated using
#' Monte Carlo simulation with \code{S} draws, and the unknown parameter is obtained
#' by numerical root finding via \code{\link[stats]{uniroot}}.
#'
#' @param distribution Character string specifying the baseline survival distribution.
#'   Must be either \code{"Weibull"} or \code{"Loglogistic"} (log-logistic).
#'   Defaults to \code{"Weibull"}.
#'
#' @param shape Positive numeric scalar. Baseline shape parameter.
#'   For \code{distribution} = \cr \code{"Weibull"}, this corresponds to the Weibull shape
#'   \eqn{\rho}. For \code{distribution} = \code{"Loglogistic"}, this corresponds to the
#'   log-logistic shape \eqn{a}. Set to \code{NULL} to solve for the shape.
#'
#' @param scale Positive numeric scalar. Baseline scale parameter.
#'   For \code{"Weibull"}, this is the Weibull scale parameter \eqn{\lambda_0}.
#'   For \code{"Loglogistic"}, this is the log-logistic scale \eqn{b}.
#'   Set to \code{NULL} to solve for the scale.
#'
#' @param median Positive numeric scalar. Median survival time.
#'   If covariates are provided, this is the *marginal* (population) median
#'   \eqn{m} defined by
#'   \deqn{\mathbb{E}_{\boldsymbol{Z}}\{S(m \mid \boldsymbol{Z})\} = 0.5,}
#'   where \eqn{\boldsymbol{Z}} denotes the covariate vector and the expectation is taken
#'   with respect to the covariate distribution implied by
#'   \code{cov_type} and \code{cov_dist}. Set to \code{NULL} to solve for \code{median}.
#'
#' @param cov_type Character vector specifying the distribution for each component
#'   of the covariate vector \eqn{\boldsymbol{Z} = (Z_1,\dots,Z_p)} used in the
#'   proportional hazards model.
#'   Each element must be \code{"binary"} or \code{"continuous"}.
#'   If \code{NULL} (default), no covariates are used.
#'
#' @param cov_dist Numeric vector of the same length as \code{cov_type}, giving
#'   parameters for the covariate-generating distribution of each \eqn{Z_j}:
#'   \itemize{
#'     \item \code{"binary"}: \eqn{Z_j \sim \mathrm{Bernoulli}(p_j)} with
#'       \eqn{p_j = cov\_dist[j]}.
#'     \item \code{"continuous"}: \eqn{Z_j \sim N(0, \sigma_j^2)} with
#'       \eqn{\sigma_j = cov\_dist[j]}.
#'   }
#'   If \code{NULL} (default), no covariates are used.
#'
#' @param beta Numeric vector of regression coefficients \eqn{\boldsymbol{\beta}} in the
#'   proportional hazards linear predictor \eqn{\boldsymbol{Z}^\top \boldsymbol{\beta}}.
#'   Must have the same length as \code{cov_type}.
#'   If \code{NULL} (default), no covariates are used.
#'
#' @param S Integer. Monte Carlo sample size used to approximate the marginal survival
#'   when covariates are provided. Defaults to \code{20000}.
#'
#' @param seed.solveparam Integer random seed for covariate simulation;
#' if \code{NULL}, the RNG state is not reset. Defaults to \code{123}.
#'
#' @param interval Numeric vector of length 2 giving the lower and upper
#'   bounds passed to \code{\link{uniroot}} for the root-finding procedure
#'   used to solve for the unknown parameter (shape, scale, or median).
#'   The interval must bracket the true solution, i.e., the function values
#'   at the two endpoints must have opposite signs.
#'   Defaults to \code{c(1e-15, 100)} when \code{scale = NULL}
#'   and \code{c(1e-6, 20000)} otherwise.
#'
#' @param tol Numeric scalar. Defaults to \code{.Machine$double.eps^0.25}.
#'  Convergence tolerance passed to \code{\link[stats]{uniroot}}.
#'
#' @param maxiter Integer giving the maximum number of iterations allowed for
#'   \code{\link[stats]{uniroot}}. An error is raised if the algorithm fails
#'   to converge within this limit. Defaults to \code{1000}.
#'
#' @return
#' A numeric scalar giving the solved parameter:
#' \itemize{
#'   \item the \code{shape} parameter if \code{shape = NULL};
#'   \item the baseline \code{scale} parameter if \code{scale = NULL};
#'   \item the \code{median} survival time if \code{median = NULL}.
#' }
#'
#' @seealso
#' \code{\link[stats]{dweibull}}
#' \code{\link[flexsurv]{rllogis}}
#'
#' @examples
#' # Weibull: convert a desired median to the corresponding scale parameter
#' ## No covariates
#' convert_median(
#'   distribution = "Weibull",
#'   shape = 3,
#'   median = 5,
#'   scale = NULL,
#'   seed = 123
#' )
#' ## With covariates
#' convert_median(
#'   distribution = "Weibull",
#'   shape = 3,
#'   median = 5,
#'   scale = NULL,
#'   cov_type = c("binary", "continuous"),
#'   cov_dist = c(0.5, 1),
#'   beta = c(1, 0.5),
#'   seed = 123
#' )
#'
#' # Log-logistic: convert median to scale when covariates enter the model
#' convert_median(
#'   distribution = "Loglogistic",
#'   shape = 1.5,
#'   median = 3,
#'   scale = NULL,
#'   cov_type = c("binary", "continuous"),
#'   cov_dist = c(0.5, 1),
#'   beta = c(1, 0.5),
#'   seed = 123
#' )
#'
#' @export
convert_median<-function(distribution = 'Weibull', shape, median,
                     scale, cov_type = NULL, cov_dist = NULL,
                     beta = NULL, S = 20000, seed.solveparam = 123,
                     interval = if (is.null(scale)) c(1e-15, 100) else c(1e-6, 20000),
                     tol = .Machine$double.eps^0.25, maxiter = 1000){
  if(!distribution%in%c('Weibull', 'Loglogistic')){
    stop("Distribution must be either 'Weibull' or 'Loglogistic'.", call. = FALSE)
    }
  if(distribution=='Weibull'){
    param<-(solveparam_weibull(shape=shape,median.event=median,scale=scale,
                              scaletype='hazard',cov_type=cov_type,cov_dist=cov_dist,
                              beta=beta,S=S,interval = interval, tol = tol,
                              maxiter = maxiter, seed = seed.solveparam))
  }
  if(distribution=='Loglogistic'){
    param<-(solveparam_log(shape=shape,median.event=median,scale=scale,
                          cov_type=cov_type, cov_dist=cov_dist, beta=beta,
                          S=S,seed=seed.solveparam,interval = interval,
                          tol = tol, maxiter = maxiter))
  }
  unknown.param<-ifelse(is.null(shape),'shape',ifelse(is.null(median),'median','scale'))
  if (!is.finite(param) || param <= 0) {
    stop(
      paste0(
        "Solved ", unknown.param, " is non-positive or not finite (", param,
        "). This parameter must be > 0 and finite. ",
        "Please check whether the supplied inputs are correct."
      ),
      call. = FALSE
    )
  }
  return(param)

}



utils::globalVariables(c(
  "No", "trt", "time", "status", "dropout",
  "eventtime", "eventstatus", "censorstatus"
))



#' Generate two-arm trial data with enrollment, event, and censoring processes,
#' and return data formatted for event-time prediction.
#'
#' @description
#' Simulates data from a two-arm clinical trial with a time-to-event endpoint.
#' The data generating process incorporates staggered enrollment, event times,
#' and random censoring, with event and censoring distributions specified as
#' Weibull or log-logistic. Treatment and covariate effects are incorporated
#' through a proportional hazards structure.
#'
#' @details
#' Subjects are randomized independently to the experimental arm with
#' probability \code{p_trt}. Baseline
#' covariates are generated independently based on \code{cov_type} and
#' \code{cov_dist}. Binary covariates follow a Bernoulli distribution, while
#' continuous covariates follow a normal distribution with mean zero and
#' standard deviation determined by \code{cov_dist}.
#'
#' Interarrival times between successive enrollments are drawn from an exponential
#' distribution with rate \code{enroll_rate} with model details documented in
#' \code{\link{fit_enroll}}. Calendar enrollment times are obtained by cumulative
#' summation of these interarrival times.
#'
#' Event times and random censoring times are generated from Weibull or
#' log-logistic baseline distributions, as specified by \code{dist.event}
#' and \code{dist.censor}. For the Weibull model, the baseline survival function
#' is parameterized as \deqn{S_0(t) = \exp\{-\lambda_0 t^{\rho}\}, \quad t \ge 0,} where
#' \eqn{\rho > 0} is the shape and \eqn{\lambda_0 > 0} is the baseline hazard
#' scale. For the log-logistic model, the baseline survival function is
#' \deqn{S_0(t) = \{1 + (t / b)^a\}^{-1}, \quad t \ge 0,} where \eqn{a > 0} and \eqn{b > 0}
#' denote the shape and scale parameters, respectively.
#' Parameter calibration via marginal median survival can be performed using
#' \code{\link{convert_median}} prior to simulation.
#'
#' Covariate effects are incorporated through a
#' proportional hazards structure for both the event and censoring
#' processes. When \code{logHR.trt} is provided, the treatment effect is
#' modeled through a proportional hazards formulation. When
#' \code{logHR.trt} is \code{NULL}, the two treatment arms are allowed to
#' differ through separate baseline parameters and covariate effects.
#' The random censoring mechanism does not depend on treatment assignment.
#'
#' @param N Integer. Total planned sample size (maximum number of subjects
#'   that can be enrolled in the trial).
#' @param E_target Integer. Target number of events for the final analysis.
#' @param E_cutoff Integer. Target number of events for the interim analysis.
#' @param p_trt Scalar randomization probability to the experimental arm, \eqn{\gamma \in (0,1)}.
#'
#' @param cov_type Character vector specifying the distribution for each component
#'   of the covariate vector \eqn{\boldsymbol{Z} = (Z_1,\dots,Z_p)} used in the proportional hazards model.
#'   Each element must be \code{"binary"} or \code{"continuous"}.
#'   If \code{NULL}, no covariates are used.
#'
#' @param cov_dist Numeric vector of the same length as \code{cov_type}, giving
#'   parameters for the covariate-generating distribution of each \eqn{Z_j}:
#'   \itemize{
#'     \item \code{"binary"}: \eqn{Z_j \sim \mathrm{Bernoulli}(p_j)} with
#'       \eqn{p_j = cov\_dist[j]}.
#'     \item \code{"continuous"}: \eqn{Z_j \sim N(0, \sigma_j^2)} with
#'       \eqn{\sigma_j = cov\_dist[j]}.
#'   }
#'
#' @param logHR.trt Numeric scalar giving the log hazard ratio for the experimental
#'   versus control arm in the event-time model. When \code{NULL} (default), the two treatment
#'   arms are generated from separate proportional hazards models with
#'   arm-specific baseline parameters and covariate effects.
#'
#' @param enroll_rate Positive numeric scalar specifying the enrollment rate.
#'
#' @param dist.event Character. Baseline distribution for event times:
#'   \code{"Weibull"} or \code{"Loglogistic"}. This distribution family is the same
#'   for both arms.
#' @param dist.censor Character. Baseline distribution for random censoring times:
#'   \code{"Weibull"} or \code{"Loglogistic"}.
#'
#' @param event.shape Numeric scalar > 0. Control-arm event baseline shape parameter.
#' @param event.scale Numeric scalar > 0. Control-arm event baseline scale parameter.
#' @param beta.event Numeric vector. Regression coefficients for baseline
#'   covariates in the event-time proportional hazards model; must have the
#'   same length and ordering as \code{cov_type}.
#'
#' @param censor.shape Numeric scalar > 0. Random censoring baseline shape parameter.
#' @param censor.scale Numeric scalar > 0. Random censoring baseline scale parameter.
#' @param beta.censor Numeric vector. Regression coefficients for baseline
#'   covariates in the random censoring-time proportional hazards model; must
#'   have the same length and ordering as \code{cov_type}.
#'
#' @param event.shape_trt Numeric scalar > 0. Experimental-arm event baseline shape parameter
#'   (used when \code{logHR.trt = NULL}).
#' @param event.scale_trt Numeric scalar > 0. Experimental-arm event
#'   baseline scale parameter (used when \code{logHR.trt = NULL}).
#' @param beta.event_trt Numeric vector. Regression coefficients for baseline
#'   covariates in the experimental-arm event-time proportional hazards model, used when
#'   \code{logHR.trt = NULL}.  Must
#'   have the same length and ordering as \code{cov_type}. Defaults to \code{beta.event}.
#'
#' @param blinded Logical. If \code{TRUE} (default), the generated interim dataset is blinded
#'   and treatment assignments in \code{data.eventcensor$trt} are set to \code{NA}.
#'   If \code{FALSE}, treatment assignments in \code{data.eventcensor$trt} are coded as \code{0} for
#'   control and \code{1} for the experimental group.
#'
#' @param assess_window Numeric scalar >= 0. Assessment window width. If > 0,
#'   observed event/censoring times are coarsened to the midpoint of the window
#'   containing \eqn{\min(T_\mathrm{event}, T_\mathrm{censor})}. Defaults to \code{0}.
#' @param seed Integer or \code{NULL}. Random seed for data generation. If the value is
#'   NULL then no random seed is used. Defaults to \code{123}.
#'
#' @return A list with elements:
#' \itemize{
#'   \item \code{data.enroll}: A data frame of observed enrollment information up to the
#'     interim data cut. Columns: subject index \code{No}, subject enrollment calendar
#'     time \code{enrolltime}, enrollment interarrival time \code{interarrivaltime},
#'     enrollment status \code{enrollstatus} (1 = enrolled, 0 = administratively
#'     censored enrollment process).
#'   \item \code{data.eventcensor}: A data frame of observed survival outcomes at the
#'     interim cut. Columns include subject index \code{No}, treatment assignment
#'     indicator \code{trt} (\code{NA} if \code{blinded = TRUE}), observed time \code{time}
#'     (administratively censored at interim cut), event status \code{eventstatus}
#'     (1 = event, 0 = right censored),
#'     random censoring status \code{censorstatus} (1 = random censoring before
#'      the interim data cut; 0 = otherwise), followed by covariates.
#'   \item \code{truesurvival}: Full underlying data without administrative censoring:
#'     \code{No}, \code{trt}, \code{t_event} (true underlying event time),
#'     \code{t_randcensor} (true underlying random censoring time),
#'     \code{t_event.obs} (underlying follow-up time without administrative censoring), \cr
#'     \code{t_event.obswithintervalassess} (underlying follow-up time without
#'     administrative censoring but applied with assessment windows),  \code{status}
#'     (1 = event before random censoring), \code{enrollmenttime}, plus covariates.
#'   \item \code{event.interim.obs}: Observed number of events at the interim
#'     data cut. If the prespecified interim event cutoff \code{E_cutoff} cannot be reached,
#'     this equals the maximum number of events observed.
#'   \item \code{event.max}: Number of events that would occur without administrative censoring
#'     (i.e., after accounting only for random censoring).
#'   \item \code{cuttime.true}: The true calendar time at which the cumulative number
#'   of observed events reaches the target event count. If the target number
#'   of events cannot be reached, this is the calendar time of the last observed
#'   event or censoring.
#'   \item \code{event.final.obs}: The latent true number of events at \code{cuttime.true}.
#' }
#'
#' @examples
#' ## --- Weibull event/censoring with a common PH treatment effect ---
#' data.weibull <- generate_data(
#'   N = 80, E_target = 50, E_cutoff = 25, p_trt = 0.5,
#'   cov_type = c("binary", "continuous"),
#'   cov_dist = c(0.5, sqrt(2)),
#'   beta.event  = c(0.2, 0.2),
#'   beta.censor = c(0, 0),
#'   logHR.trt = log(0.5),
#'   enroll_rate = 50/3, beta.event_trt = NULL,
#'   dist.event = "Weibull", dist.censor = "Weibull",
#'   event.scale = 1/5^3, event.shape = 3,
#'   censor.scale = 10^(-6), censor.shape = 6,
#'   blinded = TRUE,
#'   assess_window = 2,
#'   seed = 1
#' )
#' names(data.weibull)
#'
#' ## --- Log-logistic event/censoring with a common PH treatment effect ---
#' data.logl <- generate_data(
#'   N = 80, E_target = 50, E_cutoff = 25, p_trt = 0.5,
#'   cov_type = c("binary", "continuous"),
#'   cov_dist = c(0.5, sqrt(2)),
#'   beta.event  = c(0.2, 0.2),
#'   beta.censor = c(0, 0),
#'   logHR.trt = log(0.5),
#'   enroll_rate = 50/3, beta.event_trt = NULL,
#'   dist.event = "Loglogistic", dist.censor = "Loglogistic",
#'   event.scale = 6, event.shape = 6,
#'   censor.scale = 20, censor.shape = 4,
#'   blinded = TRUE,
#'   assess_window = 2,
#'   seed = 1
#' )
#' summary(data.logl$truesurvival$t_event)
#' ### true underlying event time without administrative censoring
#'
#' ## --- Weibull arm-specific models (logHR.trt = NULL) ---
#' data.weibull.nonPH <- generate_data(
#'   N = 80, E_target = 50, E_cutoff = 25, p_trt = 0.5,
#'   cov_type = c("binary", "continuous"),
#'   cov_dist = c(0.5, sqrt(2)),
#'   beta.event  = c(0.2, 0.2),
#'   beta.censor = c(0, 0),
#'   logHR.trt = NULL,
#'   enroll_rate = 50/3,
#'   dist.event = "Weibull", dist.censor = "Weibull",
#'   event.scale = 1/5^3, event.shape = 3,     # control
#'   event.scale_trt = 1/6^3, event.shape_trt = 3, # experiment
#'   beta.event_trt = c(0.15, 0.2),
#'   censor.scale = 10^(-6), censor.shape = 6,
#'   blinded = TRUE,
#'   assess_window = 2,
#'   seed = 1
#' )
#' data.weibull.nonPH$cuttime.true
#'
#' @export
generate_data<-function(N, E_target, E_cutoff, p_trt,
                        cov_type, cov_dist,
                        logHR.trt=NULL, enroll_rate,
                        dist.event, dist.censor,
                        blinded = TRUE,
                        event.scale=NULL, event.shape=NULL,
                        censor.scale=NULL,censor.shape=NULL,
                        beta.event, beta.censor,
                        event.scale_trt = NULL,event.shape_trt = NULL,
                        beta.event_trt = if (is.null(logHR.trt)) beta.event else NULL,
                        assess_window=0, seed=123){
  # ---- helpers ----
  .stop  <- function(msg) stop(msg, call. = FALSE)
  .req1  <- function(x, name) if (is.null(x) || length(x) != 1) .stop(paste0(name, " must be a length-1 value."))
  .req   <- function(cond, msg) if (!isTRUE(cond)) .stop(msg)

  # returns TRUE if you have enough info to define baseline without solving
  .baseline_complete <- function(shape, scale) {
    # both shape & scale should be given
    (!is.null(shape) && !is.null(scale))
  }

  # ---- Block 1: basic scalar / length checks ----
  PH<-!is.null(logHR.trt)
  .req1(N, "N"); .req1(E_target, "E_target"); .req1(E_cutoff, "E_cutoff")
  .req1(p_trt, "p_trt"); .req1(enroll_rate, "enroll_rate");

  .req(N >= E_target, "Invalid input: N smaller than E_target.")
  .req(E_target >= E_cutoff, "Invalid input: E_target smaller than E_cutoff.")
  .req(E_target>=1, "Invalid input: E_target has to be at least 1.")

  cov_type <- as.character(cov_type)
  .req(length(cov_type) == length(cov_dist) &&
         length(cov_type) == length(beta.event) &&
         length(cov_type) == length(beta.censor),
       "Invalid input: different lengths of cov_type, cov_dist, beta.event and beta.censor.")

  .req(all(cov_type %in% c("binary","continuous")),
       "Elements of cov_type must be either 'binary' or 'continuous'.")

  .req(all(!(cov_type == "binary") | (cov_dist >= 0 & cov_dist <= 1)),
       "For binary covariates, cov_dist must be between 0 and 1.")
  if (any(cov_type == "binary" & cov_dist %in% c(0, 1))) {
    warning(
      "Binary covariates with cov_dist = 0 or 1 are degenerate (no variability). ",
      "Corresponding beta coefficients are not identifiable and act as constant offsets.",
      call. = FALSE
    )}

  .req(all(!(cov_type == "continuous") | (cov_dist >= 0)),
       "For continuous covariates, cov_dist must be greater than or equal to 0.")
  if (any(cov_type == "continuous" & cov_dist == 0)) {
    warning(
      "Continuous covariates with cov_dist = 0 are degenerate (no variability). ",
      "Corresponding beta coefficients are not identifiable and act as constant offsets.",
      call. = FALSE
    )
  }

  .req(assess_window >= 0, "Invalid assess_window value: must be non-negative.")
  .req(p_trt > 0 && p_trt < 1, "p_trt must be in (0,1).")
  .req(enroll_rate > 0, "enroll_rate must be > 0.")

  # ---- Block 2: distribution choices ----
  .req(dist.event  %in% c("Weibull","Loglogistic"), "dist.event must be 'Weibull' or 'Loglogistic'.")
  .req(dist.censor %in% c("Weibull","Loglogistic"), "dist.censor must be 'Weibull' or 'Loglogistic'.")

  # ---- Block 3: PH vs non-PH logic ----
  if (PH) {

    # optional: warn if user passes non-PH-only args
    if (!is.null(beta.event_trt) || !is.null(event.scale_trt) || !is.null(event.shape_trt) ) {
      warning("PH model: logHR.trt is non-NULL, so arm-specific baseline/covariate parameters (*_trt) are ignored.", call. = FALSE)
    }

  } else {
    .req(!is.null(beta.event_trt), "beta.event_trt cannot be NULL when PH is FALSE.")
    .req(length(beta.event_trt) == length(cov_type),
         "Invalid input: different lengths of cov_type and beta.event_trt.")
  }


  # ---- Block 4: baseline parameter completeness (control + censor + trt) ----
  .req(.baseline_complete(event.shape, event.scale),
       "Event baseline for control arm is under-specified: provide (event.shape & event.scale).")

  .req(.baseline_complete(censor.shape, censor.scale),
       "Censoring baseline is under-specified: provide (censor.shape & censor.scale).")

  if (!PH) {
    .req(.baseline_complete(event.shape_trt, event.scale_trt),
         "Event baseline for experimental arm is under-specified: provide (event.shape_trt & event.scale_trt).")
  }


  p<-length(cov_type)
  if(!is.null(seed)){set.seed(as.integer(seed))}

  # ---- Simulate covariates and treatment assignments ----
  treatment_ind<-rbinom(n=N,size=1,prob=p_trt)
  cov<-matrix(NA_real_,nrow=N,ncol=p)
  for (j in seq_len(p)){
    if(cov_type[j]=='binary'){cov[,j]<-rbinom(N,1,cov_dist[j])}
    else {cov[,j]<-rnorm(N,0,cov_dist[j])}
  }
  colnames(cov) <- paste0("X", seq_len(p))
  beta.event<-as.numeric(beta.event);beta.censor<-as.numeric(beta.censor)
  if(!PH){beta.event_trt<-as.numeric(beta.event_trt)}

  # ---- Simulate enrollment time ----
  t_interarrival<-rexp(N,rate=enroll_rate) # interarrival time zeta_i
  t_enroll<-cumsum(t_interarrival) # true enrollment time t_{ri} without adminitrative censoring

  #---- Simulate event time and censoring time
  lp.event<-drop(cov%*%beta.event); lp.censor<-drop(cov%*%beta.censor) # PH model
  if(!PH){lp.event_trt<-drop(cov%*%beta.event_trt)}

  ### If distribution is Weibull then we need to change the scale parameter to match rweibull
  if((dist.event=='Weibull')){event.scale<-event.scale^(-1/event.shape)}
  if((dist.censor=='Weibull')){censor.scale<-censor.scale^(-1/censor.shape)}
  if(!PH){if((dist.event=='Weibull')){event.scale_trt<-event.scale_trt^(-1/event.shape_trt)}}

  if (PH) {
    lp.event.final    <- lp.event + logHR.trt * treatment_ind
    event.shape.final <- rep(event.shape, N)
    event.scale.final <- rep(event.scale, N)
  } else {
    lp.event.final <- lp.event
    lp.event.final[treatment_ind == 1] <- lp.event_trt[treatment_ind == 1]

    event.shape.final <- rep(event.shape, N)
    event.shape.final[treatment_ind == 1] <- event.shape_trt

    event.scale.final <- rep(event.scale, N)
    event.scale.final[treatment_ind == 1] <- event.scale_trt
  }
  if(dist.event == 'Weibull'){
    t_event<-rweibull(n=N,shape=event.shape.final,scale=event.scale.final*exp(-(lp.event.final)/event.shape.final))
  }
  if(dist.event == "Loglogistic"){
    t_event<-rloglogistic_PH(n=N,shape=event.shape.final,scale=event.scale.final,lp=lp.event.final)
  }
  if(dist.censor == 'Weibull'){
    t_censor<-rweibull(n=N,shape=censor.shape,scale=censor.scale*exp(-lp.censor/censor.shape))
  }
  if(dist.censor == 'Loglogistic'){
    t_censor<-rloglogistic_PH(n=N,shape=censor.shape,scale=censor.scale,lp=lp.censor)
  }

  # get event / random censoring status and observed time (from enrollment) without taking into account administratively censoring
  status_event <- as.integer(t_event < t_censor)     # 1 if event before censor, else 0
  t_event.obs  <- pmin(t_event, t_censor)

  ## t_event.delta is the observed event time adjusting for assessment window
  delta<-assess_window
  if (delta > 0) {t_event.delta <- delta * floor(t_event.obs / delta) + delta / 2}
  else {t_event.delta<-t_event.obs} # after adjusting for assessment windows
  event.max <-sum(status_event)

  # ---- Interim data ----
  simdata<-data.frame(data.frame(No=1:N,trt = treatment_ind, time = t_event.delta, status = status_event),data.frame(cov)) # true data without administrative censoring at interim or final (but includes random censoring)
  data.interim<-trial_data_sim(simdata=simdata,accrual=t_enroll,eventtarget=E_cutoff)
  cut.idx.enroll<-(data.interim$Info$N_patient) # the index of the last patient already enrolled
  cutofftime<-data.interim$Info$DataCutTime # cutofftime is T_1 in the paper.

  # Observed enrollment time
  if(cut.idx.enroll==N){
    data.enroll<-data.frame(No=1:N,enrolltime=t_enroll,interarrivaltime=t_interarrival,enrollstatus=1)
  } else{
    data.enroll<-data.frame(No=1:(cut.idx.enroll+1),enrolltime = c(t_enroll[1:(cut.idx.enroll)],cutofftime),interarrivaltime=c(t_interarrival[1:(cut.idx.enroll)],cutofftime-t_enroll[cut.idx.enroll]),enrollstatus = c(rep(1,cut.idx.enroll),0)) #  (1) enrolltime is t_{ri} in the paper; (2) interarrivaltime is \zeta_i; (3) enrollstatus: =1 if already enrolled; =0 if still in the process (administratively censoring). It does not include the patients who have not started the enrollment process
  }

  # Observed event time
  data.eventcensor<-data.interim$data%>%
    dplyr::mutate(eventstatus=status,censorstatus=as.integer((dropout=='Y')*(eventtime<=cutofftime)))%>% # eventtime is the latent time to event/random censoring without administrative cutoff
    dplyr::select(c(No,trt,time,eventstatus,censorstatus)) # time: t^*_{ci} and t^*_{ei} (=min(t_{ei},t_{ci},T_1-t_{ri})); eventstatus: \delta_{ei} in the paper (=1 if event, =0 if right censored); censorstatus: 1-\delta_{ci} (=0 if random censored, =1 if event/administrative censored)
  data.eventcensor<-data.frame(data.eventcensor, cov[data.eventcensor$No,])
  if(blinded){data.eventcensor$trt<-NA}
  .req(all(data.eventcensor$time >= 0), "Internal error: negative observed time.")
  .req(all(!(data.eventcensor$censorstatus == 1L) | (data.eventcensor$eventstatus == 0L)),
       "Internal error: censorstatus==1 must imply eventstatus==0.")

  # Observed survival data at the final/E_target
  data.target<-trial_data_sim(simdata=simdata,accrual=t_enroll,eventtarget=E_target)


  # Return true data: we need to return the true survival+enroll data without interim
  simdata.return<-data.frame(data.frame(No=1:N,trt = treatment_ind, t_event=t_event,t_randcensor=t_censor, t_event.obs=t_event.obs, t_event.obswithintervalassess = t_event.delta, status = status_event, enrollmenttime = t_enroll),data.frame(cov))
  return(list(data.enroll=data.enroll,
              data.eventcensor=data.eventcensor,
              truesurvival=simdata.return,
              #interimdata=data.interim,
              event.interim.obs = data.interim$Info$N_event,
              event.max = event.max,
              #targetdata=data.target,
              cuttime.true = data.target$Info$DataCutTime,
              event.final.obs = data.target$Info$N_event))
}





#' Predict time of target event count from posterior draws
#'
#' @description
#' Given posterior draws of enrollment, event, and random censoring model parameters,
#' this function generates posterior predictive event/censor outcomes for:
#' \itemize{
#'   \item \eqn{\mathcal{B}_1}: subjects enrolled before the interim cutoff but without an event
#'         or random censoring by the cutoff;
#'   \item \eqn{\mathcal{B}_2}: subjects not yet enrolled by the interim cutoff.
#' }
#' It returns posterior predictive draws of the calendar time (from the time origin) at which
#' the cumulative number of events reaches \code{E_target}.
#'
#' @param N Total planned sample size.
#' @param E_target Target number of events.
#'
#' @param data.enroll A data frame of observed enrollment information up to the interim
#'   analysis time. Must contain columns
#'   \code{enrollstatus}, \code{enrolltime}, and \code{interarrivaltime}.
#'   The required columns follow the conventions used by \code{\link{generate_data}}:
#'   \itemize{
#'     \item \code{enrolltime}: calendar time of enrollment for each subject, measured
#'       from the trial time origin.
#'     \item \code{interarrivaltime}: time between consecutive enrollments.
#'     \item \code{enrollstatus}: enrollment status indicator with \code{1} indicating
#'       an observed enrollment time and \code{0} indicating administrative censoring
#'       of the enrollment process at the interim analysis time.
#'   }
#'   The \code{No} column is optional and represents a subject index used to align
#'   enrollment records with event and censoring data. If missing, it is created
#'   internally as \code{No = 1:nrow(data.enroll)}.
#'
#'   All subjects in \eqn{\mathcal{B}_1} (defined from \code{data.eventcensor}) must be
#'   present in \code{data.enroll} and are matched by \code{No}. When
#'   \code{nrow(data.eventcensor) < N}, \code{data.enroll} must include exactly one
#'   administratively censored enrollment row (\code{enrollstatus == 0}).
#'   See \code{\link{data_example}} (element \code{example_enroll}) for a concrete
#'   example of the expected data layout.
#' @param data.eventcensor A data frame of observed event and censoring outcomes at the interim
#'   analysis time. Must contain the columns
#'   \code{time}, \code{eventstatus}, and \code{censorstatus}, which follow these conventions:
#'   \itemize{
#'     \item \code{time}: observed follow up time, administratively censored at the interim
#'       analysis time when applicable.
#'     \item \code{eventstatus}: event indicator with \code{1} indicating an observed event and
#'       \code{0} indicating right censoring.
#'     \item \code{censorstatus}: random censoring indicator with \code{1} indicating random
#'       censoring before the interim analysis time and \code{0} otherwise.
#'   }
#'   When \code{blinded = FALSE}, the data frame must also contain \code{trt} coded as
#'   \code{0} or \code{1}. When \code{blinded = TRUE}, \code{trt} may be present but can be
#'   \code{NA}.
#'   The \code{No} column is optional and represents a subject index used to align this data
#'   frame with \code{data.enroll}. If missing, it is created internally as
#'   \code{No = 1:nrow(data.eventcensor)}.
#'
#'   Any columns other than \code{No}, \code{trt}, \code{time}, \code{eventstatus}, and
#'   \code{censorstatus} are treated as numeric baseline covariates. The covariates are
#'   taken in the order they appear in \code{data.eventcensor}, and this order must match
#'   the column ordering of \code{param.draw$beta_c} and \code{param.draw$beta_e}.
#'
#' @param blinded Logical. If \code{TRUE}, treatment for current subjects is taken from \code{param.draw$x}.
#'   If \code{FALSE}, treatment is taken from \code{data.eventcensor$trt}.
#'
#' @param param.draw List of posterior draws. Required elements:
#' \itemize{
#'   \item Enrollment model: \code{mu} (length \code{S}).
#'   \item Event model: \code{rho_e}, \code{lambda_0e}, \code{eta}, and optionally \code{beta_e}
#'         as an \code{S x J} matrix (or \code{NULL} for \code{J = 0}).
#'   \item Random censoring model: \code{rho_c}, \code{lambda_0c}, and optionally \code{beta_c}
#'         as an \code{S x J} matrix (or \code{NULL} for \code{J = 0}).
#'   \item If \code{blinded = TRUE}: \code{x}, an \code{S x n.present} matrix of posterior draws of treatment
#'         assignments for the \code{n.present = nrow(data.eventcensor)} currently observed subjects. The
#'         columns for \code{x} should match the rows of \code{data.eventcensor}.
#' }
#'
#' @param assess_window Non-negative numeric. If \code{> 0}, predicted event/censor times from enrollment
#'   are recorded at the midpoint of the assessment window in which they occur.
#' @param seed.pred Optional integer seed for the RNG used in posterior predictive simulation. It controls
#'  only the RNG used in posterior predictive simulation and is independent of the 'Stan' sampling seed.
#' @param p_trt Randomization probability for experimental treatment (in (0,1)).
#'   Required whenever future subjects are simulated (i.e., when \code{nrow(data.eventcensor) < N}).
#'   Also required when \code{blinded = TRUE}.
#' @param return_all Logical. If \code{TRUE}, also return other patient-level predictive draws in addition
#'   to \code{pred.T2}.
#'
#' @return
#' A list with element \code{pred.T2}: a numeric vector of length \code{S}, where each element is a
#' posterior predictive draw of the calendar time (from the time origin) at which the cumulative number
#' of events reaches \code{E_target}. Values may be \code{Inf} if fewer than \code{E_target} events occur
#' in that draw.
#'
#' If \code{return_all = TRUE}, the list additionally contains the following matrices, each with
#'  \code{nrow =} \eqn{|\mathcal{B}_1 \cup \mathcal{B}_2|} and \code{ncol = S + 2}. Column 1 is \code{No},
#' column 2 is \code{set} (1 = \eqn{\mathcal{B}_1}, 2 = \eqn{\mathcal{B}_2}), and columns 3:(S+2)
#' correspond to posterior predictive draws:
#' \itemize{
#'   \item \code{pred.eventtime}: predicted event time from enrollment (per patient, per draw).
#'   \item \code{pred.censortime}: predicted random censoring time from enrollment (per patient, per draw).
#'   \item \code{pred.indicator}: event indicator (\code{1} if event occurs before censoring; \code{0} otherwise).
#'   \item \code{pred.obstime}: predicted observed time from enrollment
#'     (\code{min(pred.eventtime, pred.censortime)}).
#'   \item \code{pred.enrolltime}: predicted enrollment time from the time origin. For patients in
#'     \eqn{\mathcal{B}_1}, this equals the observed enrollment time in \code{data.enroll}.
#'   \item \code{pred.interarrival}: predicted interarrival time between consecutive enrollments.
#'     For patients in \eqn{\mathcal{B}_1}, this is the observed interarrival time in \code{data.enroll}.
#'   \item \code{pred.obstime.delta}: predicted observed time recorded at the mid-point of the
#'     assessment window when \code{assess_window > 0}; otherwise equals \code{pred.obstime}.
#'   \item \code{pred.exacteventtime}: predicted event/censor time (with assessment-window recording applied)
#'     from the time origin, computed as \code{pred.enrolltime + pred.obstime.delta}.
#' }
#' @keywords internal
#' @noRd
predict_event.fromdraws<-function(N, E_target, data.enroll, data.eventcensor, param.draw,
                                  blinded=TRUE, assess_window=0,
                                  seed.pred=NULL,p_trt = NULL, return_all = FALSE
){


  .stop <- function(msg) stop(msg, call. = FALSE)
  .req  <- function(cond, msg) if (!isTRUE(cond)) .stop(msg)

  # -----0) basic scalar checks----
  .req(is.numeric(N) && length(N) == 1 && is.finite(N) && N == as.integer(N) && N >= 1,
       "N must be a positive integer scalar.")
  N <- as.integer(N)

  .req(is.numeric(E_target) && length(E_target) == 1 && is.finite(E_target) &&
         E_target == as.integer(E_target) && E_target >= 1,
       "E_target must be a positive integer scalar.")
  E_target <- as.integer(E_target)

  .req(is.numeric(assess_window) && length(assess_window) == 1 && is.finite(assess_window) &&
         assess_window >= 0,
       "assess_window must be a single non-negative numeric value.")

  .req(is.logical(blinded) && length(blinded) == 1 && !is.na(blinded),
       "blinded must be TRUE/FALSE (length 1).")
  .req(is.logical(return_all) && length(return_all) == 1 && !is.na(return_all),
       "return_all must be TRUE/FALSE (length 1).")

  if (!is.null(seed.pred)) {
    .req(is.numeric(seed.pred) && length(seed.pred) == 1 && is.finite(seed.pred),
         "seed.pred must be NULL or a single finite numeric value.")
    set.seed(as.integer(seed.pred))

  }


  n.present <- nrow(data.eventcensor)
  .req(n.present >= 1, "data.eventcensor must have at least 1 row.")
  .req(n.present <= N, "The current number of patients (nrow(data.eventcensor)) cannot exceed N.")

  # ----1) coerce/standardize columns----
  if (!is.data.frame(data.enroll))      .stop("data.enroll must be a data.frame.")

  # add No if missing
  if (!("No" %in% names(data.eventcensor))) {
    data.eventcensor <- data.frame(No = seq_len(nrow(data.eventcensor)), data.eventcensor)
    warning("`data.eventcensor` has no column `No` - created internally as seq_len(nrow(data.eventcensor))",
            call. = FALSE)
  }
  if (!("No" %in% names(data.enroll))) {
    data.enroll <- data.frame(No = seq_len(nrow(data.enroll)), data.enroll)
    warning("`data.enroll` has no column `No` - created internally as seq_len(nrow(data.enroll))",
            call. = FALSE)
  }

  # required columns for data.eventcensor and data.enroll
  ## covariate matrix extracted from data.eventcensor
  cov_obs <- .check_data.eventcensor(data.eventcensor, blinded = blinded)

  if (is.null(cov_obs)) {
    cov_obs <- matrix(nrow = n.present, ncol = 0)
  }
  .req(nrow(cov_obs) == n.present, "Internal error: covariate matrix row count mismatch.")


  ## data.enroll
  req_enroll <- c("No", "enrollstatus", "enrolltime", "interarrivaltime")
  miss_en <- setdiff(req_enroll, names(data.enroll))
  .req(length(miss_en) == 0,
       paste0("data.enroll missing required column(s): ", paste(miss_en, collapse = ", ")))

  # ----2) structural consistency checks----
  # enroll rows can be n.present or n.present+1 per your convention
  #.req(nrow(data.enroll) %in% c(n.present, n.present + 1),
  #     "data.enroll should have the same number of rows as data.eventcensor or exactly one more.")

  # uniqueness + matching IDs
  .req(!anyDuplicated(data.eventcensor$No), "data.eventcensor$No must be unique.")
  .req(!anyDuplicated(data.enroll$No),      "data.enroll$No must be unique.")

  # basic type/sanity
  .req(all(is.finite(data.enroll$enrolltime)) && all(data.enroll$enrolltime >= 0),
       "data.enroll$enrolltime must be finite and >= 0.")
  .req(all(is.finite(data.enroll$interarrivaltime)) && all(data.enroll$interarrivaltime >= 0),
       "data.enroll$interarrivaltime must be finite and >= 0.")
  .req(all(data.enroll$enrollstatus %in% c(0, 1)),
       "data.enroll$enrollstatus must be 0/1.")

  # event target sanity
  E_present <- sum(data.eventcensor$eventstatus == 1)
  .req(E_present < E_target,
       "The current event number is already >= E_target; nothing to predict.")
  E_more <- E_target - E_present

  # ----3) define B1 / B2 indices early (needed for later checks)----
  idx_B1 <- data.eventcensor$No[data.eventcensor$eventstatus == 0 & data.eventcensor$censorstatus == 0]
  n.B1 <- length(idx_B1)
  n.B2 <- N - n.present

  row_B1 <- match(idx_B1, data.eventcensor$No)
  row_B1_enroll <- match(idx_B1, data.enroll$No)
  .req(!anyNA(row_B1), "Internal error: B1 indices not found in data.eventcensor.")
  .req(!anyNA(row_B1_enroll), "Some B1 patients are missing in data.enroll (No. mismatch).")
  .req(all(idx_B1 %in% data.enroll$No),
       "All B1 patients (eventstatus==0 & censorstatus==0) must appear in data.enroll$No.")


  # ----4) param.draw checks (Weibull hazard-scale parameterization)----
  .req(is.list(param.draw), "param.draw must be a list.")

  need_scalars <- c("lambda_0e", "lambda_0c", "eta", "rho_e", "rho_c", "mu")
  miss_pd <- need_scalars[!need_scalars %in% names(param.draw)]
  .req(length(miss_pd) == 0,
       paste0("param.draw missing required element(s): ", paste(miss_pd, collapse = ", ")))

  S <- length(param.draw$lambda_0e)
  .req(S >= 1, "param.draw must contain at least one posterior draw (S >= 1).")
  for (nm in need_scalars) {
    .req(length(param.draw[[nm]]) == S, paste0("param.draw$", nm, " must have length S."))
    .req(all(is.finite(param.draw[[nm]])), paste0("param.draw$", nm, " must be finite."))
  }
  .req(all(param.draw$rho_e > 0) && all(param.draw$rho_c > 0), "rho_e and rho_c must be > 0.")
  .req(all(param.draw$lambda_0e > 0) && all(param.draw$lambda_0c > 0), "lambda_0e and lambda_0c must be > 0.")
  .req(all(param.draw$mu > 0), "mu must be > 0.")

  # beta matrices: allow NULL meaning 0 covariates
  if (is.null(param.draw$beta_e)) param.draw$beta_e <- matrix(nrow = S, ncol = 0)
  if (is.null(param.draw$beta_c)) param.draw$beta_c <- matrix(nrow = S, ncol = 0)
  .req(is.matrix(param.draw$beta_e) && nrow(param.draw$beta_e) == S, "param.draw$beta_e must be an S x J matrix.")
  .req(is.matrix(param.draw$beta_c) && nrow(param.draw$beta_c) == S, "param.draw$beta_c must be an S x J matrix.")
  J <- ncol(param.draw$beta_e)
  .req(ncol(param.draw$beta_c) == J, "param.draw$beta_e and param.draw$beta_c must have the same number of columns (J).")

  .req(ncol(cov_obs) == J,
       paste0("Covariate dimension mismatch: data.eventcensor has ", ncol(cov_obs),
              " covariate columns but param.draw$beta_* have J=", J, "."))

  cov_B1 <- if (n.B1 == 0) matrix(nrow = 0, ncol = J) else cov_obs[row_B1, , drop = FALSE]

  # ----5) blinded logic + treatment draw requirements----
  if (blinded) {
    .req(!is.null(p_trt), "p_trt must be provided when blinded = TRUE.")
    .req(!is.null(param.draw$x),"param.draw missing x when blinded = TRUE.")
    .req(is.matrix(param.draw$x) && nrow(param.draw$x) == S && ncol(param.draw$x) == n.present,
         "When blinded=TRUE, param.draw$x must be an S * n.present matrix of treatment draws for current patients.")
    treatment_ind <- param.draw$x # matches the row of data.eventcensor
  } else {
    # use observed treatment for current patients
    .req(all(data.eventcensor$trt %in% c(0, 1)), "When blinded=FALSE, data.eventcensor$trt must be 0/1.")
    treatment_ind <- matrix(data.eventcensor$trt, nrow = S, ncol = n.present, byrow = TRUE)
  }
  .req(all(treatment_ind %in% c(0,1)), "Treatment indicators must be coded as 0/1.")


  # IMPORTANT: you need p_trt whenever n.B2 > 0 (future patients) regardless of blinded
  if (n.B2 > 0) {
    .req(!is.null(p_trt), "p_trt must be provided when there are future (B2) patients to simulate.")
    .req(is.numeric(p_trt) && length(p_trt) == 1 && is.finite(p_trt) && p_trt > 0 && p_trt < 1,
         "p_trt must be a single numeric value in (0,1).")
  }




  # First predict for the patients already enrolled but havent gone through event/random censor
   if(n.B1>0){
    pred_B1<-matrix(0, nrow = n.B1,ncol=4)
    pred_B1[,1]<-idx_B1; # index
    pred_B1[,2]<-rep(1,n.B1); # indicator for B1 / B2
    pred_B1[,3]<-(data.eventcensor$time)[row_B1] # T_1-t_{ri}
    pred_B1[,4]<-(data.enroll$enrolltime)[row_B1_enroll] # t_{ri}
    colnames(pred_B1)<-c('No','set','time','enrolltime')
    pred_B1.eventtime<-pred_B1.censortime<-pred_B1.indicator<-pred_B1.obstime<-
      pred_B1.enrolltime<-pred_B1.interarrival<-pred_B1.obstime.delta<-
      pred_B1.exacteventtime<-pred_B1[,c(1,2),drop = FALSE]

    # the following parameters are all matrices of nrow=n.B1 and ncol = S
    # rows = patients, columns = posterior draws
    u_B1.event<-matrix(runif(S*n.B1),nrow=n.B1,ncol = S);u_B1.censor<-matrix(runif(S*n.B1),nrow=n.B1,ncol = S) # the uniform samples used in inverse transform sampling, each row is a patient, each col is a posterior draw
    treatment_ind_B1<-t(treatment_ind[,row_B1,drop = FALSE]) # nrow = n.B1, ncol = S (matrix of treatment assignments for B_1)
    hazardscale_B1.event<-matrix(param.draw$lambda_0e, nrow=n.B1, ncol=S, byrow=TRUE)*exp(cov_B1%*%(t(param.draw$beta_e))+treatment_ind_B1*matrix(param.draw$eta, nrow=n.B1, ncol=S, byrow=TRUE))
    hazardscale_B1.censor<-matrix(param.draw$lambda_0c, nrow=n.B1, ncol=S, byrow=TRUE)*exp(cov_B1%*%(t(param.draw$beta_c)))
    shape_B1.censor<-matrix(param.draw$rho_c,     nrow=n.B1, ncol=S, byrow=TRUE)
    shape_B1.event<-matrix(param.draw$rho_e,     nrow=n.B1, ncol=S, byrow=TRUE)
    timescale_B1.event<-hazardscale_B1.event^(-1/shape_B1.event)
    timescale_B1.censor<-hazardscale_B1.censor^(-1/shape_B1.censor)
    time.truncate<-matrix(pred_B1[,"time"], nrow = n.B1, ncol = S) # each column is T_1-t_{ri}
    quantile_B1.censor<-u_B1.censor*pweibull(q=time.truncate,shape = shape_B1.censor,scale = timescale_B1.censor,lower.tail=FALSE)
    quantile_B1.event<-u_B1.event*pweibull(q=time.truncate,shape = shape_B1.event, scale=timescale_B1.event, lower.tail=FALSE)
    pred_B1.eventtime<-cbind(pred_B1.eventtime,qweibull(p=quantile_B1.event,shape=shape_B1.event, scale=timescale_B1.event, lower.tail=FALSE))
    pred_B1.censortime<-cbind(pred_B1.censortime,qweibull(p=quantile_B1.censor,shape=shape_B1.censor,scale=timescale_B1.censor,lower.tail=FALSE))
    pred_B1.indicator<-cbind(pred_B1.indicator,(pred_B1.eventtime[,c(3:(2+S)),drop=FALSE]<pred_B1.censortime[,c(3:(2+S)),drop=FALSE])*1)
    pred_B1.obstime<-cbind(pred_B1.obstime,pmin(pred_B1.eventtime[,c(3:(2+S)),drop=FALSE],pred_B1.censortime[,c(3:(2+S)),drop=FALSE])) # observed time from enrollment
    pred_B1.enrolltime<-cbind(pred_B1.enrolltime,matrix(pred_B1[,'enrolltime'],nrow=n.B1,ncol = S))
    pred_B1.interarrival<-cbind(pred_B1.interarrival,matrix((data.enroll$interarrivaltime)[row_B1_enroll],nrow=n.B1,ncol=S))

    # Need to adjust for assessment windows
    if(assess_window>0){
      pred_B1.obstime.delta<-cbind(pred_B1.obstime.delta,floor(pred_B1.obstime[,c(3:(2+S)),drop=FALSE]/assess_window)*assess_window+assess_window/2)
    }
    else{
      pred_B1.obstime.delta<-pred_B1.obstime
    }

    # Also adjust for enrollment
    pred_B1.exacteventtime<-cbind(pred_B1.exacteventtime, (pred_B1.obstime.delta[,c(3:(2+S)),drop=FALSE]+pred_B1.enrolltime[,c(3:(2+S)),drop=FALSE])) # event/randomcensor time from the time origin


  }
  else{
    pred_B1<-matrix(nrow = 0,ncol=4)
    colnames(pred_B1)<-c('No','set','time','enrolltime')
    pred_B1<-cbind(pred_B1,matrix(nrow=0,ncol=S))
    pred_B1.eventtime<-pred_B1.censortime<-pred_B1.indicator<-pred_B1.obstime<-pred_B1.enrolltime<-pred_B1.interarrival<-pred_B1.obstime.delta<-pred_B1.exacteventtime<-pred_B1[,c(1,2,(5:(4+S))),drop=FALSE]
  }


  # Now work on predicting for the patients not yet enrolled
  if(n.B2>0){ # in this case data.eventcensor has one row less than data.enroll
    #if(nrow(data.enroll)==n.present){stop('data.enroll should have one row more than data.eventcensor.')}
    .req(sum(data.enroll$enrollstatus == 0) == 1,
         "data.enroll must have exactly one patient whose enrollment is administratively censored (enrollstatus==0).")
    .req(any(data.enroll$enrollstatus == 1),
         "Cannot anchor future enrollment times: no enrolled subjects (enrollstatus==1) in data.enroll.")


    # result matrices
    B2_No<-c(data.enroll$No[which(data.enroll$enrollstatus==0)],max(union(data.enroll$No,data.eventcensor$No))+seq_len(n.B2-1))
    pred_B2<-matrix(0, nrow = n.B2, ncol = 2)
    pred_B2[,1] <- B2_No # index
    pred_B2[,2]<-rep(2,n.B2)
    colnames(pred_B2)[1:2]<-c('No','set')
    pred_B2.eventtime<-pred_B2.censortime<-pred_B2.indicator<-pred_B2.obstime<-pred_B2.obstime.delta<-pred_B2.exacteventtime<-pred_B2[,1:2,drop = FALSE]

    # the following parameters are all matrices of nrow=n.B2 and ncol = S if not specifically explained
    ## event and censor predictions
    treatment_ind_B2<-matrix(rbinom(n=n.B2*S,size=1,prob = p_trt),nrow=n.B2,ncol=S) # nrow=n.B2 and ncol = S
    ### sample the covariates
    I <- matrix(sample.int(n.present, size = n.B2 * S, replace = TRUE),nrow = n.B2, ncol = S)
    Z_big   <- cov_obs[as.vector(I) , , drop = FALSE]       # (n.B2*S) x J
    beta_big_B2.event <- param.draw$beta_e[rep(seq_len(S), each = n.B2), , drop = FALSE]  # (n.B2*S) x J
    beta_big_B2.censor <- param.draw$beta_c[rep(seq_len(S), each = n.B2), , drop = FALSE]  # (n.B2*S) x J
    LP_cov_B2.event <- matrix(rowSums(Z_big * beta_big_B2.event), nrow = n.B2, ncol = S)          # n.B2 x S
    LP_cov_B2.censor<- matrix(rowSums(Z_big * beta_big_B2.censor), nrow = n.B2, ncol = S)          # n.B2 x S
    hazardscale_B2.event<-matrix(param.draw$lambda_0e, nrow=n.B2, ncol=S, byrow=TRUE)*exp(LP_cov_B2.event+treatment_ind_B2*matrix(param.draw$eta,  nrow=n.B2, ncol=S, byrow=TRUE))
    hazardscale_B2.censor<-matrix(param.draw$lambda_0c, nrow=n.B2, ncol=S, byrow=TRUE)*exp(LP_cov_B2.censor)
    shape_B2.censor<-matrix(param.draw$rho_c, nrow=n.B2, ncol=S, byrow=TRUE)
    shape_B2.event<-matrix(param.draw$rho_e, nrow=n.B2, ncol=S, byrow=TRUE)
    timescale_B2.event<-hazardscale_B2.event^(-1/shape_B2.event)
    timescale_B2.censor<-hazardscale_B2.censor^(-1/shape_B2.censor)
    ### sample event and censor time
    pred_B2.eventtime<-cbind(pred_B2.eventtime,matrix(rweibull(n=n.B2*S,shape = shape_B2.event,scale=timescale_B2.event),nrow=n.B2,ncol = S))
    pred_B2.censortime<-cbind(pred_B2.censortime,matrix(rweibull(n=n.B2*S,shape=shape_B2.censor,scale=timescale_B2.censor),nrow=n.B2,ncol=S))
    pred_B2.indicator<-cbind(pred_B2.indicator,(pred_B2.eventtime[,c(3:(2+S)),drop=FALSE]<pred_B2.censortime[,c(3:(2+S)),drop=FALSE])*1)
    pred_B2.obstime<-cbind(pred_B2.obstime,pmin(pred_B2.eventtime[,c(3:(2+S)),drop=FALSE],pred_B2.censortime[,c(3:(2+S)),drop=FALSE])) # observed time from enrollment

    # Need to adjust for assessment windows
    if(assess_window>0){
      pred_B2.obstime.delta<-cbind(pred_B2.obstime.delta,floor(pred_B2.obstime[,c(3:(2+S)),drop=FALSE]/assess_window)*assess_window+assess_window/2)
    }
    else{
      pred_B2.obstime.delta<-pred_B2.obstime
    }

    ## enrollment time
    last_enroll_time <- max(data.enroll$enrolltime[data.enroll$enrollstatus == 1], na.rm = TRUE)
    .req(is.finite(last_enroll_time), "Cannot determine last observed enrollment time from data.enroll.")
    pred_B2.enrolltime<-pred_B2.interarrival<-pred_B2[,1:2,drop = FALSE]
    u_B2.enroll<-matrix(runif(S*n.B2),nrow=n.B2,ncol = S)
    mu_B2<-matrix(param.draw$mu, nrow=n.B2, ncol=S, byrow=TRUE)
    time.truncate.enroll<-matrix(c((data.enroll$interarrivaltime)[which(data.enroll$enrollstatus == 0)],rep(0,n.B2-1)), nrow = n.B2, ncol = S)
    quantile_B2.enroll<-u_B2.enroll*pexp(q=time.truncate.enroll,rate = mu_B2,lower.tail=FALSE)
    pred_B2.interarrival<-cbind(pred_B2.interarrival,qexp(p=quantile_B2.enroll,rate = mu_B2, lower.tail = FALSE))
    pred_B2.enrolltime <- cbind(
      pred_B2.enrolltime,
      matrix(apply(pred_B2.interarrival[, 3:(2+S), drop = FALSE], 2, cumsum), nrow = n.B2) + last_enroll_time
    )

    pred_B2.exacteventtime<-cbind(pred_B2.exacteventtime, (pred_B2.obstime.delta[,c(3:(2+S)),drop=FALSE]+pred_B2.enrolltime[,c(3:(2+S)),drop=FALSE])) # event/randomcensor time from the time origin
  }
  else{ # all N patients are enrolled
    #if(nrow(data.enroll)!=n.present){stop('data.enroll should have the same row number as data.eventcensor.')}
    if(length(which(data.enroll$enrollstatus==0))>0){stop('data.enroll should not have patients whose enrollment is administratively censored.')}
    pred_B2<-matrix(nrow = 0,ncol=2)
    colnames(pred_B2)<-c('No','set')
    pred_B2<-cbind(pred_B2,matrix(nrow=0,ncol=S))
    pred_B2.eventtime<-pred_B2.censortime<-pred_B2.indicator<-pred_B2.obstime<-pred_B2.enrolltime<-pred_B2.interarrival<-pred_B2.obstime.delta<-pred_B2.exacteventtime<-pred_B2[,c(1:(2+S)),drop=FALSE]
  }

  # Now we get the whole predicted event/censor time matrix for patients in B_1 + B_2
  pred.eventtime      <- as.matrix(rbind(pred_B1.eventtime,      pred_B2.eventtime))
  pred.censortime     <- as.matrix(rbind(pred_B1.censortime,     pred_B2.censortime))
  pred.indicator      <- as.matrix(rbind(pred_B1.indicator,      pred_B2.indicator))
  pred.obstime        <- as.matrix(rbind(pred_B1.obstime,        pred_B2.obstime))
  pred.enrolltime     <- as.matrix(rbind(pred_B1.enrolltime,     pred_B2.enrolltime))
  pred.interarrival   <- as.matrix(rbind(pred_B1.interarrival,   pred_B2.interarrival))
  pred.obstime.delta  <- as.matrix(rbind(pred_B1.obstime.delta,  pred_B2.obstime.delta))
  pred.exacteventtime <- as.matrix(rbind(pred_B1.exacteventtime, pred_B2.exacteventtime))

  pred.T2 <- vapply(seq_len(S), function(s, k) {
    t_event <- pred.exacteventtime[, 3:(2+S), drop = FALSE][
      pred.indicator[, 3:(2+S), drop = FALSE][, s] == 1, s]
    if (length(t_event) < k) Inf else sort(t_event, partial = k)[k]
  }, numeric(1), k = E_more)


  if(return_all){return(list(pred.T2=pred.T2,
                             pred.eventtime=pred.eventtime,
                             pred.censortime=pred.censortime,
                             pred.indicator=pred.indicator,
                             pred.obstime=pred.obstime,
                             pred.enrolltime=pred.enrolltime,
                             pred.interarrival=pred.interarrival,
                             pred.obstime.delta=pred.obstime.delta,
                             pred.exacteventtime=pred.exacteventtime))}
  return(list(pred.T2 = pred.T2))

}




#' Predict the calendar time at which a target number of events is
#' reached from interim analysis data
#'
#' @description
#' Fits enrollment, event-time, and random censoring models to data observed at the
#' interim analysis and predicts the calendar time at which the cumulative number
#'  of events reaches \code{E_target}.
#'
#' @details
#' The function fits three components in sequence using \code{\link{fit_models}}:
#' an enrollment model, an event-time model, and a random censoring model, all based on
#' data observed at the interim analysis. It then performs posterior predictive simulation
#' for two groups of subjects:
#' \itemize{
#'   \item \eqn{\mathcal{B}_1}: subjects enrolled before the interim analysis who have not
#'         experienced an event or random censoring by the interim data cut;
#'   \item \eqn{\mathcal{B}_2}: subjects not yet enrolled by the interim analysis.
#' }
#' For each posterior draw, the function estimates the calendar time
#' at which the cumulative number of events reaches \code{E_target}.
#'
#' @param N Integer. Total planned sample size (maximum number of subjects
#'   that can be enrolled in the trial).
#' @param E_target Integer. Target number of events for the final analysis.
#'
#' @param data.enroll A data frame of observed enrollment information up to the
#'   interim analysis time. Must contain the columns \code{enrollstatus},
#'   \code{enrolltime}, and \code{interarrivaltime}. These columns follow the
#'   conventions defined by \code{\link{generate_data}}:
#'   \itemize{
#'     \item \code{enrolltime}: Calendar time of enrollment for each subject,
#'       measured from the trial time origin.
#'     \item \code{interarrivaltime}: Time between consecutive enrollments.
#'     \item \code{enrollstatus}: Enrollment status indicator with \code{1}
#'       indicating an observed enrollment time and \code{0} indicating
#'       administrative censoring of the enrollment process at the interim
#'       analysis time.
#'   }
#'   The \code{No} column is optional and provides a subject index used to align
#'   enrollment records with \code{data.eventcensor}. If missing, it is created
#'   internally as \code{No = seq_len(nrow(data.enroll))} and a warning is returned.
#'
#'   All subjects in \eqn{\mathcal{B}_1} (subjects enrolled before the interim
#'   analysis who have not yet experienced an event or random censoring,
#'   as defined from \code{data.eventcensor}) must be
#'   present in \code{data.enroll} and are matched by \code{No}.
#'   When not all \code{N} subjects have enrolled by the interim analysis
#'   (i.e., \code{nrow(data.eventcensor) < N}), \code{data.enroll} must contain
#'   exactly one administratively censored enrollment record (\code{enrollstatus == 0}).
#'   See \code{\link{data_example}} (element \code{example_enroll}) for a concrete
#'   example of the expected data layout.
#'
#' @param data.eventcensor A data frame of observed event and censoring outcomes at the interim
#'   analysis time. Must contain the columns
#'   \code{time}, \code{eventstatus}, and \code{censorstatus}.
#'   These columns follow the conventions defined by \code{\link{generate_data}}:
#'   \itemize{
#'     \item \code{time}: observed follow up time, administratively censored at the interim
#'       analysis.
#'   \item \code{eventstatus}: event indicator (\code{1} = event, \code{0} = right-censored).
#'   \item \code{censorstatus}: random censoring indicators \eqn{\delta_{ci}}
#'   (\code{1} = random censoring observed,
#'   \code{0} = no random censoring, including administrative censoring
#'   or observed event).
#'  }
#'   When \code{blinded = FALSE}, the data frame must also contain the treatment assignment
#'   indicator column \code{trt} coded as
#'   \code{0} or \code{1}. When \code{blinded = TRUE}, \code{trt} may be present but ignored.
#'   The \code{No} column is optional and represents a subject index used to align this data
#'   frame with \code{data.enroll}. If missing, it is created internally as
#'   \code{No = 1:nrow(data.eventcensor)} and a warning is returned.
#'
#'   Any columns other than \code{No}, \code{trt}, \code{time}, \code{eventstatus}, and
#'   \code{censorstatus} are treated as numeric baseline covariates.
#'   See \code{\link{fit_models}} for covariate requirements
#'
#' @param blinded Logical. If \code{TRUE} (default), the interim analysis is blinded and treatment assignments
#'   for current subjects are not observed in the data. If \code{FALSE}, the analysis
#'   is unblinded and observed treatment assignments are used.
#'
#' @param p_trt
#' Numeric scalar in \eqn{[0,1]} giving the prespecified randomization probability
#' of assignment to the experimental treatment arm. Required only if \code{blinded = TRUE};
#' ignored otherwise. Defaults to \code{NULL}.
#'
#' @param hyperparams_enroll List of prior hyperparameters for the enrollment model.
#'   See \code{\link{fit_models}} for details.
#' @param hyperparams_censor List of prior hyperparameters for the censoring model.
#'   See \code{\link{fit_models}} for details.
#' @param hyperparams_event List of prior hyperparameters for the event-time model.
#'   See \code{\link{fit_models}} for details.
#'
#' @param chains
#' Number of Markov chain Monte Carlo (MCMC) chains. Defaults to \code{4}.
#'
#' @param iter
#' Number of iterations per chain (including warmup). Defaults to \code{4000}.
#'
#' @param mc.cores Integer. Number of CPU cores to use when executing Markov
#'   chains in parallel via \code{\link[rstan]{sampling}}. Defaults to \code{1}.
#'   We recommend setting \code{mc.cores} to the maximum number of processors
#'   supported by the available hardware and memory, up to the number of
#'   chains.
#'
#' @param seed.fit
#' Optional random seed(s) passed to \code{\link[rstan]{sampling}} for reproducibility.
#' Can be specified as:
#' \itemize{
#'   \item a single integer or \code{NULL}, in which case the same seed
#'     is used for all three submodels, or
#'   \item a list of up to three integers or \code{NULL}s, recycled to length 3,
#'     corresponding to the enrollment, censoring, and event-time models,
#'     respectively.
#' }
#' Use \code{NULL} to allow 'Stan' to select a seed internally.
#' Defaults to \code{list(123)}.
#'
#' @param refresh
#' Frequency of progress updates from \code{\link[rstan]{sampling}}. Set to \code{0} (NULL) to suppress output.
#'
#' @param warmup
#' Number of warmup (burn-in) iterations per chain. Must be strictly smaller
#' than \code{iter}. Defaults to \code{floor(iter / 2)}.
#'
#' @param control
#' Sampler control settings passed to \code{\link[rstan]{sampling}} for the three
#' submodels (enrollment, censoring, and event-time).
#' Can be specified as:
#' \itemize{
#'   \item a single named list of control parameters (shared across all
#'     three submodels), or
#'   \item a list of up to three named lists, recycled to length 3, giving
#'     separate control settings for the enrollment, censoring, and
#'     event-time models, respectively.
#' }
#' Typical entries include \code{adapt_delta} and \code{max_treedepth}.
#'
#' Defaults to \code{list(list(adapt_delta = 0.95))}.
#'
#' @param return_fit
#' Logical; if \code{TRUE}, also return the underlying 'rstan' \code{stanfit} objects for the
#' enrollment, censoring, and event models. Defaults to \code{FALSE}.
#'
#' @param quiet
#' Logical. If \code{TRUE} (default), suppress messages and diagnostic warnings
#' from 'Stan' during model fitting. Useful for large simulation studies.
#'
#' @param assess_window Non-negative numeric. If \code{> 0},
#'  predicted event/censor times from enrollment are recorded at the
#'  midpoint of the assessment window in which they occur. Defaults to \code{0}.
#'
#' @param seed.pred Optional integer seed for the RNG used in posterior predictive simulation. It controls
#' only the RNG used in posterior predictive simulation and is independent of the 'Stan' sampling seed.
#' Defaults to \code{list(123)}.
#' @param return_draws Logical. If \code{TRUE}, also return the posterior draws from the fitted submodels
#'  as \code{result$draws}. Defaults to \code{FALSE}.
#'
#' @return
#' An object of class \code{"BayesPET_predtime"}, which is a named list with components:
#' \itemize{
#'   \item \code{prediction}: A numeric vector of length \code{S},
#'   where \code{S} is the number of posterior draws from the fitted models.
#'   Each element is a posterior predictive draw of the calendar time at
#'   which the cumulative number of events reaches \code{E_target}.
#'   Values may be \code{Inf} if fewer than \code{E_target} events occur in that draw.
#'
#'   \item \code{fit}: Present only if \code{return_fit = TRUE}. A list of 'rstan' \code{stanfit} objects with
#'     components \code{enroll}, \code{censor}, and \code{event}, corresponding to the enrollment,
#'     censoring, and event-time models, respectively.
#'
#'   \item \code{draws}: Present only if \code{return_draws = TRUE}. A list of posterior draws of model
#'     parameters produced by the fitted submodels.
#'    This includes posterior draws from the enrollment, event-time, and censoring models.
#'
#'   \item \code{call}: The function call used to generate this object.
#' }
#' Methods include \code{\link{print}}, \code{\link{summary}}, and \code{\link{plot}}.
#'
#' @seealso
#' Other BayesPET prediction: \code{\link{plot.BayesPET_predtime}}, \code{\link{print.BayesPET_predtime}}, \cr
#' \code{\link{summary.BayesPET_predtime}}
#'
#' @examplesIf requireNamespace("rstan", quietly = TRUE)
#' \donttest{
#' data(data_example)
#' ## Reduced number of chains and iterations compared to defaults
#' ## to keep the example computationally manageable.
#' pred <- predict_eventtime(
#'   N = 200,
#'   E_target = 150,
#'   data.enroll = data_example$example_enroll,
#'   data.eventcensor = data_example$example_eventcensor,
#'   blinded = TRUE,
#'   p_trt = 0.5,
#'   chains = 2,
#'   iter = 2000,
#'   assess_window = 2,
#'   seed.fit = 1,
#'   seed.pred = 2,
#'   return_fit = TRUE,
#'   return_draws = TRUE,
#'   quiet = TRUE
#' )
#'
#' print(pred)
#' summary(pred)
#' plot(pred)
#' }
#' @export
predict_eventtime<-function(N, E_target, data.enroll, data.eventcensor, blinded=TRUE, p_trt = NULL,
                        hyperparams_enroll = list(), hyperparams_event = list(),
                        hyperparams_censor = list(), chains = 4, iter = 4000, warmup = floor(iter/2),
                        seed.fit =list(123), refresh = 0, control = list(list(adapt_delta = 0.95)), mc.cores = 1,
                        assess_window=0, seed.pred=1, return_fit = FALSE, quiet = TRUE,
                        return_draws = FALSE){

  call <- match.call()
  .stop <- function(msg) stop(msg, call. = FALSE)
  .req  <- function(cond, msg) if (!isTRUE(cond)) .stop(msg)

  # N / E_target scalar + integer-ish
  .req(is.numeric(N) && length(N)==1 && is.finite(N) && N==as.integer(N) && N>=1, "N must be positive integer.")
  .req(is.numeric(E_target) && length(E_target)==1 && is.finite(E_target) && E_target==as.integer(E_target) && E_target>=1,
       "E_target must be positive integer.")
  N <- as.integer(N); E_target <- as.integer(E_target)
  .req(E_target <= N, "E_target must be <= N.")

  # assess_window
  .req(is.numeric(assess_window) && length(assess_window)==1 && is.finite(assess_window) && assess_window>=0,
       "assess_window must be a single nonnegative number.")

  # data
  .req(is.data.frame(data.eventcensor) && nrow(data.eventcensor)>=1, "data.eventcensor must be a nonempty data.frame.")
  n.present <- nrow(data.eventcensor)
  .req(n.present <= N, "nrow(data.eventcensor) cannot exceed N.")

  .req(is.logical(return_fit) && length(return_fit)==1 && !is.na(return_fit),
       "return_fit must be TRUE/FALSE (length 1).")
  .req(is.logical(return_draws) && length(return_draws)==1 && !is.na(return_draws),
       "return_draws must be TRUE/FALSE (length 1).")


  # ---- fit the model ----
  param.draw <- fit_models(data.enroll = data.enroll, data.eventcensor = data.eventcensor,
                         blinded=blinded, if (blinded) p_trt else NULL,
                         hyperparams_enroll = hyperparams_enroll,
                         hyperparams_event = hyperparams_event,
                         hyperparams_censor = hyperparams_censor, chains = chains,
                         iter = iter, warmup = warmup, seed=seed.fit, mc.cores = mc.cores,
                         refresh = refresh, control = control, return_fit = return_fit,
                         quiet = quiet)


  # predict the event time
  prediction<-predict_event.fromdraws(N = N, E_target = E_target,
                                      data.enroll= data.enroll,
                                      data.eventcensor = data.eventcensor,
                                      param.draw = param.draw, blinded=blinded,
                                      assess_window=assess_window,
                                      seed.pred=seed.pred, p_trt = p_trt,
                                      return_all = FALSE)

  result<-list(prediction = prediction$pred.T2)
  if (isTRUE(return_fit)) {result$fit <- param.draw$fit}
  if (isTRUE(return_draws)) {
    # drop fit if present
    result$draws <- param.draw[setdiff(names(param.draw), "fit")] # will include `call` and preserve class attributes
  }
  class(result) <- c("BayesPET_predtime", "list")
  result$call <- call
  return(result)
}




#' Generate operating characteristics for event prediction
#'
#' @description Generate operating characteristics for multiple simulated trials
#'
#' @details
#' This function first simulates a two-arm time-to-event trial using
#' \code{\link{generate_data}} and constructs interim datasets
#' (\code{data.enroll} and \code{data.eventcensor}). It then fits the enrollment,
#' event-time, and random censoring models and generates posterior predictive draws
#' of the calendar time at which the cumulative number of
#' events reaches \code{E_target} using \code{\link{predict_eventtime}}.
#'
#' Only simulated datasets in which the target number of events \code{E_target}
#' is reachable are retained for analysis. Up to \code{nsim.max} datasets are
#' generated in order to obtain at most \code{nsim} valid replicates. As a result,
#' the retained replicates correspond to a subset of the attempted simulations,
#' and their associated seeds are recorded explicitly in the returned object.
#'
#' @param N Integer. Total planned sample size (maximum number of subjects
#'   that can be enrolled in the trial).
#' @param E_target Integer. Target number of events for the final analysis.
#' @param E_cutoff Integer. Target number of events for the interim analysis.
#' @param p_trt Scalar randomization probability to the experimental arm, \eqn{\gamma \in (0,1)}.
#'
#' @param cov_type Character vector specifying the distribution for each component
#'   of the covariate vector \eqn{\boldsymbol{Z} = (Z_1,\dots,Z_p)} used in the
#'   proportional hazards  model.
#'   Each element must be \code{"binary"} or \code{"continuous"}.
#'   If \code{NULL}, no covariates are used.
#'
#' @param cov_dist Numeric vector of the same length as \code{cov_type}, giving
#'   parameters for the covariate-generating distribution of each \eqn{Z_j}:
#'   \itemize{
#'     \item \code{"binary"}: \eqn{Z_j \sim \mathrm{Bernoulli}(p_j)} with
#'       \eqn{p_j = cov\_dist[j]}.
#'     \item \code{"continuous"}: \eqn{Z_j \sim N(0, \sigma_j^2)} with
#'       \eqn{\sigma_j = cov\_dist[j]}.
#'   }
#'
#' @param logHR.trt Numeric scalar giving the log hazard ratio for the experimental
#'   versus control arm in the event-time model. When \code{NULL} (default), the two treatment
#'   arms are generated from separate proportional hazards models with
#'   arm-specific baseline parameters and covariate effects.
#'
#' @param enroll_rate Positive numeric scalar specifying the enrollment rate.
#'
#' @param dist.event Character. Baseline distribution for event times:
#'   \code{"Weibull"} or \code{"Loglogistic"}. This distribution family is the same
#'   for both arms.
#' @param dist.censor Character. Baseline distribution for random censoring times:
#'   \code{"Weibull"} or \code{"Loglogistic"}.
#'
#' @param event.shape Numeric scalar > 0. Control-arm event baseline shape parameter.
#' @param event.scale Numeric scalar > 0. Control-arm event baseline scale parameter.
#' @param beta.event Numeric vector. Regression coefficients for baseline
#'   covariates in the event-time proportional hazards model; must have the
#'   same length and ordering as \code{cov_type}.
#'
#' @param censor.shape Numeric scalar > 0. Random censoring baseline shape parameter.
#' @param censor.scale Numeric scalar > 0. Random censoring baseline scale parameter.
#' @param beta.censor Numeric vector. Regression coefficients for baseline
#'   covariates in the random censoring-time proportional hazards model; must
#'   have the same length and ordering as \code{cov_type}.
#'
#' @param event.shape_trt Numeric scalar > 0. Experimental-arm event baseline shape parameter
#'   (used when \code{logHR.trt = NULL}). Defaults to \code{NULL}.
#' @param event.scale_trt Numeric scalar > 0. Experimental-arm event
#'   baseline scale parameter (used when \code{logHR.trt = NULL}). Defaults to \code{NULL}.
#' @param beta.event_trt Numeric vector. Regression coefficients for baseline
#'   covariates in the experimental-arm event-time proportional hazards model, used when
#'   \code{logHR.trt = NULL}. Must
#'   have the same length and ordering as \code{cov_type}. Defaults to \code{beta.event}.
#'
#' @param blinded Logical. If \code{TRUE} (default), the generated interim dataset is blinded
#'   and treatment assignments in \code{data.eventcensor$trt} are set to \code{NA}.
#'   If \code{FALSE}, treatment assignments in \code{data.eventcensor$trt} are coded as \code{0} for
#'   control and \code{1} for the experimental group.
#'
#' @param assess_window Numeric scalar >= 0. Assessment window width. If > 0,
#'   observed event/censoring times are coarsened to the midpoint of the window
#'   containing \eqn{\min(T_\mathrm{event}, T_\mathrm{censor})}. Defaults to \code{0}.
#' @param seed Integer. Base random seed used to generate simulated datasets.
#'   Replicate \code{i} uses seed \code{seed + i - 1}. Defaults to \code{123}.
#'
#' @param hyperparams_enroll List of prior hyperparameters for the enrollment model.
#'   See \code{\link{fit_models}} for details.
#' @param hyperparams_censor List of prior hyperparameters for the censoring model.
#'   See \code{\link{fit_models}} for details.
#' @param hyperparams_event List of prior hyperparameters for the event-time model.
#'   See \code{\link{fit_models}} for details.
#'
#' @param chains
#' Number of Markov chain Monte Carlo (MCMC) chains. Defaults to \code{4}.
#'
#' @param iter
#' Number of iterations per chain (including warmup). Defaults to \code{4000}.
#'
#' @param refresh
#' Frequency of progress updates from \code{\link[rstan]{sampling}}. Set to \code{0} (default) to suppress output.
#'
#' @param warmup
#' Number of warmup (burn-in) iterations per chain. Must be strictly smaller
#' than \code{iter}. Defaults to \code{floor(iter / 2)}.
#'
#' @param control
#' Sampler control settings passed to \code{\link[rstan]{sampling}} for the three
#' submodels (enrollment, censoring, and event-time).
#' Can be specified as:
#' \itemize{
#'   \item a single named list of control parameters (shared across all
#'     three submodels), or
#'   \item a list of up to three named lists, recycled to length 3, giving
#'     separate control settings for the enrollment, censoring, and
#'     event-time models, respectively.
#' }
#' Typical entries include \code{adapt_delta} and \code{max_treedepth}.
#'
#' Defaults to \code{list(list(adapt_delta = 0.95))}.
#'
#' @param nsim Integer. Number of valid simulated trial replicates used to compute
#' operating characteristics. Defaults to \code{1000}.
#' @param nsim.max Integer. Maximum number of simulated datasets to attempt in order
#'   to obtain \code{nsim} valid replicates (i.e., replicates that can reach \code{E_target}).
#'   Defaults to \code{3*n.sim}.
#' @param n_workers
#' Integer or \code{NULL}. Number of parallel workers used by the 'future'
#' backend. If \code{NULL}, defaults to one fewer than the number of available
#' CPU cores.
#'
#' @param \dots
#' Additional arguments passed to \code{\link[furrr]{future_map}}.
#' These can be used to control parallel execution behavior (e.g.,
#' scheduling, chunk size, or progress reporting) and do not affect
#' data generation or model fitting.
#'
#' @return
#' An object of class \code{"BayesPET_oc"} containing operating
#' characteristics for event-time prediction, based on simulated trial
#' replicates for which the target number of events \code{E_target} is reachable.
#' The object is a named list with components:
#' \itemize{
#'   \item \code{replicate}: A data frame with one row per retained (valid) simulated trial
#'     replicate. Columns include:
#'     \itemize{
#'       \item \code{median}: Median of the posterior predictive draws of the
#'         calendar time at which \code{E_target} events are reached.
#'       \item \code{cuttime.true}: True calendar time at which the cumulative number
#'         of observed events in the simulated trial first reaches \code{E_target}.
#'       \item \code{difference}: Absolute prediction error,
#'         \code{abs(median - cuttime.true)}.
#'     }
#'   \item \code{n_valid}: Number of retained replicates.
#'   \item \code{n_attempt}: Total number of simulated datasets attempted.
#'   \item \code{nsim_target}: Target number of replicates requested.
#'   \item \code{nsim_max}: Maximum number of datasets that may be generated.
#'   \item \code{seed0}: Base random seed used to generate simulated datasets;
#'     equals the input argument \code{seed}.
#'   \item \code{seeds}: Integer vector of seeds corresponding to the retained
#'     replicates (one per row of \code{replicate}).
#'   \item \code{call}: Matched function call.
#' }
#'
#' @examplesIf requireNamespace("rstan", quietly = TRUE)
#' \donttest{
#' ## Using nsim = 2, chains = 2, and iter = 2000 to reduce runtime.
#' ## Use larger nsim, chains and iter in real analyses.
#' oc <- get_oc(
#'   N = 200, E_target = 150,
#'   E_cutoff = 75, p_trt = 0.5,
#'   cov_type = c("binary", "continuous"),
#'   cov_dist = c(0.5, 2),
#'   beta.event = c(-0.2, -0.2),
#'   beta.censor = c(0, 0),
#'   logHR.trt = log(0.65),
#'   enroll_rate = 16,
#'   dist.event = "Weibull", dist.censor = "Weibull",
#'   event.scale = 1/5^3, event.shape = 3,
#'   censor.scale = 1/10^6, censor.shape = 6,
#'   blinded = TRUE,
#'   assess_window = 2,
#'   seed = 1,
#'   chains = 2, iter = 2000,
#'   nsim = 2,
#'   n_workers = 1
#' )
#'
#' summary(oc)
#' }
#'
#' @family BayesPET operating characteristics
#'
#' @export
get_oc <- function(N, E_target, E_cutoff, p_trt,
                   cov_type, cov_dist,
                   logHR.trt=NULL, enroll_rate,
                   dist.event, dist.censor,
                   blinded = TRUE,
                   event.scale = NULL, event.shape = NULL,
                   censor.scale = NULL,censor.shape = NULL,
                   beta.event, beta.censor,
                   event.scale_trt = NULL,event.shape_trt = NULL,
                   beta.event_trt = if (is.null(logHR.trt)) beta.event else NULL,
                   assess_window = 0, seed = 123,
                   hyperparams_enroll = list(), hyperparams_event = list(),
                   hyperparams_censor = list(), chains = 4,
                   iter = 4000, warmup = floor(iter/2),
                   refresh = 0, control = list(list(adapt_delta = 0.95)),
                   nsim = 1000, nsim.max = 3*nsim, n_workers = 1,
                   ...) {

  if (!requireNamespace("future", quietly = TRUE)) stop("Please install 'future'.")
  if (!requireNamespace("furrr", quietly = TRUE))  stop("Please install 'furrr'.")
  .check_pos_int(seed,'seed')
  if (is.null(n_workers)) workers <- max(1L, parallel::detectCores() - 1L)
  else workers <- n_workers

  old_plan <- future::plan()
  on.exit(future::plan(old_plan), add = TRUE)
  future::plan(future::multisession, workers = workers)

  data.list <- list()
  seeds <- c()

  seed.0 <- seed
  i <- 0L
  n_attempt <- 0L

  # Generate data together (serial)
  while (TRUE) {
    # stop if reached max attempts
    if (n_attempt >= nsim.max) {
      # apply your nsim/nsim.max criterion using observed success rate
      success_rate <- if (n_attempt > 0L) i / n_attempt else 0
      target_rate  <- nsim / nsim.max

      if (success_rate < target_rate) {
        warning(
          paste0(
            "Low success rate in data generation: ",
            round(100 * (1 - success_rate), 1), "% of attempts cannot reach E_target; ",
            "expected at most ", round(100 * (1 - target_rate), 1),
            "% (= 1 - nsim/nsim.max). Please check input parameters."
          ),
          call. = FALSE
        )
      }
      warning(paste0("Using a total of ", i, " valid data replicates to calculate the oc."), call. = FALSE)
      break
    }

    data.i <- generate_data(
      N = N, E_target = E_target, E_cutoff = E_cutoff, p_trt = p_trt,
      cov_type = cov_type, cov_dist = cov_dist, logHR.trt = logHR.trt,
      enroll_rate = enroll_rate, dist.event = dist.event, dist.censor = dist.censor,
      blinded = blinded, event.scale = event.scale, event.shape = event.shape,
      censor.scale = censor.scale, censor.shape = censor.shape,
      beta.event = beta.event, beta.censor = beta.censor,
      event.scale_trt = event.scale_trt, event.shape_trt = event.shape_trt,
      beta.event_trt = beta.event_trt,
      assess_window = assess_window, seed = seed
    )
    n_attempt <- n_attempt + 1L

    if (data.i$event.final.obs >= E_target) { # E_target can actually be reached
      data.list[[i + 1L]] <- data.i
      i <- i + 1L
      seeds[i] <- seed
    }

    if (length(data.list) >= nsim) break

    seed <- seed + 1L
  }

  # Run OC replicates in parallel
  old_opt <- options(future.rng.onMisuse = "ignore")
  on.exit(options(old_opt), add = TRUE)
  out_list <- withCallingHandlers(
    furrr::future_map(
    seq_len(length(seeds)),
    function(i) {
      s <- seeds[i]

      result.i <- predict_eventtime(
        N = N, E_target = E_target,
        data.enroll = data.list[[i]]$data.enroll,
        data.eventcensor = data.list[[i]]$data.eventcensor,
        blinded = blinded,
        p_trt = p_trt,
        hyperparams_enroll = hyperparams_enroll,
        hyperparams_event  = hyperparams_event,
        hyperparams_censor = hyperparams_censor,
        chains = chains, iter = iter, warmup = warmup,
        seed.fit = list(s + 1), refresh = refresh,
        control = control, mc.cores = 1,
        assess_window = assess_window, seed.pred = s + 2,
        return_fit = FALSE, quiet = TRUE,
        return_draws = FALSE
      )

      summary.i <- list(
        median = median(result.i$prediction),
        cuttime.true = data.list[[i]]$cuttime.true,
        E_cut.true = data.list[[i]]$event.final.obs
      )

      df.add <- c(
        median = summary.i$median,
        cuttime.true = summary.i$cuttime.true,
        difference = abs(summary.i$median - summary.i$cuttime.true)
      )

      df.add <- data.frame(t(df.add))
      colnames(df.add) <- c("median", "cuttime.true", "difference")
      list(df = df.add)
    },
    .options = furrr::furrr_options(seed = NULL),
    ...
  ),
  warning = function(w) {
    msg <- conditionMessage(w)
    if (grepl("UNRELIABLE VALUE", msg, fixed = TRUE) ||
        grepl("future\\.rng\\.onMisuse", msg) ||
        grepl("unexpectedly generated random numbers", msg, fixed = TRUE)) {
      invokeRestart("muffleWarning")
    }
  }

  )

  result.df <- do.call(rbind, lapply(out_list, `[[`, "df"))
  out <- list(
    replicate = result.df,
    n_valid = nrow(result.df),
    n_attempt = n_attempt,
    nsim_target = nsim,
    nsim_max = nsim.max,
    seed0 = seed.0,
    seeds = seeds,
    call = match.call()
  )
  class(out) <- "BayesPET_oc"
  out
}







