%% File Name: IRT.jackknife.Rd
%% File Version: 0.23

\name{IRT.jackknife}
\alias{IRT.jackknife}
\alias{IRT.jackknife.gdina}
\alias{coef.IRT.jackknife}
\alias{vcov.IRT.jackknife}
\alias{IRT.derivedParameters}

%- Also NEED an '\alias' for EACH other topic documented here.

\title{
Jackknifing an Item Response Model
}

\description{
This function performs a Jackknife procedure for estimating
standard errors for an item response model. The replication
design must be defined by \code{\link{IRT.repDesign}}.
Model fit is also assessed via Jackknife.

Statistical inference for derived parameters is performed
by \code{IRT.derivedParameters} with a fitted object of
class \code{IRT.jackknife} and a list with defining formulas.
}

\usage{
IRT.jackknife(object,repDesign, ... )

IRT.derivedParameters(jkobject, derived.parameters )

\method{IRT.jackknife}{gdina}(object, repDesign, \dots)

\method{coef}{IRT.jackknife}(object, bias.corr=FALSE, \dots)

\method{vcov}{IRT.jackknife}(object, \dots)
}

%- maybe also 'usage' for other objects documented here.
\arguments{
  \item{object}{
Objects for which S3 method \code{IRT.jackknife} is defined.
}
\item{repDesign}{Replication design generated by \code{\link{IRT.repDesign}}.}
\item{jkobject}{Object of class \code{IRT.jackknife}.}
\item{derived.parameters}{List with defined derived parameters
(see Example 2, Model 2).}
\item{bias.corr}{Optional logical indicating whether a bias correction
should be employed.}
\item{\dots}{Further arguments to be passed.}
}

%\details{
%%  ~~ If necessary, more details than the description above ~~
%}


\value{
List with following entries
\item{jpartable}{Parameter table with Jackknife estimates}
\item{parsM}{Matrix with replicated statistics}
\item{vcov}{Variance covariance matrix of parameters}
}

%\references{
%% ~put references to the literature/web site here ~
%}
%\author{
%%  ~~who you are~~
%}
%\note{
%%  ~~further notes~~
%}

%% ~Make other sections like Warning with \section{Warning }{....} ~

%\seealso{
%See also \code{\link{anova.din}} for model comparisons.
%A general method is defined in \code{\link{IRT.compareModels}}.
%}


\examples{
\dontrun{
library(BIFIEsurvey)

#############################################################################
# EXAMPLE 1: Multiple group DINA model with TIMSS data | Cluster sample
#############################################################################

data(data.timss11.G4.AUT.part, package="CDM")

dat <- data.timss11.G4.AUT.part$data
q.matrix <- data.timss11.G4.AUT.part$q.matrix2
# extract items
items <- paste(q.matrix$item)

# generate replicate design
rdes <- CDM::IRT.repDesign( data=dat,  wgt="TOTWGT", jktype="JK_TIMSS",
                   jkzone="JKCZONE", jkrep="JKCREP" )

#--- Model 1: fit multiple group GDINA model
mod1 <- CDM::gdina( dat[,items], q.matrix=q.matrix[,-1],
            weights=dat$TOTWGT, group=dat$female +1  )
# jackknife Model 1
jmod1 <- CDM::IRT.jackknife( object=mod1, repDesign=rdes )
summary(jmod1)
coef(jmod1)
vcov(jmod1)

#############################################################################
# EXAMPLE 2: DINA model | Simple random sampling
#############################################################################

data(sim.dina, package="CDM")
data(sim.qmatrix, package="CDM")
dat <- sim.dina
q.matrix <- sim.qmatrix

# generate replicate design with 50 jackknife zones (50 random groups)
rdes <- CDM::IRT.repDesign( data=dat,  jktype="JK_RANDOM", ngr=50 )

#--- Model 1: DINA model
mod1 <- CDM::gdina( dat, q.matrix=q.matrix, rule="DINA")
summary(mod1)
# jackknife DINA model
jmod1 <- CDM::IRT.jackknife( object=mod1, repDesign=rdes )
summary(jmod1)

#--- Model 2: DINO model
mod2 <- CDM::gdina( dat, q.matrix=q.matrix, rule="DINO")
summary(mod2)
# jackknife DINA model
jmod2 <- CDM::IRT.jackknife( object=mod2, repDesign=rdes )
summary(jmod2)
IRT.compareModels( mod1, mod2 )

# statistical inference for derived parameters
derived.parameters <- list( "skill1"=~ 0 + I(prob_skillV1_lev1_group1),
    "skilldiff12"=~ 0 + I( prob_skillV2_lev1_group1 - prob_skillV1_lev1_group1 ),
    "skilldiff13"=~ 0 + I( prob_skillV3_lev1_group1 - prob_skillV1_lev1_group1 )
                    )
jmod2a <- CDM::IRT.derivedParameters( jmod2, derived.parameters=derived.parameters )
summary(jmod2a)
coef(jmod2a)
}
}
% Add one or more standard keywords, see file 'KEYWORDS' in the
% R documentation directory.
%% \keyword{Jackknife}
% \keyword{ ~kwd2 }% __ONLY ONE__ keyword per line
