
<!-- README.md is generated from README.Rmd. Please edit that file -->

# ExactVaRTest

Fast exact finite-sample back-testing for Value-at-Risk (VaR) models in
R.

<!-- badges: start -->
<!-- badges: end -->

**ExactVaRTest** provides fast dynamic-programming algorithms in
C++/Rcpp (with pure R fallbacks) for the exact finite-sample
distributions and p-values of Christoffersen (1998) independence (IND)
and conditional-coverage (CC) VaR backtests. For completeness, it also
provides the exact unconditional-coverage (UC) test following Kupiec
(1995) via a closed-form binomial enumeration.

In particular, it corrects the severe size distortions from which the
usual asymptotic $\chi^2$ approximation suffers in small samples and
under extreme coverage rates.

- `backtest_lr()` returns the LR statistic, its exact p‑value, and a
  reject / fail‑to‑reject decision for one chosen test type (UC, IND, or
  CC).

- `backtest_all()` runs the UC, IND, and CC tests jointly and returns
  summaries for all three statistics.

## Installation

You can install the development version of ExactVaRTest from
[GitHub](https://github.com/) with:

``` r
# install.packages("pak")
pak::pak("YujianCHEN219/ExactVaRTest")
```

## Example

``` r
library(ExactVaRTest)

set.seed(42)
x <- rbinom(300, 1, 0.03)          # synthetic 0/1 exception series

bt <- backtest_lr(x, alpha = 0.05, type = "cc")  # exact LR_cc back-test
print(bt)
#> Exact finite-sample back-test
#> --------------------------------
#> Test           : Conditional coverage (LR_cc)
#> Sample size    : 300
#> Model alpha    : 0.0500
#> Signif. level  : 0.0500
#> LR statistic   : 5.8882
#> Exact p-value  : 0.0442
#> Decision       : REJECT null at 5.00% level
```

## Main features

- Exact finite‑sample distributions and p‑values for LR_ind and LR_cc at
  any sample size *n*; for *n* ≤ 2000 the computation finishes in
  milliseconds to a few seconds.

- C++ implementation via `Rcpp`, with automatic fallback to a pure‑R
  reference engine.

- Minimal dependencies (`Rcpp`, `stats`); works on macOS, Linux, and
  Windows.

## Extensions

Freely extends to *CoVaR backtesting*: pass the institution’s hit
sequence on system‑VaR‑breach days into `backtest_lr()` for exact UC/IND
p‑values; for short windows or extreme tails, one option is to treat the
systemic‑breach count as random and apply the mixture‑tail test to
maintain correct size. (see vignettes and \[Francq & Zakoïan 2025\]).

## Suggested readings

1.  Christoffersen, P. F. (1998). *Evaluating interval forecasts.*
    International economic review, 841-862.

2.  Mehta, C. R., Patel, N. R., & Gray, R. (1985). *Computing an exact
    confidence interval for the common odds ratio in several 2× 2
    contingency tables.* Journal of the American Statistical
    Association, 80(392), 969-973.

3.  Francq, C., & Zakoïan, J. M. (2025). Inference on dynamic systemic
    risk measures. Journal of Econometrics, 247, 105936.

## Acknowledgements

I greatly appreciate Christian Francq, Christophe Hurlin, and
Jean-Michel Zakoïan’s guidance and support.

In particular, Christian Francq generously shared the initial idea;
without his help, this package would not exist.

## License

This package is free and open source, licensed under GPL.
