#' Plot binomial confidence intervals
#'
#' Plot method for the S3 object class BinomCIframe generated by the
#' [BinomialCIsByCategorical()] function
#'
#' @param x an S3 object of class BinomCIframe
#' @param ... optional named parameters to be passed to [plot()]
#' @param CIrange two-element vector giving the minimum and maximum y-axis
#' values to plot (default NULL uses minimum lower confidence limit and
#' maximum upper confidence limit from `x`)
#' @param addRef logical: add a reference line at the average probability
#' of a positive response? (default = TRUE)
#'
#' @return None: this method generates a plot from `x`
#' @export
#'
#' @examples
#' catVar <- c(rep("A", 100), rep("B", 100), rep("C", 100))
#' binVar <- c(rep(0,80),rep(1,20), rep(0,50),rep(1,50), rep(0,20),rep(1,80))
#' DF <- data.frame(catVar = catVar, binVar = binVar)
#' CIframe <- BinomialCIsByCategorical(DF, "binVar", "catVar", 1)
#' plot(CIframe)
plot.BinomCIframe <- function(x, ..., CIrange = NULL, addRef = TRUE){
  #
  pEst <- x$pEst
  loCI <- x$loCI
  upCI <- x$upCI
  #
  if (is.null(CIrange)){
    CIrange <- c(min(loCI), max(upCI))
  }
  #
  plot(pEst, pch = 16, ylim = CIrange, ...)
  graphics::points(loCI, pch = 2)
  graphics::points(upCI, pch = 6)
  #
  if (addRef){
    n <- sum(x$nTotal)
    w <- sum(x$nWith)
    pRef <- w/n
    graphics::abline(h = pRef, lty = 2, lwd = 2)
  }
}
