#' @title Stress Susceptibility Percentage Index (SSPI)
#' @description
#' This function computes Stress Susceptibility Percentage Index (SSPI) using any traits (like yield) under stress and non-stress conditions. The lower values of SSPI indicates greater tolerance. For more details see Moosavi et al. (2008).
#'
#' @param Gen Character vector of genotype identifiers.
#' @param YN Numeric vector: yield (any trait) under non-stress (normal) environment.
#' @param YS Numeric vector: yield (any trait) under stress environment.
#' @param YMN Optional numeric scalar: mean of \code{YN}. If NULL, computed.
#' @return A list with \code{YMN} and \code{Result} (data frame with \code{Gen}, \code{SSPI}).
#'
#' \deqn{SSPI = \frac{(YN - YS)}{2 \times YMN} \times 100}
#'
#' @references Mousavi, S.S., YAZDI, S.B., Naghavi, M.R., Zali, A.A., Dashti, H. and Pourshahbazi, A. (2008). Introduction of new indices to identify relative drought tolerance and resistance in wheat genotypes. Desert 12, 165–178.
#' @examples
#' out=SSPI(Gen=c("G1","G2","G3"), YN=c(10,8,5), YS=c(7,5,3))
#' print(out)
#' @export
SSPI <- function(Gen, YN, YS, YMN = NULL) {
  .check_lengths(Gen, YN, YS)
  YN <- as.numeric(YN); YS <- as.numeric(YS)
  if (is.null(YMN)) YMN <- mean(YN, na.rm = TRUE)
  denom <- 2 * YMN
  out <- ((YN - YS) / denom) * 100
  out[denom == 0] <- NA_real_
  Result <- data.frame(Gen = as.character(Gen), SSPI = out, row.names = NULL)
  return(list(YMN = YMN, Result = Result))
}
