% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/qq_regression.R
\name{qq_regression}
\alias{qq_regression}
\title{Quantile-on-Quantile Regression}
\usage{
qq_regression(
  y,
  x,
  y_quantiles = seq(0.05, 0.95, by = 0.05),
  x_quantiles = seq(0.05, 0.95, by = 0.05),
  min_obs = 10,
  se_method = "boot",
  verbose = TRUE
)
}
\arguments{
\item{y}{Numeric vector. The dependent variable.}

\item{x}{Numeric vector. The independent variable.}

\item{y_quantiles}{Numeric vector. Quantiles of the dependent variable to
estimate. Default is \code{seq(0.05, 0.95, by = 0.05)}.}

\item{x_quantiles}{Numeric vector. Quantiles of the independent variable
to condition on. Default is \code{seq(0.05, 0.95, by = 0.05)}.}

\item{min_obs}{Integer. Minimum number of observations required for
quantile regression. Default is 10.}

\item{se_method}{Character string. Method for computing standard errors
in quantile regression. One of "boot" (bootstrap), "nid" (no i.i.d.
assumption), "iid", or "ker" (kernel). Default is "boot".}

\item{verbose}{Logical. If \code{TRUE}, prints progress messages.
Default is \code{TRUE}.}
}
\value{
An object of class "qq_regression" containing:
\itemize{
  \item \code{results} - Data frame with columns: y_quantile, x_quantile,
    coefficient, std_error, t_value, p_value, r_squared
  \item \code{y_quantiles} - Quantiles of y used
  \item \code{x_quantiles} - Quantiles of x used
  \item \code{n_obs} - Number of complete observations
  \item \code{call} - The matched call
  \item \code{method} - Method used ("quantile_regression" or "correlation")
}
}
\description{
Performs Quantile-on-Quantile (QQ) regression analysis as described in
Sim and Zhou (2015). This approach estimates the effect that quantiles of
an independent variable have on quantiles of a dependent variable, capturing
the dependence between their distributions.
}
\details{
The QQ approach combines quantile regression with local linear regression
to estimate the effect of the \eqn{\tau}{tau}-quantile of x on the 
\eqn{\theta}{theta}-quantile of y.

The method proceeds as follows:
\enumerate{
  \item For each x_quantile (\eqn{\tau}{tau}), subset the data where 
    x <= quantile(x, \eqn{\tau}{tau})
  \item For each y_quantile (\eqn{\theta}{theta}), perform quantile 
    regression of y on x
  \item Extract coefficients, standard errors, t-values, p-values
  \item Compute pseudo R-squared based on check function
}

If the subset has fewer than \code{min_obs} observations, a correlation-based
approach is used instead.
}
\examples{
# Generate example data
set.seed(42)
n <- 200
x <- rnorm(n)
y <- 0.5 * x + rnorm(n, sd = 0.5)

# Run QQ regression with default quantiles
result <- qq_regression(y, x, verbose = FALSE)

# Print summary
print(result)

# View first few results
head(result$results)

\donttest{
# Run with custom quantiles
result2 <- qq_regression(y, x,
                         y_quantiles = seq(0.1, 0.9, by = 0.1),
                         x_quantiles = seq(0.1, 0.9, by = 0.1),
                         verbose = FALSE)
}

}
\references{
Sim, N. and Zhou, H. (2015). Oil Prices, US Stock Return, and the
Dependence Between Their Quantiles. \emph{Journal of Banking & Finance},
55, 1-12. \doi{10.1016/j.jbankfin.2015.01.013}
}
