#' Dereplicate sequences
#'
#' @description \code{vs_fastx_uniques} performs dereplication of sequences in a
#' FASTA/FASTQ file or object by merging identical sequences using
#' \code{VSEARCH}.
#'
#' @param fastx_input (Required). A FASTA/FASTQ file path or FASTA/FASTQ object.
#' See \emph{Details}.
#' @param output_format (Optional). Desired output format of file or tibble:
#' \code{"fasta"} (default) or \code{"fastq"}. If \code{fastx_input} is a FASTA
#' file path or a FASTA object, \code{output_format} cannot be \code{"fastq"}.
#' @param fastx_output (Optional). Name of the output file for dereplicated
#' reads from \code{fastx_input}. File can be in either FASTA or FASTQ format,
#' depending on \code{output_format}. If \code{NULL} (default), no sequences are
#' written to file. See \emph{Details}.
#' @param minuniquesize (Optional). Minimum abundance value post-dereplication
#' for a sequence not to be discarded. Defaults to \code{1}.
#' @param strand (Optional). Specifies which strand to consider when comparing
#' sequences. Can be either \code{"plus"} (default) or \code{"both"}.
#' @param sizein (Optional). If \code{TRUE} (default), abundance annotations
#' present in sequence headers are taken into account.
#' @param sizeout (Optional). If \code{TRUE} (default), abundance annotations
#' are added to FASTA headers.
#' @param relabel (Optional). Relabel sequences using the given prefix and a
#' ticker to construct new headers. Defaults to \code{NULL}.
#' @param relabel_sha1 (Optional). If \code{TRUE} (default), relabel sequences
#' using the SHA1 message digest algorithm. Defaults to \code{FALSE}.
#' @param fastq_qout_max (Optional). If \code{TRUE}, the quality score will be
#' the highest (best) quality score observed in each position. Defaults to
#' \code{FALSE}.
#' @param fasta_width (Optional). Number of characters per line in the output
#' FASTA file. Defaults to \code{0}, which eliminates wrapping.
#' @param sample (Optional). Add the given sample identifier string to sequence
#' headers. For nstance, if the given string is "ABC", the text ";sample=ABC"
#' will be added to the header. If \code{NULL} (default), no identifier is added.
#' @param vsearch_options (Optional). A character string of additional arguments
#' to pass to \code{VSEARCH}. Defaults to \code{NULL}. See \emph{Details}.
#' @param tmpdir (Optional). Path to the directory where temporary files should
#' be written when tables are used as input or output. Defaults to
#' \code{NULL}, which resolves to the session-specific temporary directory
#' (\code{tempdir()}).
#'
#' @details
#' Sequences in the input file/object (\code{fastx_input}) are dereplicated by
#' merging identical sequences. Identical sequences are defined as sequences
#' with the same length and the same string of nucleotides (case insensitive, T
#' and U are considered the same).
#'
#' \code{fastx_input} can either be a FASTA/FASTQ file or a FASTA/FASTQ object.
#' FASTA objects are tibbles that contain the columns \code{Header} and
#' \code{Sequence}, see \code{\link[microseq]{readFasta}}. FASTQ objects are
#' tibbles that contain the columns \code{Header}, \code{Sequence}, and
#' \code{Quality}, see \code{\link[microseq]{readFastq}}.
#'
#' By default, the quality scores in FASTQ output files will correspond to the
#' average error probability of the nucleotides in the each position.
#' If \code{fastq_qout_max = TRUE}, the quality score will be the highest (best)
#' quality score observed in each position.
#'
#' If \code{fastx_output} is specified, the dereplicated sequences are output to
#' this file in format given by \code{output_format}.
#' If \code{fastx_output} is \code{NULL}, the dereplicated  sequences are
#' returned as a FASTA or FASTQ object, depending on \code{output_format}.
#'
#' \code{vsearch_options} allows users to pass additional command-line arguments
#' to \code{VSEARCH} that are not directly supported by this function. Refer to
#' the \code{VSEARCH} manual for more details.
#'
#' @return A tibble or \code{NULL}.
#'
#' If \code{fastx_output} is specified, the dereplicated sequences are written
#' to the specified output file, and no tibble is returned.
#'
#' If \code{fastx_output} \code{NULL}, a tibble containing the dereplicated
#' reads in the format specified by \code{output_format} is returned.
#'
#' @examples
#' \dontrun{
#' # Define arguments
#' fastx_input <- file.path(file.path(path.package("Rsearch"), "extdata"),
#'                          "small_R1.fq")
#' fastx_output <- NULL
#' output_format <- "fastq"
#'
#' # Dereplicate sequences and return a FASTQ tibble
#' derep_R1 <- vs_fastx_uniques(fastx_input = fastx_input,
#'                              fastx_output = fastx_output,
#'                              output_format = output_format)
#'
#' # Dereplicate sequences and write derelicated sequences to a file
#' vs_fastx_uniques(fastx_input = fastx_input,
#'                  fastx_output = "dereplicated_sequences.fq",
#'                  output_format = output_format)
#' }
#'
#' @references \url{https://github.com/torognes/vsearch}
#'
#' @aliases vs_fastx_uniques vs_fastq_uniques vs_fasta_uniques vs_fastx_dereplicate
#' vs_fastq_dereplicate vs_fasta_dereplicate fastx_uniques dereplicate
#'
#' @export
#'
vs_fastx_uniques <- function(fastx_input,
                             output_format = "fasta",
                             fastx_output = NULL,
                             minuniquesize = 1,
                             strand = "plus",
                             sizein = TRUE,
                             sizeout = TRUE,
                             relabel = NULL,
                             relabel_sha1 = FALSE,
                             fastq_qout_max = FALSE,
                             fasta_width = 0,
                             sample = NULL,
                             vsearch_options = NULL,
                             tmpdir = NULL) {

  # Check if vsearch is available
  vsearch_executable <- options("Rsearch.vsearch_executable")[[1]]
  vsearch_available(vsearch_executable)

  # Set temporary directory if not provided
  if (is.null(tmpdir)) tmpdir <- tempdir()

  # Validate output_format
  if (!output_format %in% c("fasta", "fastq")) {
    stop("Invalid output_format. Choose from fasta or fastq.")
  }

  # Validate strand
  if (!strand %in% c("plus", "both")) {
    stop("Invalid value for 'strand'. Choose from 'plus' or 'both'.")
  }

  # Create empty vector for collecting temporary files
  temp_files <- character()

  # Set up cleanup of temporary files
  on.exit({
    existing_files <- temp_files[file.exists(temp_files)]
    if (length(existing_files) > 0) {
      file.remove(existing_files)
    }
  }, add = TRUE)

  # Handle input
  if (!is.character(fastx_input)){
    if ("Quality" %in% colnames(fastx_input)){

      # Validate tibble
      required_cols <- c("Header", "Sequence", "Quality")
      if (!all(required_cols %in% colnames(fastx_input))) {
        stop("FASTQ object must contain columns: Header, Sequence, Quality")
      }

      temp_file <- tempfile(pattern = "input",
                            tmpdir = tmpdir,
                            fileext = ".fq")
      temp_files <- c(temp_files, temp_file)
      microseq::writeFastq(fastx_input, temp_file)
      input_file <- temp_file
    } else {

      if (output_format == "fastq") {
        stop("Invalid output_format when input tibble is of type 'fasta'")
      }

      required_cols <- c("Header", "Sequence")
      if (!all(required_cols %in% colnames(fastx_input))) {
        stop("FASTA object must contain columns: Header and Sequence")
      }

      temp_file <- tempfile(pattern = "input",
                            tmpdir = tmpdir,
                            fileext = ".fa")
      temp_files <- c(temp_files, temp_file)
      microseq::writeFasta(fastx_input, temp_file)
      input_file <- temp_file
    }
  } else {
    input_file <- fastx_input
  }

  # Handle output_format = "fasta"
  if (output_format == "fasta"){
    if (is.null(fastx_output)) {
      output_file <- tempfile(pattern = "derep",
                              tmpdir = tmpdir,
                              fileext = ".fa")
      temp_files <- c(temp_files, output_file)
    } else {
      output_file <- fastx_output
    }
  }

  # Handle output_format = "fastq"
  if (output_format == "fastq"){
    if (is.null(fastx_output)) {
      output_file <- tempfile(pattern = "derep",
                              tmpdir = tmpdir,
                              fileext = ".fq")
      temp_files <- c(temp_files, output_file)
    } else {
      output_file <- fastx_output
    }
  }

  # Check is input file exists at given path
  if (!file.exists(input_file)) stop("Cannot find input file: ", input_file)

  # Normalize file paths
  input_file <- normalizePath(input_file)

  # Build argument string for command line
  args <- c("--fastx_uniques", shQuote(input_file),
            "--threads", 1,
            "--minuniquesize", minuniquesize,
            "--strand", strand)

  if (output_format == "fasta") {
    args <- c(args,
              "--fasta_width", fasta_width,
              "--fastaout", output_file)
  }

  if (output_format == "fastq") {
    args <- c(args, "--fastqout", output_file)
  }

  if (sizein) {
    args <- c(args, "--sizein", "")
  }

  if (sizeout) {
    args <- c(args, "--sizeout", "")
  }

  if (relabel_sha1) {
    args <- c(args, "--relabel_sha1", "")
  }

  if (!is.null(relabel)) {
    args <- c(args, "--relabel", relabel)
  }

  if (fastq_qout_max) {
    args <- c(args, "--fastq_qout_max", "")
  }

  # Add sample identifier if specified
  if (!is.null(sample)) {
    args <- c(args, "--sample", sample)
  }

  # Add additional arguments if specified
  if (!is.null(vsearch_options)) {
    args <- c(args, vsearch_options)
  }

  # Run VSEARCH
  vsearch_output <- system2(command = vsearch_executable,
                            args = args,
                            stdout = TRUE,
                            stderr = TRUE)

  # Check for VSEARCH failure
  check_vsearch_status(vsearch_output, args)

  if (is.null(fastx_output)) {
    if (output_format == "fasta"){
      # Read output into FASTA object
      derep_tbl <- microseq::readFasta(output_file)
    }
    if (output_format == "fastq"){
      # Read output into FASTQ object
      derep_tbl <- microseq::readFastq(output_file)
    }
  }

  # Return results
  if (is.null(fastx_output)) { # Return tibble
    return(derep_tbl)
  } else {
    return(invisible(NULL)) # No return when output file is written
  }
}
