% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/cor_test.R
\name{z_cor_test}
\alias{z_cor_test}
\title{Test for Association/Correlation Between Paired Samples}
\usage{
z_cor_test(
  x,
  y,
  alternative = c("two.sided", "less", "greater", "equivalence", "minimal.effect"),
  method = c("pearson", "kendall", "spearman"),
  alpha = 0.05,
  null = 0
)
}
\arguments{
\item{x, y}{numeric vectors of data values. x and y must have the same length.}

\item{alternative}{a character string specifying the alternative hypothesis:
\itemize{
\item "two.sided": correlation is not equal to null (default)
\item "greater": correlation is greater than null
\item "less": correlation is less than null
\item "equivalence": correlation is within the equivalence bounds (TOST)
\item "minimal.effect": correlation is outside the equivalence bounds (TOST)
}

You can specify just the initial letter.}

\item{method}{a character string indicating which correlation coefficient is to be used for the test.
One of "pearson", "kendall", or "spearman", can be abbreviated.}

\item{alpha}{alpha level (default = 0.05)}

\item{null}{a number or vector indicating the null hypothesis value(s):
\itemize{
\item For standard tests: a single value (default = 0)
\item For equivalence/minimal effect tests: either a single value (symmetric bounds ±value will be created)
or a vector of two values representing the lower and upper bounds
}}
}
\value{
A list with class "htest" containing the following components:
\itemize{
\item \strong{p.value}: the p-value of the test.
\item \strong{statistic}: the value of the test statistic with a name describing it.
\item \strong{parameter}: the degrees of freedom or number of observations.
\item \strong{conf.int}: a confidence interval for the measure of association appropriate to the specified alternative hypothesis.
\item \strong{estimate}: the estimated measure of association, with name "cor", "tau", or "rho" corresponding to the method employed.
\item \strong{stderr}: the standard error of the test statistic.
\item \strong{null.value}: the value of the association measure under the null hypothesis.
\item \strong{alternative}: character string indicating the alternative hypothesis.
\item \strong{method}: a character string indicating how the association was measured.
\item \strong{data.name}: a character string giving the names of the data.
\item \strong{call}: the matched call.
}
}
\description{
\ifelse{html}{\href{https://lifecycle.r-lib.org/articles/stages.html#stable}{\figure{lifecycle-stable.svg}{options: alt='[Stable]'}}}{\strong{[Stable]}}

Test for association between paired samples, using one of Pearson's product moment correlation
coefficient, Kendall's \eqn{\tau} (tau) or Spearman's \eqn{\rho} (rho). Unlike the stats version of
cor.test, this function allows users to set the null to a value other than zero and perform
equivalence testing.
}
\details{
This function uses Fisher's z transformation for the correlations, but uses Fieller's
correction of the standard error for Spearman's \eqn{\rho} and Kendall's \eqn{\tau}.

The function supports both standard hypothesis testing and equivalence/minimal effect testing:
\itemize{
\item For standard tests (two.sided, less, greater), the function tests whether the correlation
differs from the null value (typically 0).
\item For equivalence testing ("equivalence"), it determines whether the correlation falls within
the specified bounds, which can be set asymmetrically.
\item For minimal effect testing ("minimal.effect"), it determines whether the correlation falls
outside the specified bounds.
}

When performing equivalence or minimal effect testing:
\itemize{
\item If a single value is provided for \code{null}, symmetric bounds ± value will be used
\item If two values are provided for \code{null}, they will be used as the lower and upper bounds
}

See \code{vignette("correlations")} for more details.
}
\examples{
# Example 1: Standard significance test
x <- c(44.4, 45.9, 41.9, 53.3, 44.7, 44.1, 50.7, 45.2, 60.1)
y <- c( 2.6,  3.1,  2.5,  5.0,  3.6,  4.0,  5.2,  2.8,  3.8)
z_cor_test(x, y, method = "kendall", alternative = "t", null = 0)

# Example 2: Minimal effect test
# Testing if correlation is meaningfully different from ±0.2
z_cor_test(x, y, method = "kendall", alternative = "min", null = 0.2)

# Example 3: Equivalence test with Pearson correlation
# Testing if correlation is equivalent to zero within ±0.3
z_cor_test(x, y, method = "pearson", alternative = "equivalence", null = 0.3)

# Example 4: Using asymmetric bounds
# Testing if correlation is within bounds of -0.1 and 0.4
z_cor_test(x, y, method = "spearman",
           alternative = "equivalence", null = c(-0.1, 0.4))

}
\references{
Goertzen, J. R., & Cribbie, R. A. (2010). Detecting a lack of association: An equivalence
testing approach. British Journal of Mathematical and Statistical Psychology, 63(3), 527-537.
https://doi.org/10.1348/000711009X475853, formula page 531.
}
\seealso{
Other Correlations: 
\code{\link{boot_cor_test}()},
\code{\link{corsum_test}()},
\code{\link{plot_cor}()},
\code{\link{power_z_cor}()}
}
\concept{Correlations}
