#' @title Pessimistic claims game associated with a claims problem
#' @description Given a claims problem, this function returns the associated pessimistic claims game.
#' @param E An endowment, as a positive number.
#' @param d A vector of claims.
#' @param binary A logical value. By default, \code{binary=FALSE}.
#' @return The characteristic function of the pessimistic claims game, as a vector in binary order if \code{binary=TRUE} and in lexicographic order otherwise.
#' @details A claims problem is a triple \eqn{(N,E,d)} where \eqn{E\ge 0} is an amount to be distributed among a set \eqn{N} of agents and \eqn{d\in \mathbb{R}^{|N|}} is a vector of claims satisfying \eqn{\sum_{i=1}^{n} d_i\ge E}.
#'
#' Given a claims problem \eqn{(N,E,d)}, its associated claims game, \eqn{v_{E,d}\in G^N}, is defined by
#' \deqn{v_{E,d}(S)=\max\{0, \; E-\sum_{i\in N\backslash S}d_i\}\text{ for all }S\in 2^N.}
#'
#' For further analysis and computational methods related to conflicting claims problems,
#' see the \pkg{ClaimsProblems} package \cite{Sánchez Rodríguez et al. (2025)}.
#' @seealso \link{airfieldgame}
#' @examples
#' E <- 10
#' d <- c(2,4,7,8)
#' claimsgame(E,d)
#' @references O’Neill, B. (1982). A problem of rights arbitration from the Talmud. \emph{Mathematical Social Sciences}, 2, 345–371.
#' @references Sánchez Rodríguez, E., Núñez Lugilde, I., Mirás Calvo, M., & Quinteiro Sandomingo, C. (2025). \emph{ClaimsProblems: Analysis of Conflicting Claims}. R package version 1.0.0.
#'
#'             \url{https://CRAN.R-project.org/package=ClaimsProblems}
#' @export

claimsgame <- function(E, d, binary = FALSE) {

  ################################
  ###Comprobación datos entrada###
  ################################

  n = length(d)
  D = sum(d) #The number of claims and the total claim
  # Required: (E,d) must be a claims problem, i.e., E >=0, d >=0, E <= sum(d)
  if (E < 0 || sum((d < 0)) > 0 || E > D) {
    stop("'E' and 'd' cannot be negative; 'E' cannot be greater than the sum of the elements of 'd'.")
  }

  ################################
  #####Creación de variables######
  ################################

  nC=2^n-1 #Número de coaliciones
  v=rep(0, nC) #Claims game

  ################################
  ######Cuerpo de la función######
  ################################

  if (E > 0) #Si E=0, devuelve el juego nulo.
  {
    for(S in 1:((nC-1)/2)) #Por dualidad, recorremos solo la mitad de las coaliciones.
    {
      #Calculamos los jugadores pertenecientes a cada coalición y su valor en v.
      num=S
      J=floor(log2(num))+1 #Jugador más grande
      Sjug=c()
      for (ii in 1:J)#Algoritmo de paso de un número en decimal a binario.
      {
        if (num %% 2 ==1)
        {
          Sjug=c(Sjug,ii)
        }
        num=num %/% 2
      }

      v[S]=E-D+sum(d[Sjug]) #v(S)=E-d(N\S)=E-d(N)+d(S).
      v[nC-S]=E-sum(d[Sjug]) #v(N\S)=E-d(S).
      v[S]=max(0,v[S]) #v(S)=max(0,E-d(N\S)).
      v[nC-S]=max(0,v[nC-S]) #v(N\S)=max(0,E-d(S)).
    }
  }
  v[nC] = E #v(N)=E.

  ################################
  ######Salidas de la función#####
  ################################

  if (binary == FALSE) { # Devolvemos el juego en orden lexicográfico.
    return(bin2lex(v))
  } else { # Devolvemos el juego en orden binario.
    return(v)
  }
}# Fin de la función
