% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/plot.effect_curve.R
\name{plot.effect_curve}
\alias{plot.effect_curve}
\title{Plot an effect curve}
\usage{
\method{plot}{effect_curve}(
  x,
  conf_level = 0.95,
  simultaneous = TRUE,
  null = NULL,
  subset = NULL,
  proj = NULL,
  transform = TRUE,
  ci.type = "perc",
  df = NULL,
  ...
)
}
\arguments{
\item{x}{an \code{effect_curve} object; the output of a call to \code{\link[=adrf]{adrf()}} or functions that modify it.}

\item{conf_level}{the desired confidence level. Default is .95 for 95\% confidence bands.}

\item{simultaneous}{\code{logical}; whether the computed confidence bands should be simultaneous (\code{TRUE}) or pointwise (\code{FALSE}). Simultaneous (also known as uniform) bands cover the full line at the desired confidence level, whereas pointwise confidence bands only cover each point at the desired level. Default is \code{TRUE}. See Details.}

\item{null}{the value at which to plot a horizontal reference line. Default is to plot a line with a y-intercept of 0 when the effect curve is an AMEF, a curve contrast, or a reference effect curve, and to omit a line otherwise. Set to \code{NA} to manually omit the line.}

\item{subset}{an optional logical expression indicating the subset of the subgroups to plot. Can only be used when \code{by} was supplied to the original call to \code{\link[=adrf]{adrf()}}, and only to refer to variables defining subgroups.}

\item{proj}{an optional \code{curve_projection} object, the output of a call to \code{\link[=curve_projection]{curve_projection()}}. Supplying this adds the projection curve to the effect curve plot.}

\item{transform}{whether to compute intervals on the transformed estimates. Allowable options include \code{TRUE}, \code{FALSE}, or a function specifying a transformation. Ignored unless \code{x} is an ADRF. See Details.}

\item{ci.type}{string; when bootstrapping is used in the original effect curve, what type of confidence interval is to be computed. Allowable options include \code{"perc"} for percentile intervals, \code{"wald"} for Wald intervals, and other options allowed by \pkgfun{fwb}{summary.fwb}. When \code{simultaneous = TRUE}, only \code{"perc"} and \code{"wald"} are allowed. Default is \code{"perc"}. Ignored when bootstrapping is not used.}

\item{df}{the "denominator" degrees of freedom to use for the critical test statistics for confidence bands. Default is to use the residual degrees of freedom from the original model if it is a linear model and \code{Inf} otherwise.}

\item{...}{ignored.}
}
\value{
A \code{ggplot} object that can modified using functions in \pkg{ggplot2}. Below are some common tasks and instructions to perform them. Note all should be run after running \code{library("ggplot2")}.

\itemize{
\item{Change the position of the legend:
\preformatted{theme(legend.position = "{POSITION}")}
}
\item{Remove the legend:
\preformatted{theme(legend.position = "none")}
}
\item{Change the color of the plotted line:
\preformatted{theme(geom = element_geom(ink = "{COLOR}"))}
}
\item{Change the color scheme of the plotted lines:
\preformatted{scale_color_brewer(aesthetics = c("color", "fill"),
                   palette = "{PALETTE}")}
}
\item{Change the title, subtitle, or axis labels:
\preformatted{labs(title = "{TITLE}", subtitle = "{SUBTITLE}",
     x = "{X-AXIS}", y = "{Y-AXIS}")}
}
\item{Change the y-axis range:
\preformatted{coord_cartesian(ylim = c({LOWER}, {UPPER}),
                expand = FALSE)}
}
}

Values in brackets are to be changed by the user. Refer to the \CRANpkg{ggplot2} documentation for other options.
}
\description{
\code{plot()} plots an effect curve and its confidence band.
}
\details{
\code{plot()} displays the effect curve in a plot. The solid line corresponds to the effect curve and the ribbon around it corresponds to its confidence band. When \code{null} is not \code{NA}, an additional flat line at \code{null} is displayed. When \code{proj} is supplied, a dashed line corresponding to the projection is added.

When \code{by} is supplied to \code{\link[=adrf]{adrf()}}, \code{plot()} produces an effect curve plot for each subgroup. When \code{x} is the output of a call to \code{\link[=curve_contrast]{curve_contrast()}}, \code{plot()} produces an effect curve plot for each treatment contrast.
\subsection{Transform}{

The usual confidence bands assume the estimates along the effect curve are normally distributed (or t-distributed when \code{df} is not \code{Inf}). However, when the outcome is bounded (e.g., a probability bounded between 0 and 1), this assumption may not be valid for the ADRF in finite samples. \code{transform} transforms the estimates to ones that are unbounded and computes the corresponding distribution of transformed estimates using the delta method. By default, if a generalized linear model is used for the outcome with a non-identity link function, the estimates are transformed by the link function to be on an unbounded scale. Note this is not the same as using the linear predictor for the effect curve; this is simple a transformation of the estimated points along the curve already computed. Confidence bands are computed using the transformed estimates before being back-transformed to ensure they are within the bounds of the outcome.
}

\subsection{Simultaneous confidence bands}{

Simultaneous confidence bands ensure the whole effect curve, not just a given individual point, is contained within the band at the given confidence level. These are wider than pointwise bands to reflect that they are covering multiple estimates, which otherwise would decrease the true coverage rate from that specified. \code{plot()} uses the "sup-t" simultaneous confidence band, which is the smallest one-parameter band that covers the whole effect curve at the desired rate.
}
}
\examples{
data("nhanes3lead")

fit <- glm(Block >= 12 ~ poly(logBLL, 3) *
             Male * (Age + Race + PIR + NICU +
                       Smoke_Pregnant),
           data = nhanes3lead,
           family = binomial)

# ADRF of logBLL on P(Block >= 12)
adrf1 <- adrf(fit, treat = "logBLL",
              n = 50) #using 50 to speed up examples

# Plot the ADRF; simultaneous inference,
# CIs computed on transformed estimates
plot(adrf1)

# Plot the ADRF; simultaneous inference,
# CIs computed on original estimates
plot(adrf1, transform = FALSE)

# Plot the ADRF; pointwise inference
plot(adrf1, simultaneous = FALSE)

\donttest{
# ADRF within subgroups
adrf2 <- adrf(fit, treat = "logBLL",
              by = ~Male, n = 50)

# Plot subgroup ADRFs
plot(adrf2)

# Plot ADRF in one subgroup
plot(adrf2, subset = Male == 1)

# ADRF contrast
adrf_contrast <- curve_contrast(adrf2)

plot(adrf_contrast)
}
}
\seealso{
\itemize{
\item \code{\link[=adrf]{adrf()}} for computing the ADRF
\item \code{\link[=summary.effect_curve]{summary.effect_curve()}} for testing hypotheses about an effect curve
}
}
