% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/autoEnsemble.R
\name{autoEnsemble}
\alias{autoEnsemble}
\title{Automatically Trains H2O Models and Builds a Stacked Ensemble Model}
\usage{
autoEnsemble(
  x,
  y,
  training_frame,
  validation_frame = NULL,
  nfolds = 10,
  balance_classes = TRUE,
  max_runtime_secs = NULL,
  max_runtime_secs_per_model = NULL,
  max_models = NULL,
  sort_metric = "AUCPR",
  include_algos = c("GLM", "DeepLearning", "DRF", "XGBoost", "GBM"),
  save_models = FALSE,
  directory = paste("autoEnsemble", format(Sys.time(), "\%d-\%m-\%y-\%H:\%M")),
  ...,
  newdata = NULL,
  family = "binary",
  strategy = c("search"),
  model_selection_criteria = c("auc", "aucpr", "mcc", "f2"),
  min_improvement = 1e-05,
  max = NULL,
  top_rank = seq(0.01, 0.99, 0.01),
  stop_rounds = 3,
  reset_stop_rounds = TRUE,
  stop_metric = "auc",
  seed = -1,
  verbatim = FALSE,
  startH2O = FALSE,
  nthreads = NULL,
  max_mem_size = NULL,
  min_mem_size = NULL,
  ignore_config = FALSE,
  bind_to_localhost = FALSE,
  insecure = TRUE
)
}
\arguments{
\item{x}{Vector. Predictor column names or indices.}

\item{y}{Character. The response column name or index.}

\item{training_frame}{An H2OFrame containing the training data.
Default is \code{h2o.getFrame("hmda.train.hex")}.}

\item{validation_frame}{An H2OFrame for early stopping. Default is \code{NULL}.}

\item{nfolds}{Integer. Number of folds for cross-validation.
Default is 10.}

\item{balance_classes}{Logical. Specify whether to oversample the minority
classes to balance the class distribution; only applicable to classification}

\item{max_runtime_secs}{Integer. This argument specifies the maximum time that
the AutoML process will run for in seconds.}

\item{max_runtime_secs_per_model}{Maximum runtime in seconds dedicated to each
individual model training process.}

\item{max_models}{Maximum number of models to build in the AutoML training (passed to autoML)}

\item{sort_metric}{Metric to sort the leaderboard by (passed to autoML).
For binomial classification
choose between "AUC", "AUCPR", "logloss", "mean_per_class_error",
"RMSE", "MSE". For regression choose between "mean_residual_deviance",
"RMSE", "MSE", "MAE", and "RMSLE". For multinomial classification choose
between "mean_per_class_error", "logloss", "RMSE", "MSE". Default is
"AUTO". If set to "AUTO", then "AUC" will be used for binomial classification,
"mean_per_class_error" for multinomial classification, and
"mean_residual_deviance" for regression.}

\item{include_algos}{Vector of character strings naming the algorithms to
restrict to during the model-building phase. this argument
is passed to autoML.}

\item{save_models}{Logical. if TRUE, the models trained will be stored locally}

\item{directory}{path to a local directory to store the trained models}

\item{...}{parameters to be passed to autoML algorithm in h2o package}

\item{newdata}{h2o frame (data.frame). the data.frame must be already uploaded
on h2o server (cloud). when specified, this dataset will be used
for evaluating the models. if not specified, model performance
on the training dataset will be reported.}

\item{family}{model family. currently only \code{"binary"} classification models
are supported.}

\item{strategy}{character. the current available strategies are \code{"search"}
(default) and \code{"top"}. The \code{"search"} strategy searches
for the best combination of top-performing diverse models
whereas the \code{"top"} strategy is more simplified and just
combines the specified of top-performing diverse models without
examining the possibility of improving the model by searching for
larger number of models that can further improve the model. generally,
the \code{"search"} strategy is preferable, unless the computation
runtime is too large and optimization is not possible.}

\item{model_selection_criteria}{character, specifying the performance metrics that
should be taken into consideration for model selection. the default are
\code{"c('auc', 'aucpr', 'mcc', 'f2')"}. other possible criteria are
\code{"'f1point5', 'f3', 'f4', 'f5', 'kappa', 'mean_per_class_error', 'gini', 'accuracy'"},
which are also provided by the \code{"evaluate"} function.}

\item{min_improvement}{numeric. specifies the minimum improvement in model
evaluation metric to qualify further optimization search.}

\item{max}{integer. specifies maximum number of models for each criteria to be extracted. the
default value is the \code{"top_rank"} percentage for each model selection
criteria.}

\item{top_rank}{numeric vector. specifies percentage of the top models taht
should be selected. if the strategy is \code{"search"}, the
algorithm searches for the best best combination of the models
from top ranked models to the bottom. however, if the strategy
is \code{"top"}, only the first value of the vector is used
(default value is top 1\%).}

\item{stop_rounds}{integer. number of stoping rounds, in case the model stops
improving}

\item{reset_stop_rounds}{logical. if TRUE, everytime the model improves the
stopping rounds penalty is resets to 0.}

\item{stop_metric}{character. model stopping metric. the default is \code{"auc"},
but \code{"aucpr"} and \code{"mcc"} are also available.}

\item{seed}{random seed (recommended)}

\item{verbatim}{logical. if TRUE, it reports additional information about the
progress of the model training, particularly used for debugging.}

\item{startH2O}{Logical. if TRUE, h2o server will be initiated.}

\item{nthreads}{arguments to be passed to h2o.init()}

\item{max_mem_size}{arguments to be passed to h2o.init()}

\item{min_mem_size}{arguments to be passed to h2o.init()}

\item{ignore_config}{arguments to be passed to h2o.init()}

\item{bind_to_localhost}{arguments to be passed to h2o.init()}

\item{insecure}{arguments to be passed to h2o.init()}
}
\value{
a list including the ensemble model and the top-rank models that were
        used in the model
}
\description{
Automatically trains various algorithms to build base-learners and then
             automatically creates a stacked ensemble model
}
\examples{

\dontrun{
# load the required libraries for building the base-learners and the ensemble models
library(h2o)
library(autoEnsemble)

# initiate the h2o server
h2o.init(ignore_config = TRUE, nthreads = 2, bind_to_localhost = FALSE, insecure = TRUE)

# upload data to h2o cloud
prostate_path <- system.file("extdata", "prostate.csv", package = "h2o")
prostate <- h2o.importFile(path = prostate_path, header = TRUE)

### H2O provides 2 types of grid search for tuning the models, which are
### AutoML and Grid. Below, I tune 2 set of model grids and use them both
### for building the ensemble, just to set an example ...

#######################################################
### PREPARE AutoML Grid (takes a couple of minutes)
#######################################################
# run AutoML to tune various models (GLM, GBM, XGBoost, DRF, DeepLearning) for 120 seconds
y <- "CAPSULE"
prostate[,y] <- as.factor(prostate[,y])  #convert to factor for classification
aml <- h2o.automl(y = y, training_frame = prostate, max_runtime_secs = 120,
                 include_algos=c("DRF","GLM", "XGBoost", "GBM", "DeepLearning"),

                 # this setting ensures the models are comparable for building a meta learner
                 seed = 2023, nfolds = 10,
                 keep_cross_validation_predictions = TRUE)

#######################################################
### PREPARE H2O Grid (takes a couple of minutes)
#######################################################
# make sure equal number of "nfolds" is specified for different grids
grid <- h2o.grid(algorithm = "gbm", y = y, training_frame = prostate,
                 hyper_params = list(ntrees = seq(1,50,1)),
                 grid_id = "ensemble_grid",

                 # this setting ensures the models are comparable for building a meta learner
                 seed = 2023, fold_assignment = "Modulo", nfolds = 10,
                 keep_cross_validation_predictions = TRUE)

#######################################################
### PREPARE ENSEMBLE MODEL
#######################################################

### get the models' IDs from the AutoML and grid searches.
### this is all that is needed before building the ensemble,
### i.e., to specify the model IDs that should be evaluated.

ids    <- c(h2o.get_ids(aml), h2o.get_ids(grid))
top    <- ensemble(models = ids, training_frame = prostate, strategy = "top")
search <- ensemble(models = ids, training_frame = prostate, strategy = "search")

#######################################################
### EVALUATE THE MODELS
#######################################################
h2o.auc(aml@leader)                          # best model identified by h2o.automl
h2o.auc(h2o.getModel(grid@model_ids[[1]]))   # best model identified by grid search
h2o.auc(top$model).                          # ensemble model with 'top' search strategy
h2o.auc(search$model).                       # ensemble model with 'search' search strategy

}
}
\author{
E. F. Haghish
}
