% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/scapply.R
\name{scapply}
\alias{scapply}
\title{Single-cell apply a function to a matrix split by a factor}
\usage{
scapply(
  x,
  INDEX,
  FUN,
  combine = NULL,
  combine2 = "c",
  progress = TRUE,
  sliceMem = 16,
  cores = 1L,
  ...
)
}
\arguments{
\item{x}{matrix, sparse matrix or DelayedMatrix of raw counts with genes in
rows and cells in columns.}

\item{INDEX}{a factor whose length matches the number of columns in \code{x}. It
is coerced to a factor. \code{NA} are tolerated and the matching columns in \code{x}
are skipped.}

\item{FUN}{Function to be applied to each subblock of the matrix.}

\item{combine}{A function or a name of a function to apply to the list
output to bind the final results together, e.g. 'cbind' or 'rbind' to
return a matrix, or 'unlist' to return a vector.}

\item{combine2}{A function or a name of a function to combine results after
slicing. As the function is usually applied to blocks of 30000 genes or so,
the result is usually a vector with an element per gene. Hence 'c' is the
default function for combining vectors into a single longer vector. However
if each gene returns a number of results (e.g. a vector or dataframe), then
\code{combine2} could be set to 'rbind'.}

\item{progress}{Logical, whether to show progress.}

\item{sliceMem}{Max amount of memory in GB to allow for each subsetted count
matrix object. When \code{x} is subsetted by each cell subclass, if the amount
of memory would be above \code{sliceMem} then slicing is activated and the
subsetted count matrix is divided into chunks and processed separately.
The limit is just under 17.2 GB (2^34 / 1e9). At this level the subsetted
matrix breaches the long vector limit (>2^31 elements).}

\item{cores}{Integer, number of cores to use for parallelisation using
\code{mclapply()}. Parallelisation is not available on windows. Warning:
parallelisation increases the memory requirement by multiples of
\code{sliceMem}.}

\item{...}{Optional arguments passed to \code{FUN}.}
}
\value{
By default returns a list, unless \code{combine} is invoked in which case
the returned data type will depend on the functions specified by \code{FUN} and
\code{combine}.
}
\description{
Workhorse function designed to handle large scRNA-Seq gene expression
matrices such as embedded Seurat matrices, and apply a function to columns of
the matrix split as a ragged array by an index factor, similar to \code{\link[=tapply]{tapply()}},
\code{\link[=by]{by()}} or \code{\link[=aggregate]{aggregate()}}. Note that here the index is applied to columns as
these represent cells in the single-cell format, rather than rows as in
\code{\link[=aggregate]{aggregate()}}. Very large matrices are handled by slicing rows into blocks to
avoid excess memory requirements.
}
\details{
The limit on \code{sliceMem} is that the number of elements manipulated in each
block must be
kept below the long vector limit of 2^31 (around 2e9). Increasing \code{cores}
requires substantial amounts of spare RAM. \code{combine} works
in a similar way to \code{.combine} in \code{foreach()}; it works across the levels in
\code{INDEX}. \code{combine2} is nested and works across slices of genes (an inner
loop), so it is only invoked if slicing occurs which is when a matrix has a
larger memory footprint than \code{sliceMem}.
}
\examples{
# equivalent
m <- matrix(sample(0:100, 1000, replace = TRUE), nrow = 10)
cell_index <- sample(letters[1:5], 100, replace = TRUE)
o <- scmean(m, cell_index)
o2 <- scapply(m, cell_index, function(x) rowMeans(log2(x +1)),
              combine = "cbind")
identical(o, o2)

}
\seealso{
\code{\link[=scmean]{scmean()}} which applies a fixed function \code{logmean()} in a similar
manner, and \code{\link[=slapply]{slapply()}} which applies a function to a big matrix with
slicing but without splitting by an index factor.
}
\author{
Myles Lewis
}
