% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/scale_process_mirai.R
\name{par_multirasters_mirai}
\alias{par_multirasters_mirai}
\title{Parallelize spatial computation over multiple raster files}
\usage{
par_multirasters_mirai(filenames, fun_dist, ..., .debug = TRUE)
}
\arguments{
\item{filenames}{character. A vector or list of
full file paths of raster files. n is the total number of raster files.}

\item{fun_dist}{terra or chopin functions that accept \code{SpatRaster}
object in an argument. In particular, \code{x} and \code{y} arguments
should be present and \code{x} should be a \code{SpatRaster}.}

\item{...}{Arguments passed to the argument \code{fun_dist}.}

\item{.debug}{logical(1). Default is \code{FALSE}. If \code{TRUE} and
a unit computation fails, the error message and the file path
where the error occurred will be included in the output.}
}
\value{
a data.frame object with computation results.
For entries of the results,
consult the function used in \code{fun_dist} argument.
}
\description{
Large raster files usually exceed the memory capacity in size.
This function can be helpful to process heterogenous raster files with
homogeneous summary functions. Heterogenous raster files refer to
rasters with different spatial extents and resolutions.
Cropping a large raster into a small subset even consumes
a lot of memory and adds processing time.
This function leverages \code{terra} \code{SpatRaster}
to distribute computation jobs over multiple threads.
It is assumed that users have multiple large raster files
in their disk, then each file path is assigned to a thread.
Each thread will directly read raster values from
the disk using C++ pointers that operate in terra functions.
For use, it is strongly recommended to use vector data with
small and confined spatial extent for computation to avoid
out-of-memory error. \code{y} argument in \code{fun_dist} will be used as-is.
That means no preprocessing or subsetting will be
applied. Please be aware of the spatial extent and size of the
inputs.
}
\examples{
\donttest{
lastpar <- par(mfrow = c(1, 1))
library(terra)
library(sf)
library(mirai)
options(sf_use_s2 = FALSE)
mirai::daemons(4)

nccnty <- sf::st_read(
  system.file("shape/nc.shp", package = "sf")
)
nccnty <- sf::st_transform(nccnty, "EPSG:5070")
nccnty <- nccnty[seq_len(30L), ]

nccntygrid <- sf::st_make_grid(nccnty, n = c(200, 100))
nccntygrid <- sf::st_as_sf(nccntygrid)
nccntygrid$GEOID <- sprintf("\%05d", seq_len(nrow(nccntygrid)))
nccntygrid <- sf::st_intersection(nccntygrid, nccnty)

rrast <- terra::rast(nccnty, nrow = 600, ncol = 1320)
terra::values(rrast) <- rgamma(7.92e5, 4, 2)

tdir <- tempdir(check = TRUE)
terra::writeRaster(rrast, file.path(tdir, "test1.tif"), overwrite = TRUE)
terra::writeRaster(rrast, file.path(tdir, "test2.tif"), overwrite = TRUE)
testfiles <- list.files(tdir, pattern = "tif$", full.names = TRUE)

res <- par_multirasters_mirai(
  filenames = testfiles,
  fun_dist = extract_at,
  x = rrast,
  y = nccnty,
  id = "GEOID",
  func = "mean"
)
mirai::daemons(0L)
par(lastpar)
}
}
\seealso{
\code{\link[mirai:mirai]{mirai::mirai}}, \code{\link[mirai:mirai_map]{mirai::mirai_map}}, \code{\link[mirai:daemons]{mirai::daemons}},
\code{\link{par_convert_f}}

Other Parallelization: 
\code{\link{par_cut_coords}()},
\code{\link{par_grid}()},
\code{\link{par_grid_mirai}()},
\code{\link{par_hierarchy}()},
\code{\link{par_hierarchy_mirai}()},
\code{\link{par_make_dggrid}()},
\code{\link{par_make_grid}()},
\code{\link{par_make_h3}()},
\code{\link{par_merge_grid}()},
\code{\link{par_multirasters}()},
\code{\link{par_pad_balanced}()},
\code{\link{par_pad_grid}()},
\code{\link{par_split_list}()}
}
\author{
Insang Song \email{geoissong@gmail.com}
}
\concept{Parallelization}
