% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/problem.R
\name{mle_problem}
\alias{mle_problem}
\alias{print.mle_problem}
\title{Create an MLE Problem Specification}
\usage{
mle_problem(
  loglike,
  score = NULL,
  fisher = NULL,
  constraint = NULL,
  theta_names = NULL,
  n_obs = NULL,
  cache_derivatives = FALSE
)

\method{print}{mle_problem}(x, ...)
}
\arguments{
\item{loglike}{Log-likelihood function taking parameter vector theta}

\item{score}{Score function (gradient of log-likelihood). If NULL,
computed numerically via numDeriv::grad when needed.}

\item{fisher}{Fisher information matrix function. If NULL, computed
numerically via numDeriv::hessian when needed.}

\item{constraint}{Domain constraints as mle_constraint object}

\item{theta_names}{Character vector of parameter names for nice output}

\item{n_obs}{Number of observations (for AIC/BIC computation)}

\item{cache_derivatives}{Logical; if TRUE and score/fisher are computed
numerically, cache the most recent result to avoid redundant computation.
This is particularly useful during line search where the same point may
be evaluated multiple times. Default is FALSE.}

\item{x}{An mle_problem object.}

\item{...}{Additional arguments (unused).}
}
\value{
An mle_problem object

The input object, invisibly (for method chaining).
}
\description{
Encapsulates a maximum likelihood estimation problem, separating the
statistical specification from the optimization strategy.
}
\details{
The problem object provides lazy evaluation of derivatives. If you don't
provide analytic score or fisher functions, they will be computed
numerically when requested.

When \code{cache_derivatives = TRUE}, numerical derivatives are cached
using a single-value cache (stores the most recent theta and result).
This is efficient for optimization where consecutive calls often evaluate
at the same point (e.g., during line search or convergence checking).
Use \code{\link{clear_cache}} to manually clear the cache if needed.
}
\examples{
# With analytic derivatives
problem <- mle_problem(
  loglike = function(theta) sum(dnorm(data, theta[1], theta[2], log = TRUE)),
  score = function(theta) {
    c(sum(data - theta[1]) / theta[2]^2,
      -length(data)/theta[2] + sum((data - theta[1])^2) / theta[2]^3)
  },
  constraint = mle_constraint(
    support = function(theta) theta[2] > 0,
    project = function(theta) c(theta[1], max(theta[2], 1e-8))
  ),
  theta_names = c("mu", "sigma")
)

# Without analytic derivatives (computed numerically)
problem <- mle_problem(
  loglike = function(theta) sum(dnorm(data, theta[1], theta[2], log = TRUE)),
  constraint = mle_constraint(
    support = function(theta) theta[2] > 0
  )
)

}
