distfreereg.default <-
  function(test_mean = NULL,
           ordering = "simplex",
           group = TRUE,
           stat = c("KS", "CvM"),
           B = 1e4,
           control = NULL,
           override = NULL,
           verbose = TRUE,
           ...,
           Y,
           X = NULL,
           covariance,
           J,
           fitted_values){
    output <- list()
    
    cl <- match.call()
    output[["call"]] <- cl
    output[["data"]] <- list(Y = Y, X = X)
    
    extra_arg_list <- list(...)
    
    vargs <-
      validate_args_distfreereg_default(Y = Y, X = X, covariance = covariance,
                                        ordering = ordering, J = J,
                                        fitted_values = fitted_values,
                                        group = group, stat = stat, B = B,
                                        control = control, override = override,
                                        verbose = verbose,
                                        extra_arg_list = extra_arg_list)
    Y <- vargs[["Y"]]; X <- vargs[["X"]]; n <- vargs[["n"]]; J <- vargs[["J"]]
    covariance = vargs[["covariance"]]; matsqrt_tol = vargs[["matsqrt_tol"]]
    solve_tol = vargs[["solve_tol"]]; qr_tol <- vargs[["qr_tol"]]
    orth_tol <- vargs[["orth_tol"]]; trans_tol <- vargs[["trans_tol"]]
    B <- vargs[["B"]]; return_on_error <- vargs[["return_on_error"]]
    
    output[["data"]] <- list(Y = Y, X = X)
    
    p <- ncol(J)
    
    if(is.null(covariance[["Q"]])){
      if(isTRUE(verbose)) message("Calculating the inverse square root of the covariance matrix...")
      covariance <- fill_covariance_list(need = "Q", covariance_list = covariance,
                                         matsqrt_tol = matsqrt_tol, solve_tol = solve_tol)
    } else{
      if(isTRUE(verbose)) message("Using supplied inverse square root of the covariance matrix...")
    }
    Q <- covariance[["Q"]]
    # if(any(diag(as.matrix(Q)) == 0)) warning("At least one diagonal element of Q is indistinguishable from zero")
    
    output[["covariance"]] <- covariance
    output[["fitted_values"]] <- fitted_values
    output[["J"]] <- J
    
    J_for_mu <- covprod(Q, J)
    
    if(isTRUE(verbose)) message("Calculating mu...")
    output[["mu"]] <- mu <- tryReturn(
      calc_mu(J = J_for_mu, solve_tol = solve_tol, orth_tol = orth_tol),
      output = output, return_on_error = return_on_error)
    
    if(is.null(override[["res_order"]])){
      X_for_ordering <- if(is.null(control[["data"]])) X else control[["data"]]
      output[["res_order"]] <- res_order <- tryReturn(
        determine_order(X = X_for_ordering, ordering = ordering, n = n,
                        verbose = verbose),
        output = output,
        return_on_error = return_on_error)
    } else {
      if(isTRUE(verbose)) message("Using supplied observation ordering...")
      output[["res_order"]] <- res_order <- override[["res_order"]]
    }
    
    # Verify that res_order is an ordering.
    tryReturn(
      if(!isOrdering(x = res_order, len = n)) stop("res_order not an ordering"),
      return_on_error = return_on_error, output = output
    )

    
    # Define r
    if(is.null(override[["r"]])){
      if(isTRUE(verbose)) message("Calculating transformation anchors...")
      output[["r"]] <- r <- tryReturn(
        define_r(n = n, p = p, res_order = order(res_order), gs_tol = qr_tol),
        output = output, return_on_error = return_on_error)
    } else {
      if(isTRUE(verbose)) message("Using supplied transformation anchors...")
      output[["r"]] <- r <- as.matrix(override[["r"]])
      stopifnot(all(dim(mu) == dim(r)))
    }
    
    # Validate r
    rtr <- crossprod(r)
    tryReturn(
      if(!isTRUE(all.equal(rtr, diag(nrow(rtr)), check.attributes = FALSE,
                           tolerance = orth_tol)))
        stop("Columns of r are not orthogonal"),
      output = output, return_on_error = return_on_error
    )
    
    
    if(isTRUE(verbose)) message("Calculating r_tilde...")
    output[["r_tilde"]] <- r_tilde <- tryReturn(
      define_r_tilde(r = r, mu = mu, k2_tol = trans_tol),
      output = output, return_on_error = return_on_error)
    
    if(isTRUE(verbose)) message("Calculating residuals...")
    output[["residuals"]][["raw"]] <- raw_residuals <- tryReturn(
      calc_resid(Y = Y, fitted_values = fitted_values),
      output = output, return_on_error = return_on_error)
    
    tryReturn(
      validate_numeric(raw_residuals, min_len = 1, func = warning,
                       message = "Raw residuals failed validation: "),
      output = output, return_on_error = return_on_error)
    
    output[["residuals"]][["sphered"]] <- sphered_residuals <- tryReturn(
      calc_sphered_resid(raw_residuals = raw_residuals, Q = Q),
      output = output, return_on_error = return_on_error)
    
    tryReturn(
      validate_numeric(sphered_residuals, min_len = 1, func = warning,
                       message = "Sphered residuals failed validation: "),
      output = output, return_on_error = return_on_error)
    
    output[["residuals"]][["transformed"]] <- transformed_residuals <- tryReturn(
      calc_k2_resid(x = sphered_residuals, r_tilde = r_tilde, mu = mu,
                    k2_tol = trans_tol),
      output = output, return_on_error = return_on_error)
    
    tryReturn(
      validate_numeric(transformed_residuals, min_len = 1, func = warning,
                       message = "Transformed residuals failed validation: "),
      output = output, return_on_error = return_on_error)
    
    
    if(isTRUE(group)){
      if(length(unique(res_order)) == n) {
        if(isTRUE(verbose)) message("All covariate observations are unique; no grouping done...")
        grouping_matrix <- NULL
      } else {
        if(isTRUE(verbose)) message("Determining grouping...")
        # control[["data"]] should NOT have response in it here if "ordering" is
        # "natural".
        X_for_gm <- if(is.null(control[["data"]])) as.data.frame(X) else control[["data"]]
        output[["grouping_matrix"]] <- grouping_matrix <- tryReturn(
          define_grouping_matrix(X = X_for_gm, res_order = res_order),
          output = output, return_on_error = return_on_error)
      }
    } else {
      grouping_matrix <- NULL
    }
    
    if(isTRUE(verbose)) message("Calculating empirical partial sum process...")
    output[["epsp"]] <- epsp <- tryReturn(
      calc_epsp(transformed_residuals = transformed_residuals,
                res_order = order(res_order), grouping_matrix = grouping_matrix),
      output = output, return_on_error = return_on_error)
    
    tryReturn(
      validate_numeric(epsp, min_len = 1, func = warning,
                       message = "Partial sum process failed validation: "),
      output = output, return_on_error = return_on_error)
    
    if(isTRUE(verbose)) message("Calculating observed statistic(s)...")
    observed_stats <- tryReturn(
      as.list(calc_stats(epsp = epsp, stat = stat)),
      output = output, return_on_error = return_on_error)
    names(observed_stats) <- stat
    output[["observed_stats"]] <- observed_stats
    
    if(is.null(override[["mcsim_stats"]])){
      if(isTRUE(verbose)) message("Running Monte Carlo simulation...")
      mcsim_stats <- mcsim(B = B, r = r, grouping_matrix = grouping_matrix,
                           stat = stat, verbose = verbose)
    } else {
      if(isTRUE(verbose)) message("Using supplied Monte Carlo simulation results...")
      mcsim_stats <- override[["mcsim_stats"]]
    }
    output[["mcsim_stats"]] <- mcsim_stats
    
    # Now redefine "p" to hold p values (not dimension of parameter space).
    p <- tryReturn(
      calc_p(observed_stats = observed_stats, mcsim_stats = mcsim_stats),
      output = output, return_on_error = return_on_error)
    output[["p"]] <- list()
    output[["p"]][["value"]] <- p
    output[["p"]][["mcse"]] <- tryReturn(
      calc_mcse_binom(p = p, B = B),
      output = output, return_on_error = return_on_error)
    
    
    class(output) <- "distfreereg"
    return(output)
  }
