#' Calculate Multifunctionality Richness (MFric)
#'
#' @description
#' This function calculates the Multifunctionality Richness (MFric) for each row in a dataset.
#' MFric represents the weighted average level of multiple ecosystem function indicators,
#' reflecting the overall performance of an ecosystem across various functional metrics.
#' Only positive function values are included in the calculation.
#'
#' @param data A numeric data frame or matrix where rows represent observations (e.g., sites, plots)
#'   and columns represent different ecosystem functions. Function values should be non-negative,
#'   with values <= 0 excluded from calculations.
#' @param weights A numeric vector of weights for each function (column) in the data.
#'   If NULL (default), equal weights of 1 are assigned to all functions.
#'   The length of the weights vector must match the number of columns in data.
#'
#' @return A data frame with a single column named "MFric" containing the calculated
#'   Multifunctionality Richness values for each row in the input data.
#'   Row names are preserved from the input data if available.
#'
#' @details
#' The MFric is calculated as a weighted average of function values:
#' \deqn{MFric = \frac{\sum_{i=1}^{n} w_i f_i}{\sum_{i=1}^{n} w_i}}
#' where \eqn{f_i} represents the performance level of function i (with \eqn{f_i > 0}),
#' and \eqn{w_i} denotes the weight assigned to function i.
#'
#' @note
#' \itemize{
#'   \item Function values <= 0 are automatically excluded from calculations
#'   \item If all function values in a row are <= 0, the result will be NaN
#'   \item Weights are applied only to functions with positive values
#' }
#'
#' @examples
#' # Example with sample data
#' data <- data.frame(
#'   func1 = c(0.5, 0.3, 0.2),
#'   func2 = c(0.2, 0.4, 0.4),
#'   func3 = c(0.3, 0.3, 0.4)
#' )
#' rownames(data) <- c("Site1", "Site2", "Site3")
#'
#' # Calculate MFric with equal weights
#' MFric(data)
#'
#' # Calculate MFric with custom weights
#' MFric(data, weights = c(1, 2, 1))
#'
#'
#' @export
MFric <- function(data, weights = NULL) {
  # If no weights are provided, create a weight vector with all 1's
  if (is.null(weights)) {
    weights <- rep(1, ncol(data))
  }

  if (length(weights) != ncol(data)) {
    stop("The length of the weight vector must be equal to the number of columns in the data frame")
  }

  mfric_values <- apply(data, 1, function(fi) {
    wi <- weights[fi > 0]
    fi <- fi[fi > 0]
    sum(wi * fi) / sum(wi)
  })

  result <- data.frame(MFric = mfric_values)
  rownames(result) <- rownames(data)

  return(result)
}
