#' Marginal Reliability of a Unidimensional IRT Model
#'
#' @description
#' Compute marginal reliability for a unidimensional IRT model using either
#' MLE-based or EAP-based information, via Gaussian quadrature over a
#' standard normal ability distribution.
#'
#' @param ip A data frame or matrix of item parameters with columns in the order
#'   \code{b}, \code{a}, \code{c}, where \code{a} is on the \code{D = 1.702}
#'   metric. If only 1 or 2 columns are supplied, the \code{info()} function is
#'   expected to treat them as 1PL/2PL accordingly.
#' @param est A character string specifying the ability estimation method:
#'   \code{"MLE"} for maximum likelihood or \code{"EAP"} for empirical Bayes.
#'
#' @details
#' Gaussian quadrature with 41 nodes on \code{[-5, 5]} is used to approximate
#' the integrals.
#'
#' @return A single numeric value: the marginal reliability (MLE or EAP,
#'   depending on \code{est}).
#'
#' @examples
#' data(ip.u)
#' rel_info(ip.u, "MLE")
#'
#' @export
rel_info <- function(ip, est) {

  # --- argument checks --------------------------------------------------------
  if (missing(ip)) {
    stop("`ip` must be supplied as a data frame or matrix of item parameters.")
  }
  if (!is.data.frame(ip) && !is.matrix(ip)) {
    stop("`ip` must be a data frame or a matrix.")
  }

  ip <- as.data.frame(ip)
  if (!all(vapply(ip, is.numeric, logical(1L)))) {
    stop("All columns in `ip` must be numeric.")
  }
  if (nrow(ip) < 1L) {
    stop("`ip` must contain at least one item (one row).")
  }

  if (missing(est)) {
    stop("`est` must be supplied and be either 'MLE' or 'EAP'.")
  }

  est <- match.arg(est, choices = c("MLE", "EAP"))

  # --- quadrature nodes and weights (standard normal) -------------------------
  qp <- normal_quadra(41, 5)  # nodes in [-5, 5], weights sum to 1

  if (est == "MLE") {

    # test information at quadrature nodes (MLE)
    ipinfo <- as.data.frame(info(theta = qp$nodes, ip = ip, est = "MLE"))

    # attach weights
    ipinfo$weight <- qp$weights

    # weighted information
    ipinfo$info_weighted <- ipinfo$weight * ipinfo$infoMLE

    # marginal reliability (MLE)
    I_bar <- sum(ipinfo$info_weighted)
    marginalRelMLE <- I_bar / (I_bar + 1)

    return(marginalRelMLE)

  } else { # est == "EAP"

    # test information at quadrature nodes (EAP)
    ipinfo <- as.data.frame(info(theta = qp$nodes, ip = ip, est = "EAP"))

    # attach weights
    ipinfo$weight <- qp$weights

    # inverse information
    ipinfo$inv_infoEAP <- 1 / ipinfo$infoEAP

    # marginal reliability (EAP)
    marginalRelEAP <- 1 - sum(ipinfo$inv_infoEAP * ipinfo$weight)

    return(marginalRelEAP)
  }
}
