
#' Switch X's and O's
#'
#' @description
#' Convenience function called internally by \code{play4inaRow} that switches
#' the player symbols, so that each player sees their own pieces as \code{"X"}
#' and their opponents' as \code{"O"}.
#'
#' @param game A \code{6x7} matrix object representing the current game board.
#'
#' @returns Returns the \code{game} matrix with \code{"X"} and \code{"O"}
#'   symbols switched.
#'
#' @export
#'
#' @examples
#' game <- matrix(sample(c('X','O','.'), 6*7, replace=TRUE), nrow = 6, ncol = 7)
#' game
#' invertPieces(game)
invertPieces <- function(game){
  game[which(game=='X')] <- 'N'
  game[which(game=='O')] <- 'X'
  game[which(game=='N')] <- 'O'
  return(game)
}


#' Simulate many games of Four in a Row
#'
#' @description
#' This is a convenient way to test two bots against each other over a large
#' number of games while ensuring that neither bot has an advantage from going
#' first more often than the other.
#'
#' @param playerOne A function that takes the current board as input and returns
#'   the next move (1-7) for Player 1.
#' @param playerTwo Same for Player 2. Note that both functions see their pieces
#'   as "X" and opponent's pieces as "O".
#' @param n Numerical value of how many games to simulate. Must be an even
#'   number, so that both bots get the same number of games with the first move
#'   (default = 100).
#'
#' @returns Returns a vector of counts for the number of \code{ties}, wins by
#'   \code{playerOne}, and wins by \code{playerTwo}.
#'
#' @examples
#' testBots(easyBot, randomBot)
#'
#' @export
testBots <- function(playerOne, playerTwo, n = 100){
  if(n %% 2 != 0 | n < 2){
    stop('Number of tests (n) must be even and greater than 0.')
  }
  res1 <- replicate(n/2, play4inaRow(playerOne, playerTwo, verbose = FALSE))
  res2 <- replicate(n/2, play4inaRow(playerTwo, playerOne, verbose = FALSE))
  return(c(ties = sum(res1 == 0) + sum(res2 == 0),
           playerOne = sum(res1 == 1) + sum(res2 == 2),
           playerTwo = sum(res1 == 2) + sum(res2 == 1)))
}


#' All possible sets of 4 in a row
#'
#' @description
#' Manually constructs a dataset containing all possible winning sets of four
#' in a standard (\code{6x7}) game of Four in a Row.
#'
#' @returns A matrix with 69 rows and 4 columns. Each row contains numeric
#'   indices for a potential winning set of four in a row. For example,
#'   \code{sets[1,]} represents the set of four vertical spaces starting in the
#'   top left corner.
#'
#' @examples
#' getSetsof4()
#'
#' @export
getSetsof4 <- function(){
  # make all possible winning rows of four
  sets <- list()
  # vertical
  for(R in 1:3){
    for(C in 1:7){
      sets[[length(sets)+1]] <- cbind(c(R:(R+3)), C)
    }
  }
  # horizontal
  for(R in 1:6){
    for(C in 1:4){
      sets[[length(sets)+1]] <- cbind(R, c(C:(C+3)))
    }
  }
  # down + right
  for(R in 1:3){
    for(C in 1:4){
      sets[[length(sets)+1]] <- cbind(c(R:(R+3)), c(C:(C+3)))
    }
  }
  # down + left
  for(R in 1:3){
    for(C in 1:4){
      sets[[length(sets)+1]] <- cbind(c(R:(R+3)), c((C+3):C))
    }
  }
  # convert to vector indices
  sets <- t(sapply(sets, function(s){
    6*(s[,2]-1) + s[,1]
  }))
  return(sets)
}


