% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/geom.R
\name{g_factory}
\alias{g_factory}
\alias{g_create}
\alias{g_add_geom}
\alias{g_get_geom}
\title{Create WKB/WKT geometries from vertices, and add/get sub-geometries}
\usage{
g_create(
  geom_type,
  pts = NULL,
  as_wkb = TRUE,
  as_iso = FALSE,
  byte_order = "LSB"
)

g_add_geom(
  sub_geom,
  container,
  as_wkb = TRUE,
  as_iso = FALSE,
  byte_order = "LSB"
)

g_get_geom(
  container,
  sub_geom_idx,
  as_wkb = TRUE,
  as_iso = FALSE,
  byte_order = "LSB"
)
}
\arguments{
\item{geom_type}{Character string (case-insensitive), one of \code{"POINT"},
\code{"MULTIPOINT"}, \code{"LINESTRING"}, \code{"POLYGON"} (see Note) or
\code{"GEOMETRYCOLLECTION"}.}

\item{pts}{Numeric matrix of points (x, y, z, m), or \code{NULL} to create an
empty geometry. The points can be given as (x, y), (x, y, z) or
(x, y, z, m), so the input must have two, three or four columns.
Data frame input will be coerced to numeric matrix. Rings for polygon
geometries should be closed.}

\item{as_wkb}{Logical value, \code{TRUE} to return the output geometry in WKB
format (the default), or \code{FALSE} to return a WKT string.}

\item{as_iso}{Logical value, \code{TRUE} to export as ISO WKB/WKT (ISO 13249
SQL/MM Part 3), or \code{FALSE} (the default) to export as "Extended WKB/WKT".}

\item{byte_order}{Character string specifying the byte order when output is
WKB. One of \code{"LSB"} (the default) or \code{"MSB"} (uncommon).}

\item{sub_geom}{Either a raw vector of WKB or a character string of WKT.}

\item{container}{Either a raw vector of WKB or a character string of WKT for
a container geometry type.}

\item{sub_geom_idx}{An integer value giving the 1-based index of a
sub-geometry (numeric values will be coerced to integer by truncation).}
}
\value{
A geometry as WKB raw vector by default, or a WKT string if
\code{as_wkb = FALSE}. In the case of multiple input points for creating Point
geometry type, a list of WKB raw vectors or character vector of WKT strings
will be returned.
}
\description{
These functions create WKB/WKT geometries from input vertices, and build
container geometry types from sub-geometries.
}
\details{
These functions use the GEOS library via GDAL headers.

\code{g_create()} creates a geometry object from the given point(s) and returns
a raw vector of WKB (the default) or a character string of WKT. Currently
supports creating \code{Point}, \code{MultiPoint}, \code{LineString}, \code{Polygon}, and
\code{GeometryCollection.}
If multiple input points are given for creating \code{Point} type, then multiple
geometries will be returned as a list of WKB raw vectors, or character
vector of WKT strings (if \code{as_wkb = FALSE}). Otherwise, a single geometry
is created from the input points. Only an empty \code{GeometryCollection} can be
created with this function, for subsequent use with \code{g_add_geom()}.

\code{g_add_geom()} adds a geometry to a geometry container, e.g.,
\code{Polygon} to \code{Polygon} (to add an interior ring), \code{Point} to \code{MultiPoint},
\code{LineString} to \code{MultiLineString}, \code{Polygon} to \code{MultiPolygon}, or mixed
geometry types to a \code{GeometryCollection}. Returns a new geometry, i.e,
the container geometry is not modified.

\code{g_get_geom()} fetches a geometry from a geometry container (1-based
indexing). For a polygon, requesting the first sub-geometry returns the
exterior ring (\code{sub_geom_idx = 1}), and the interior rings are returned for
\code{sub_geom_idx > 1}.
}
\note{
A \code{POLYGON} can be created for a single ring which will be the
exterior ring. Additional \code{POLYGON}s can be created and added to an
existing \code{POLYGON} with \code{g_add_geom()}. These will become interior rings.
Alternatively, an empty polygon can be created with \code{g_create("POLYGON")},
followed by creation and addition of \code{POLYGON}s as sub-geometries. In that
case, the first added \code{POLYGON} will be the exterior ring. The next ones will
be the interior rings.

Only an empty \code{GeometryCollection} can be created with \code{g_create()}, which
can then be used as a container with \code{g_add_geom()}. If given, input points
will be ignored by \code{g_create()} if \code{geom_type = "GEOMETRYCOLLECTION"}.
}
\examples{
# raw vector of WKB by default
g_create("POINT", c(1, 2))

# as WKT
g_create("POINT", c(1, 2), as_wkb = FALSE)

# or convert in either direction
g_create("POINT", c(1, 2)) |> g_wk2wk()
g_create("POINT", c(1, 2), as_wkb = FALSE) |> g_wk2wk()

# create MultiPoint from a matrix of xyz points
x <- c(9, 1)
y <- c(1, 9)
z <- c(0, 10)
pts <- cbind(x, y, z)
mp <- g_create("MULTIPOINT", pts)
g_wk2wk(mp)
g_wk2wk(mp, as_iso = TRUE)

# create an empty container and add sub-geometries
mp2 <- g_create("MULTIPOINT")
mp2 <- g_create("POINT", c(11, 2)) |> g_add_geom(mp2)
mp2 <- g_create("POINT", c(12, 3)) |> g_add_geom(mp2)
g_wk2wk(mp2)

# get sub-geometry from container
g_get_geom(mp2, 2, as_wkb = FALSE)

# plot WKT strings or a list of WKB raw vectors
pts <- c(0, 0,
         3, 0,
         3, 4,
         0, 0)
m <- matrix(pts, ncol = 2, byrow = TRUE)
(g <- g_create("POLYGON", m, as_wkb = FALSE))
plot_geom(g)
}
