% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/06-spatial-join.R
\name{universal_spatial_join}
\alias{universal_spatial_join}
\title{Universal Spatial Join - Complete Implementation}
\usage{
universal_spatial_join(
  source_data,
  target_data,
  method = "auto",
  scale_factor = NULL,
  summary_function = "mean",
  buffer_distance = NULL,
  temporal_tolerance = NULL,
  crs_target = NULL,
  na_strategy = "remove",
  chunk_size = 1e+06,
  parallel = FALSE,
  verbose = FALSE
)
}
\arguments{
\item{source_data}{Source spatial data. Can be:
\itemize{
\item File path: "/path/to/data.tif" or "/path/to/data.shp"
\item Directory: "/path/to/spatial_files/" (processes all spatial files)
\item R object: SpatRaster, sf object, data.frame with coordinates
\item List: Multiple files, raster stack, or sf objects
}}

\item{target_data}{Target spatial data (same format options as source_data).
Can be NULL for scaling operations with scale_factor.}

\item{method}{Spatial join method:
\itemize{
\item \strong{"auto"}: Automatically detect best method (default)
\item \strong{"extract"}: Extract raster values to vector features
\item \strong{"overlay"}: Spatial intersection/overlay of vectors
\item \strong{"resample"}: Resample raster to match target geometry
\item \strong{"zonal"}: Calculate zonal statistics (raster → vector)
\item \strong{"nearest"}: Nearest neighbor spatial join
\item \strong{"interpolate"}: Spatial interpolation (IDW, kriging)
\item \strong{"temporal"}: Time-aware spatial join
}}

\item{scale_factor}{Numeric (> 0 if provided). Scale factor for resolution changes:
\itemize{
\item \code{NULL}: Use target data resolution (default)
\item \code{> 1}: Coarser resolution (e.g., 2 = half resolution)
\item \code{< 1}: Finer resolution (e.g., 0.5 = double resolution)
\item Custom: Any positive number for specific scaling
}}

\item{summary_function}{Character. Function for aggregating overlapping values:
\itemize{
\item \strong{"mean"}: Average values (default for continuous data)
\item \strong{"median"}: Median values (robust to outliers)
\item \strong{"max"}/\strong{"min"}: Maximum/minimum values
\item \strong{"sum"}: Sum values (useful for counts, areas)
\item \strong{"sd"}: Standard deviation (measure variability)
\item \strong{"mode"}/\strong{"majority"}: Most frequent value (categorical data)
}}

\item{buffer_distance}{Numeric (>= 0 if provided). Buffer distance in map units:
\itemize{
\item For point extraction: Buffer around points
\item For line extraction: Buffer along lines
\item For nearest neighbor: Search radius
\item Units: Same as source data CRS (meters, degrees, etc.)
}}

\item{temporal_tolerance}{Numeric (>= 0 if provided). Time tolerance for temporal joins (in days):
\itemize{
\item Maximum time difference for matching observations
\item Only used with method = "temporal"
\item Example: 7 = match within 7 days
}}

\item{crs_target}{Character or numeric. Target coordinate reference system:
\itemize{
\item EPSG code: 4326, 3857, etc.
\item PROJ string: "+proj=utm +zone=33 +datum=WGS84"
\item NULL: Use source data CRS (default)
}}

\item{na_strategy}{Character. Strategy for handling NA values:
\itemize{
\item \strong{"remove"}: Keep NAs as missing (default)
\item \strong{"nearest"}: Replace with nearest neighbor value
\item \strong{"interpolate"}: Spatial interpolation of NAs
\item \strong{"zero"}: Replace NAs with zero
}}

\item{chunk_size}{Numeric (> 0). Chunk size for processing large datasets:
\itemize{
\item Number of features/cells to process at once
\item Larger = faster but more memory
\item Smaller = slower but less memory
\item Default: 1,000,000
}}

\item{parallel}{Logical. Use parallel processing:
\itemize{
\item TRUE: Use multiple cores (faster for large data)
\item FALSE: Single core processing (default)
\item Requires 'parallel' package
}}

\item{verbose}{Logical. Print detailed progress messages:
\itemize{
\item TRUE: Show processing steps and diagnostics
\item FALSE: Silent processing (default)
}}
}
\value{
Spatial data object with joined attributes. Return type depends on operation:
\describe{
\item{extract (vector → raster)}{sf object with new columns containing extracted raster values. Original geometry preserved, new columns named "extracted_" followed by the raster layer name}
\item{zonal (raster → vector)}{sf object with new columns containing zonal statistics. Original geometry preserved, new columns named "zonal_" followed by the statistic name and raster layer name}
\item{resample (raster → raster)}{SpatRaster with resampled/processed data matching target resolution or scale factor}
\item{overlay (vector → vector)}{sf object with intersected/overlaid features combining attributes from both datasets}
\item{nearest}{sf object with attributes from nearest features joined}
}

\strong{Returned objects include 'spatial_join_info' attribute containing:}
\itemize{
\item method: Join method used
\item source_type, target_type: Data types processed
\item processing_time: Time taken (if verbose=TRUE)
\item timestamp: Processing timestamp
\item summary_function: Aggregation function used
}
}
\description{
Comprehensive spatial join system that handles ALL spatial data combinations:
Vector to Vector, Vector to Raster, Raster to Raster with full documentation,
error handling, and extensive examples. This replaces all previous spatial
join functions with a unified, robust system.
}
\details{
\subsection{Quick Start Guide:}{

Most common use case - extract raster values to point locations:

\if{html}{\out{<div class="sourceCode r">}}\preformatted{result <- universal_spatial_join("my_points.csv", "my_raster.tif", method="extract")
}\if{html}{\out{</div>}}
}

\subsection{Supported Operations:}{
\subsection{Data Type Combinations:}{
\itemize{
\item \strong{Vector → Raster}: Extract raster values to points/polygons/lines
\item \strong{Raster → Vector}: Calculate zonal statistics for polygons
\item \strong{Raster → Raster}: Resample, overlay, mathematical operations
\item \strong{Vector → Vector}: Spatial intersections, overlays, nearest neighbor
}
}

\subsection{Input Format Support:}{
\itemize{
\item \strong{File paths}: ".tif", ".shp", ".gpkg", ".geojson", ".nc"
\item \strong{Directories}: Automatically processes all spatial files
\item \strong{R objects}: SpatRaster, sf, data.frame with coordinates
\item \strong{Lists}: Multiple files or raster stacks
}
}

\subsection{Scaling Operations:}{
\itemize{
\item \strong{Up-scaling}: Aggregate to coarser resolution (scale_factor > 1)
\item \strong{Down-scaling}: Interpolate to finer resolution (scale_factor < 1)
\item \strong{Custom resolution}: Match target raster geometry
}
}

\subsection{Error Handling:}{
\itemize{
\item \strong{Auto CRS reprojection}: Handles coordinate system mismatches
\item \strong{Geometry alignment}: Auto-crops, extends, or resamples as needed
\item \strong{NA handling}: Multiple strategies for missing data
\item \strong{Memory management}: Chunked processing for large datasets
}
}

\subsection{Method Selection Guide:}{

\describe{
\item{extract}{Use when you have point/polygon locations and want to get values from a raster}
\item{zonal}{Use when you have polygons and want statistics from raster data within each polygon}
\item{resample}{Use when you need to change raster resolution or align two rasters}
\item{overlay}{Use when joining two vector datasets based on spatial relationships}
\item{nearest}{Use when you want to find the closest features between two vector datasets}
\item{auto}{Let the function choose - works well for standard extract/resample operations}
}
}

}
}
\section{Common Error Solutions}{

\describe{
\item{CRS Mismatch}{"CRS mismatch detected" - Function automatically reprojects data, but manual CRS checking recommended for precision}
\item{Memory Issues}{"Large dataset processing" - Reduce chunk_size parameter (try 500000) or set parallel=FALSE}
\item{No Spatial Overlap}{"No spatial overlap found" - Check that source and target data cover the same geographic area}
\item{File Not Found}{"File does not exist" - Verify file paths and ensure files exist at specified locations}
\item{Missing Bands}{"Required bands not found" - For raster operations, ensure expected spectral bands are present}
\item{Invalid Geometries}{"Geometry errors" - Function attempts to fix automatically, but check input data quality}
}
}

\section{Performance Tips}{

\itemize{
\item For large datasets (>1M cells): set chunk_size=500000 and parallel=TRUE
\item Use method="resample" with scale_factor > 1 to reduce data size before complex operations
\item For time series analysis: consider temporal_tolerance to balance accuracy vs processing speed
\item When processing multiple datasets: ensure consistent CRS to avoid reprojection overhead
\item For point extraction: use smaller buffer_distance when possible to reduce processing time
}
}

\examples{
\dontrun{
# These examples require satellite imagery files (Landsat/Sentinel data etc.)
# =================================================================
# MOST COMMON USE CASE: Extract raster values to CSV points
# =================================================================

# Your typical workflow: CSV file with coordinates + raster file
results <- universal_spatial_join(
  source_data = "my_field_sites.csv",    # CSV with lon, lat columns
  target_data = "satellite_image.tif",   # Any raster file
  method = "extract",                     # Extract raster values to points
  buffer_distance = 100,                  # 100m buffer around each point
  summary_function = "mean",              # Average within buffer
  verbose = TRUE                          # See what's happening
)

# Check results - original data + new columns with raster values
head(results)
#   site_id       lon       lat           geometry extracted_satellite_image
# 1       1 -83.12345  40.12345 POINT (-83.1 40.1)                   0.752
# 2       2 -83.23456  40.23456 POINT (-83.2 40.2)                   0.681
# 3       3 -83.34567  40.34567 POINT (-83.3 40.3)                   0.594

# Access the extracted values
results$extracted_satellite_image

# =================================================================
# ZONAL STATISTICS: Calculate statistics by polygon areas
# =================================================================

# Calculate average precipitation by watershed
watershed_precip <- universal_spatial_join(
  source_data = "precipitation_raster.tif",  # Raster data
  target_data = "watershed_boundaries.shp",  # Polygon boundaries
  method = "zonal",                           # Calculate zonal statistics
  summary_function = "mean",                  # Average precipitation per watershed
  verbose = TRUE
)

# Result: polygons with precipitation statistics
head(watershed_precip)
#   watershed_id                     geometry zonal_mean_precipitation_raster
# 1            1 POLYGON ((-84.2 40.1, ...))                             42.3
# 2            2 POLYGON ((-84.5 40.3, ...))                             38.7

# =================================================================
# RESAMPLE RASTER: Change resolution or align rasters
# =================================================================

# Aggregate 30m Landsat to 250m MODIS resolution
landsat_resampled <- universal_spatial_join(
  source_data = "landsat_30m.tif",      # High resolution input
  target_data = "modis_250m.tif",       # Target resolution template
  method = "resample",                   # Resample operation
  summary_function = "mean",             # Average when aggregating
  verbose = TRUE
)

# Check new resolution
terra::res(landsat_resampled)
# [1] 250 250

# Scale by factor instead of template
coarser_raster <- universal_spatial_join(
  source_data = "fine_resolution.tif",
  target_data = NULL,                    # No template needed
  method = "resample",
  scale_factor = 5,                      # 5x coarser resolution
  summary_function = "mean"
)

# =================================================================
# VECTOR OVERLAY: Join two vector datasets
# =================================================================

# Find which counties contain each field site
sites_with_counties <- universal_spatial_join(
  source_data = "field_sites.shp",      # Point data
  target_data = "county_boundaries.shp", # Polygon data
  method = "overlay",                     # Spatial intersection
  verbose = TRUE
)

# Result: points with county attributes added
head(sites_with_counties)
#   site_id           geometry county_name state_name
# 1       1 POINT (-83.1 40.1)    Franklin       Ohio
# 2       2 POINT (-83.2 40.2)    Delaware       Ohio

# =================================================================
# AUTO-DETECTION: Let function choose best method
# =================================================================

# Function automatically detects: points + raster = extract method
auto_result <- universal_spatial_join(
  source_data = my_points,               # Any point data
  target_data = my_raster,               # Any raster data
  method = "auto",                       # Automatically choose method
  verbose = TRUE                         # See what method was chosen
)
# Output: "Auto-detected method: extract for vector to raster"

# =================================================================
# ERROR HANDLING EXAMPLES
# =================================================================

# Function handles common issues automatically
robust_result <- universal_spatial_join(
  source_data = "points_wgs84.csv",     # WGS84 coordinate system
  target_data = "raster_utm.tif",       # UTM coordinate system
  method = "extract",
  na_strategy = "nearest",               # Handle missing values
  verbose = TRUE                         # See CRS handling messages
)
# Output: "CRS mismatch detected. Reprojecting to match raster CRS..."
}

}
\seealso{
\itemize{
\item \code{\link{raster_to_raster_ops}} for specialized raster operations
\item \code{\link{multiscale_operations}} for multi-scale analysis
\item \code{\link{process_vector_data}} for vector data preprocessing
}
}
