/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 * Copyright by The HDF Group.                                               *
 * All rights reserved.                                                      *
 *                                                                           *
 * This file is part of HDF5.  The full HDF5 copyright notice, including     *
 * terms governing use, modification, and redistribution, is contained in    *
 * the LICENSE file, which can be found at the root of the source code       *
 * distribution tree, or in https://www.hdfgroup.org/licenses.               *
 * If you do not have access to either file, you may request a copy from     *
 * help@hdfgroup.org.                                                        *
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

#define H5F_FRIEND 

#include "H5private.h"   
#include "H5Eprivate.h"  
#include "H5Fpkg.h"      
#include "H5FLprivate.h" 
#include "H5FOprivate.h" 
#include "H5Oprivate.h"  
#include "H5SLprivate.h" 

typedef struct H5FO_open_obj_t {
    haddr_t addr;    
    void   *obj;     
    bool    deleted; 
} H5FO_open_obj_t;

typedef struct H5FO_obj_count_t {
    haddr_t addr;  
    hsize_t count; 
} H5FO_obj_count_t;

H5FL_DEFINE_STATIC(H5FO_open_obj_t);

H5FL_DEFINE_STATIC(H5FO_obj_count_t);

herr_t
H5FO_create(const H5F_t *f)
{
    herr_t ret_value = SUCCEED; 

    FUNC_ENTER_NOAPI(FAIL)

    
    assert(f);
    assert(f->shared);

    
    if ((f->shared->open_objs = H5SL_create(H5SL_TYPE_HADDR, NULL)) == NULL)
        HGOTO_ERROR(H5E_FILE, H5E_CANTINIT, FAIL, "unable to create open object container");

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

void *
H5FO_opened(const H5F_t *f, haddr_t addr)
{
    H5FO_open_obj_t *open_obj;  
    void            *ret_value; 

    FUNC_ENTER_NOAPI_NOERR

    
    assert(f);
    assert(f->shared);
    assert(f->shared->open_objs);
    assert(H5_addr_defined(addr));

    
    if (NULL != (open_obj = (H5FO_open_obj_t *)H5SL_search(f->shared->open_objs, &addr))) {
        ret_value = open_obj->obj;
        assert(ret_value != NULL);
    } 
    else
        ret_value = NULL;

    FUNC_LEAVE_NOAPI(ret_value)
} 

herr_t
H5FO_insert(const H5F_t *f, haddr_t addr, void *obj, bool delete_flag)
{
    H5FO_open_obj_t *open_obj;            
    herr_t           ret_value = SUCCEED; 

    FUNC_ENTER_NOAPI(FAIL)

    
    assert(f);
    assert(f->shared);
    assert(f->shared->open_objs);
    assert(H5_addr_defined(addr));
    assert(obj);

    
    if ((open_obj = H5FL_MALLOC(H5FO_open_obj_t)) == NULL)
        HGOTO_ERROR(H5E_CACHE, H5E_NOSPACE, FAIL, "memory allocation failed");

    
    open_obj->addr    = addr;
    open_obj->obj     = obj;
    open_obj->deleted = delete_flag;

    
    if (H5SL_insert(f->shared->open_objs, &open_obj->addr, open_obj) < 0)
        HGOTO_ERROR(H5E_CACHE, H5E_CANTINSERT, FAIL, "can't insert object into container");

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

herr_t
H5FO_delete(H5F_t *f, haddr_t addr)
{
    H5FO_open_obj_t *open_obj;            
    herr_t           ret_value = SUCCEED; 

    FUNC_ENTER_NOAPI(FAIL)

    
    assert(f);
    assert(f->shared);
    assert(f->shared->open_objs);
    assert(H5_addr_defined(addr));

    
    if (NULL == (open_obj = (H5FO_open_obj_t *)H5SL_remove(f->shared->open_objs, &addr)))
        HGOTO_ERROR(H5E_CACHE, H5E_CANTRELEASE, FAIL, "can't remove object from container");

    
    if (open_obj->deleted) {
        if (H5O_delete(f, addr) < 0)
            HGOTO_ERROR(H5E_OHDR, H5E_CANTDELETE, FAIL, "can't delete object from file");
    } 

    
    open_obj = H5FL_FREE(H5FO_open_obj_t, open_obj);

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

herr_t
H5FO_mark(const H5F_t *f, haddr_t addr, bool deleted)
{
    H5FO_open_obj_t *open_obj;            
    herr_t           ret_value = SUCCEED; 

    FUNC_ENTER_NOAPI_NOERR

    
    assert(f);
    assert(f->shared);
    assert(f->shared->open_objs);
    assert(H5_addr_defined(addr));

    
    if (NULL != (open_obj = (H5FO_open_obj_t *)H5SL_search(f->shared->open_objs, &addr)))
        open_obj->deleted = deleted;
    else
        ret_value = FAIL;

    FUNC_LEAVE_NOAPI(ret_value)
} 

bool
H5FO_marked(const H5F_t *f, haddr_t addr)
{
    H5FO_open_obj_t *open_obj;          
    bool             ret_value = false; 

    FUNC_ENTER_NOAPI_NOERR

    
    assert(f);
    assert(f->shared);
    assert(f->shared->open_objs);
    assert(H5_addr_defined(addr));

    
    if (NULL != (open_obj = (H5FO_open_obj_t *)H5SL_search(f->shared->open_objs, &addr)))
        ret_value = open_obj->deleted;

    FUNC_LEAVE_NOAPI(ret_value)
} 

herr_t
H5FO_dest(const H5F_t *f)
{
    herr_t ret_value = SUCCEED; 

    FUNC_ENTER_NOAPI(FAIL)

    
    assert(f);
    assert(f->shared);
    assert(f->shared->open_objs);

    
    if (H5SL_count(f->shared->open_objs) != 0)
        HGOTO_ERROR(H5E_CACHE, H5E_CANTRELEASE, FAIL, "objects still in open object info set");

    
    if (H5SL_close(f->shared->open_objs) < 0)
        HGOTO_ERROR(H5E_CACHE, H5E_CANTCLOSEOBJ, FAIL, "can't close open object info set");

    f->shared->open_objs = NULL;

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

herr_t
H5FO_top_create(H5F_t *f)
{
    herr_t ret_value = SUCCEED; 

    FUNC_ENTER_NOAPI(FAIL)

    
    assert(f);

    
    if ((f->obj_count = H5SL_create(H5SL_TYPE_HADDR, NULL)) == NULL)
        HGOTO_ERROR(H5E_FILE, H5E_CANTINIT, FAIL, "unable to create open object container");

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

herr_t
H5FO_top_incr(const H5F_t *f, haddr_t addr)
{
    H5FO_obj_count_t *obj_count;           
    herr_t            ret_value = SUCCEED; 

    FUNC_ENTER_NOAPI(FAIL)

    
    assert(f);
    assert(f->obj_count);
    assert(H5_addr_defined(addr));

    
    if (NULL != (obj_count = (H5FO_obj_count_t *)H5SL_search(f->obj_count, &addr))) {
        (obj_count->count)++;
    } 
    else {
        
        if (NULL == (obj_count = H5FL_MALLOC(H5FO_obj_count_t)))
            HGOTO_ERROR(H5E_CACHE, H5E_NOSPACE, FAIL, "memory allocation failed");

        
        obj_count->addr  = addr;
        obj_count->count = 1;

        
        if (H5SL_insert(f->obj_count, &obj_count->addr, obj_count) < 0)
            HGOTO_ERROR(H5E_CACHE, H5E_CANTINSERT, FAIL, "can't insert object into container");
    } 

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

herr_t
H5FO_top_decr(const H5F_t *f, haddr_t addr)
{
    H5FO_obj_count_t *obj_count;           
    herr_t            ret_value = SUCCEED; 

    FUNC_ENTER_NOAPI(FAIL)

    
    assert(f);
    assert(f->obj_count);
    assert(H5_addr_defined(addr));

    
    if (NULL != (obj_count = (H5FO_obj_count_t *)H5SL_search(f->obj_count, &addr))) {
        
        (obj_count->count)--;

        if (obj_count->count == 0) {
            
            if (NULL == (obj_count = (H5FO_obj_count_t *)H5SL_remove(f->obj_count, &addr)))
                HGOTO_ERROR(H5E_CACHE, H5E_CANTRELEASE, FAIL, "can't remove object from container");

            
            obj_count = H5FL_FREE(H5FO_obj_count_t, obj_count);
        } 
    }     
    else
        HGOTO_ERROR(H5E_CACHE, H5E_NOTFOUND, FAIL, "can't decrement ref. count");

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

hsize_t
H5FO_top_count(const H5F_t *f, haddr_t addr)
{
    H5FO_obj_count_t *obj_count; 
    hsize_t           ret_value; 

    FUNC_ENTER_NOAPI_NOINIT_NOERR

    
    assert(f);
    assert(f->obj_count);
    assert(H5_addr_defined(addr));

    
    if (NULL != (obj_count = (H5FO_obj_count_t *)H5SL_search(f->obj_count, &addr)))
        ret_value = obj_count->count;
    else
        ret_value = 0;

    FUNC_LEAVE_NOAPI(ret_value)
} 

herr_t
H5FO_top_dest(H5F_t *f)
{
    herr_t ret_value = SUCCEED; 

    FUNC_ENTER_NOAPI(FAIL)

    
    assert(f);
    assert(f->obj_count);

    
    if (H5SL_count(f->obj_count) != 0)
        HGOTO_ERROR(H5E_CACHE, H5E_CANTRELEASE, FAIL, "objects still in open object info set");

    
    if (H5SL_close(f->obj_count) < 0)
        HGOTO_ERROR(H5E_CACHE, H5E_CANTCLOSEOBJ, FAIL, "can't close open object info set");

    f->obj_count = NULL;

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 
