/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 * Copyright by The HDF Group.                                               *
 * All rights reserved.                                                      *
 *                                                                           *
 * This file is part of HDF5.  The full HDF5 copyright notice, including     *
 * terms governing use, modification, and redistribution, is contained in    *
 * the LICENSE file, which can be found at the root of the source code       *
 * distribution tree, or in https://www.hdfgroup.org/licenses.               *
 * If you do not have access to either file, you may request a copy from     *
 * help@hdfgroup.org.                                                        *
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

#include <assert.h>
#include <stdarg.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>

#include "hdf5.h"

#include "H5VLpassthru.h"

#define R_NO_REMAP 1
#include <R_ext/Print.h>

typedef struct H5VL_pass_through_t {
    hid_t under_vol_id; 
    void *under_object; 
} H5VL_pass_through_t;

typedef struct H5VL_pass_through_wrap_ctx_t {
    hid_t under_vol_id;   
    void *under_wrap_ctx; 
} H5VL_pass_through_wrap_ctx_t;

static H5VL_pass_through_t *H5VL_pass_through_new_obj(void *under_obj, hid_t under_vol_id);
static herr_t               H5VL_pass_through_free_obj(H5VL_pass_through_t *obj);

static herr_t H5VL_pass_through_init(hid_t vipl_id);
static herr_t H5VL_pass_through_term(void);

static void  *H5VL_pass_through_info_copy(const void *info);
static herr_t H5VL_pass_through_info_cmp(int *cmp_value, const void *info1, const void *info2);
static herr_t H5VL_pass_through_info_free(void *info);
static herr_t H5VL_pass_through_info_to_str(const void *info, char **str);
static herr_t H5VL_pass_through_str_to_info(const char *str, void **info);

static void  *H5VL_pass_through_get_object(const void *obj);
static herr_t H5VL_pass_through_get_wrap_ctx(const void *obj, void **wrap_ctx);
static void  *H5VL_pass_through_wrap_object(void *obj, H5I_type_t obj_type, void *wrap_ctx);
static void  *H5VL_pass_through_unwrap_object(void *obj);
static herr_t H5VL_pass_through_free_wrap_ctx(void *obj);

static void  *H5VL_pass_through_attr_create(void *obj, const H5VL_loc_params_t *loc_params, const char *name,
                                            hid_t type_id, hid_t space_id, hid_t acpl_id, hid_t aapl_id,
                                            hid_t dxpl_id, void **req);
static void  *H5VL_pass_through_attr_open(void *obj, const H5VL_loc_params_t *loc_params, const char *name,
                                          hid_t aapl_id, hid_t dxpl_id, void **req);
static herr_t H5VL_pass_through_attr_read(void *attr, hid_t mem_type_id, void *buf, hid_t dxpl_id,
                                          void **req);
static herr_t H5VL_pass_through_attr_write(void *attr, hid_t mem_type_id, const void *buf, hid_t dxpl_id,
                                           void **req);
static herr_t H5VL_pass_through_attr_get(void *obj, H5VL_attr_get_args_t *args, hid_t dxpl_id, void **req);
static herr_t H5VL_pass_through_attr_specific(void *obj, const H5VL_loc_params_t *loc_params,
                                              H5VL_attr_specific_args_t *args, hid_t dxpl_id, void **req);
static herr_t H5VL_pass_through_attr_optional(void *obj, H5VL_optional_args_t *args, hid_t dxpl_id,
                                              void **req);
static herr_t H5VL_pass_through_attr_close(void *attr, hid_t dxpl_id, void **req);

static void  *H5VL_pass_through_dataset_create(void *obj, const H5VL_loc_params_t *loc_params,
                                               const char *name, hid_t lcpl_id, hid_t type_id, hid_t space_id,
                                               hid_t dcpl_id, hid_t dapl_id, hid_t dxpl_id, void **req);
static void  *H5VL_pass_through_dataset_open(void *obj, const H5VL_loc_params_t *loc_params, const char *name,
                                             hid_t dapl_id, hid_t dxpl_id, void **req);
static herr_t H5VL_pass_through_dataset_read(size_t count, void *dset[], hid_t mem_type_id[],
                                             hid_t mem_space_id[], hid_t file_space_id[], hid_t plist_id,
                                             void *buf[], void **req);
static herr_t H5VL_pass_through_dataset_write(size_t count, void *dset[], hid_t mem_type_id[],
                                              hid_t mem_space_id[], hid_t file_space_id[], hid_t plist_id,
                                              const void *buf[], void **req);
static herr_t H5VL_pass_through_dataset_get(void *dset, H5VL_dataset_get_args_t *args, hid_t dxpl_id,
                                            void **req);
static herr_t H5VL_pass_through_dataset_specific(void *obj, H5VL_dataset_specific_args_t *args, hid_t dxpl_id,
                                                 void **req);
static herr_t H5VL_pass_through_dataset_optional(void *obj, H5VL_optional_args_t *args, hid_t dxpl_id,
                                                 void **req);
static herr_t H5VL_pass_through_dataset_close(void *dset, hid_t dxpl_id, void **req);

static void *H5VL_pass_through_datatype_commit(void *obj, const H5VL_loc_params_t *loc_params,
                                               const char *name, hid_t type_id, hid_t lcpl_id, hid_t tcpl_id,
                                               hid_t tapl_id, hid_t dxpl_id, void **req);
static void *H5VL_pass_through_datatype_open(void *obj, const H5VL_loc_params_t *loc_params, const char *name,
                                             hid_t tapl_id, hid_t dxpl_id, void **req);
static herr_t H5VL_pass_through_datatype_get(void *dt, H5VL_datatype_get_args_t *args, hid_t dxpl_id,
                                             void **req);
static herr_t H5VL_pass_through_datatype_specific(void *obj, H5VL_datatype_specific_args_t *args,
                                                  hid_t dxpl_id, void **req);
static herr_t H5VL_pass_through_datatype_optional(void *obj, H5VL_optional_args_t *args, hid_t dxpl_id,
                                                  void **req);
static herr_t H5VL_pass_through_datatype_close(void *dt, hid_t dxpl_id, void **req);

static void  *H5VL_pass_through_file_create(const char *name, unsigned flags, hid_t fcpl_id, hid_t fapl_id,
                                            hid_t dxpl_id, void **req);
static void  *H5VL_pass_through_file_open(const char *name, unsigned flags, hid_t fapl_id, hid_t dxpl_id,
                                          void **req);
static herr_t H5VL_pass_through_file_get(void *file, H5VL_file_get_args_t *args, hid_t dxpl_id, void **req);
static herr_t H5VL_pass_through_file_specific(void *file, H5VL_file_specific_args_t *args, hid_t dxpl_id,
                                              void **req);
static herr_t H5VL_pass_through_file_optional(void *file, H5VL_optional_args_t *args, hid_t dxpl_id,
                                              void **req);
static herr_t H5VL_pass_through_file_close(void *file, hid_t dxpl_id, void **req);

static void  *H5VL_pass_through_group_create(void *obj, const H5VL_loc_params_t *loc_params, const char *name,
                                             hid_t lcpl_id, hid_t gcpl_id, hid_t gapl_id, hid_t dxpl_id,
                                             void **req);
static void  *H5VL_pass_through_group_open(void *obj, const H5VL_loc_params_t *loc_params, const char *name,
                                           hid_t gapl_id, hid_t dxpl_id, void **req);
static herr_t H5VL_pass_through_group_get(void *obj, H5VL_group_get_args_t *args, hid_t dxpl_id, void **req);
static herr_t H5VL_pass_through_group_specific(void *obj, H5VL_group_specific_args_t *args, hid_t dxpl_id,
                                               void **req);
static herr_t H5VL_pass_through_group_optional(void *obj, H5VL_optional_args_t *args, hid_t dxpl_id,
                                               void **req);
static herr_t H5VL_pass_through_group_close(void *grp, hid_t dxpl_id, void **req);

static herr_t H5VL_pass_through_link_create(H5VL_link_create_args_t *args, void *obj,
                                            const H5VL_loc_params_t *loc_params, hid_t lcpl_id, hid_t lapl_id,
                                            hid_t dxpl_id, void **req);
static herr_t H5VL_pass_through_link_copy(void *src_obj, const H5VL_loc_params_t *loc_params1, void *dst_obj,
                                          const H5VL_loc_params_t *loc_params2, hid_t lcpl_id, hid_t lapl_id,
                                          hid_t dxpl_id, void **req);
static herr_t H5VL_pass_through_link_move(void *src_obj, const H5VL_loc_params_t *loc_params1, void *dst_obj,
                                          const H5VL_loc_params_t *loc_params2, hid_t lcpl_id, hid_t lapl_id,
                                          hid_t dxpl_id, void **req);
static herr_t H5VL_pass_through_link_get(void *obj, const H5VL_loc_params_t *loc_params,
                                         H5VL_link_get_args_t *args, hid_t dxpl_id, void **req);
static herr_t H5VL_pass_through_link_specific(void *obj, const H5VL_loc_params_t *loc_params,
                                              H5VL_link_specific_args_t *args, hid_t dxpl_id, void **req);
static herr_t H5VL_pass_through_link_optional(void *obj, const H5VL_loc_params_t *loc_params,
                                              H5VL_optional_args_t *args, hid_t dxpl_id, void **req);

static void  *H5VL_pass_through_object_open(void *obj, const H5VL_loc_params_t *loc_params,
                                            H5I_type_t *opened_type, hid_t dxpl_id, void **req);
static herr_t H5VL_pass_through_object_copy(void *src_obj, const H5VL_loc_params_t *src_loc_params,
                                            const char *src_name, void *dst_obj,
                                            const H5VL_loc_params_t *dst_loc_params, const char *dst_name,
                                            hid_t ocpypl_id, hid_t lcpl_id, hid_t dxpl_id, void **req);
static herr_t H5VL_pass_through_object_get(void *obj, const H5VL_loc_params_t *loc_params,
                                           H5VL_object_get_args_t *args, hid_t dxpl_id, void **req);
static herr_t H5VL_pass_through_object_specific(void *obj, const H5VL_loc_params_t *loc_params,
                                                H5VL_object_specific_args_t *args, hid_t dxpl_id, void **req);
static herr_t H5VL_pass_through_object_optional(void *obj, const H5VL_loc_params_t *loc_params,
                                                H5VL_optional_args_t *args, hid_t dxpl_id, void **req);

static herr_t H5VL_pass_through_introspect_get_conn_cls(void *obj, H5VL_get_conn_lvl_t lvl,
                                                        const H5VL_class_t **conn_cls);
static herr_t H5VL_pass_through_introspect_get_cap_flags(const void *info, uint64_t *cap_flags);
static herr_t H5VL_pass_through_introspect_opt_query(void *obj, H5VL_subclass_t cls, int opt_type,
                                                     uint64_t *flags);

static herr_t H5VL_pass_through_request_wait(void *req, uint64_t timeout, H5VL_request_status_t *status);
static herr_t H5VL_pass_through_request_notify(void *obj, H5VL_request_notify_t cb, void *ctx);
static herr_t H5VL_pass_through_request_cancel(void *req, H5VL_request_status_t *status);
static herr_t H5VL_pass_through_request_specific(void *req, H5VL_request_specific_args_t *args);
static herr_t H5VL_pass_through_request_optional(void *req, H5VL_optional_args_t *args);
static herr_t H5VL_pass_through_request_free(void *req);

static herr_t H5VL_pass_through_blob_put(void *obj, const void *buf, size_t size, void *blob_id, void *ctx);
static herr_t H5VL_pass_through_blob_get(void *obj, const void *blob_id, void *buf, size_t size, void *ctx);
static herr_t H5VL_pass_through_blob_specific(void *obj, void *blob_id, H5VL_blob_specific_args_t *args);
static herr_t H5VL_pass_through_blob_optional(void *obj, void *blob_id, H5VL_optional_args_t *args);

static herr_t H5VL_pass_through_token_cmp(void *obj, const H5O_token_t *token1, const H5O_token_t *token2,
                                          int *cmp_value);
static herr_t H5VL_pass_through_token_to_str(void *obj, H5I_type_t obj_type, const H5O_token_t *token,
                                             char **token_str);
static herr_t H5VL_pass_through_token_from_str(void *obj, H5I_type_t obj_type, const char *token_str,
                                               H5O_token_t *token);

static herr_t H5VL_pass_through_optional(void *obj, H5VL_optional_args_t *args, hid_t dxpl_id, void **req);

const H5VL_class_t H5VL_pass_through_g = {
    H5VL_VERSION,                            
    (H5VL_class_value_t)H5VL_PASSTHRU_VALUE, 
    H5VL_PASSTHRU_NAME,                      
    H5VL_PASSTHRU_VERSION,                   
    0,                                       
    H5VL_pass_through_init,                  
    H5VL_pass_through_term,                  
    {
        
        sizeof(H5VL_pass_through_info_t), 
        H5VL_pass_through_info_copy,      
        H5VL_pass_through_info_cmp,       
        H5VL_pass_through_info_free,      
        H5VL_pass_through_info_to_str,    
        H5VL_pass_through_str_to_info     
    },
    {
        
        H5VL_pass_through_get_object,    
        H5VL_pass_through_get_wrap_ctx,  
        H5VL_pass_through_wrap_object,   
        H5VL_pass_through_unwrap_object, 
        H5VL_pass_through_free_wrap_ctx  
    },
    {
        
        H5VL_pass_through_attr_create,   
        H5VL_pass_through_attr_open,     
        H5VL_pass_through_attr_read,     
        H5VL_pass_through_attr_write,    
        H5VL_pass_through_attr_get,      
        H5VL_pass_through_attr_specific, 
        H5VL_pass_through_attr_optional, 
        H5VL_pass_through_attr_close     
    },
    {
        
        H5VL_pass_through_dataset_create,   
        H5VL_pass_through_dataset_open,     
        H5VL_pass_through_dataset_read,     
        H5VL_pass_through_dataset_write,    
        H5VL_pass_through_dataset_get,      
        H5VL_pass_through_dataset_specific, 
        H5VL_pass_through_dataset_optional, 
        H5VL_pass_through_dataset_close     
    },
    {
        
        H5VL_pass_through_datatype_commit,   
        H5VL_pass_through_datatype_open,     
        H5VL_pass_through_datatype_get,      
        H5VL_pass_through_datatype_specific, 
        H5VL_pass_through_datatype_optional, 
        H5VL_pass_through_datatype_close     
    },
    {
        
        H5VL_pass_through_file_create,   
        H5VL_pass_through_file_open,     
        H5VL_pass_through_file_get,      
        H5VL_pass_through_file_specific, 
        H5VL_pass_through_file_optional, 
        H5VL_pass_through_file_close     
    },
    {
        
        H5VL_pass_through_group_create,   
        H5VL_pass_through_group_open,     
        H5VL_pass_through_group_get,      
        H5VL_pass_through_group_specific, 
        H5VL_pass_through_group_optional, 
        H5VL_pass_through_group_close     
    },
    {
        
        H5VL_pass_through_link_create,   
        H5VL_pass_through_link_copy,     
        H5VL_pass_through_link_move,     
        H5VL_pass_through_link_get,      
        H5VL_pass_through_link_specific, 
        H5VL_pass_through_link_optional  
    },
    {
        
        H5VL_pass_through_object_open,     
        H5VL_pass_through_object_copy,     
        H5VL_pass_through_object_get,      
        H5VL_pass_through_object_specific, 
        H5VL_pass_through_object_optional  
    },
    {
        
        H5VL_pass_through_introspect_get_conn_cls,  
        H5VL_pass_through_introspect_get_cap_flags, 
        H5VL_pass_through_introspect_opt_query,     
    },
    {
        
        H5VL_pass_through_request_wait,     
        H5VL_pass_through_request_notify,   
        H5VL_pass_through_request_cancel,   
        H5VL_pass_through_request_specific, 
        H5VL_pass_through_request_optional, 
        H5VL_pass_through_request_free      
    },
    {
        
        H5VL_pass_through_blob_put,      
        H5VL_pass_through_blob_get,      
        H5VL_pass_through_blob_specific, 
        H5VL_pass_through_blob_optional  
    },
    {
        
        H5VL_pass_through_token_cmp,     
        H5VL_pass_through_token_to_str,  
        H5VL_pass_through_token_from_str 
    },
    H5VL_pass_through_optional 
};

static H5VL_pass_through_t *
H5VL_pass_through_new_obj(void *under_obj, hid_t under_vol_id)
{
    H5VL_pass_through_t *new_obj;

    new_obj               = (H5VL_pass_through_t *)calloc(1, sizeof(H5VL_pass_through_t));
    new_obj->under_object = under_obj;
    new_obj->under_vol_id = under_vol_id;

    H5Iinc_ref(new_obj->under_vol_id);

    return new_obj;
} 

static herr_t
H5VL_pass_through_free_obj(H5VL_pass_through_t *obj)
{
    hid_t err_id;

    err_id = H5Eget_current_stack();

    H5Idec_ref(obj->under_vol_id);

    H5Eset_current_stack(err_id);

    free(obj);

    return 0;
} 

static herr_t
H5VL_pass_through_init(hid_t vipl_id)
{
#ifdef ENABLE_PASSTHRU_LOGGING
    Rprintf("------- PASS THROUGH VOL INIT\n");
#endif

    
    (void)vipl_id;

    return 0;
} 

static herr_t
H5VL_pass_through_term(void)
{
#ifdef ENABLE_PASSTHRU_LOGGING
    Rprintf("------- PASS THROUGH VOL TERM\n");
#endif

    return 0;
} 

static void *
H5VL_pass_through_info_copy(const void *_info)
{
    const H5VL_pass_through_info_t *info = (const H5VL_pass_through_info_t *)_info;
    H5VL_pass_through_info_t       *new_info;

#ifdef ENABLE_PASSTHRU_LOGGING
    Rprintf("------- PASS THROUGH VOL INFO Copy\n");
#endif

    
    if (!info) {
        Rprintf("\nH5VLpassthru.c line %d in %s: info for pass-through VOL can't be null\n", __LINE__,
               __func__);
        return NULL;
    }

    if (H5Iis_valid(info->under_vol_id) <= 0) {
        Rprintf("\nH5VLpassthru.c line %d in %s: not a valid underneath VOL ID for pass-through VOL\n",
               __LINE__, __func__);
        return NULL;
    }

    
    new_info = (H5VL_pass_through_info_t *)calloc(1, sizeof(H5VL_pass_through_info_t));

    
    new_info->under_vol_id = info->under_vol_id;

    H5Iinc_ref(new_info->under_vol_id);

    if (info->under_vol_info)
        H5VLcopy_connector_info(new_info->under_vol_id, &(new_info->under_vol_info), info->under_vol_info);

    return new_info;
} 

static herr_t
H5VL_pass_through_info_cmp(int *cmp_value, const void *_info1, const void *_info2)
{
    const H5VL_pass_through_info_t *info1 = (const H5VL_pass_through_info_t *)_info1;
    const H5VL_pass_through_info_t *info2 = (const H5VL_pass_through_info_t *)_info2;

#ifdef ENABLE_PASSTHRU_LOGGING
    Rprintf("------- PASS THROUGH VOL INFO Compare\n");
#endif

    
    assert(info1);
    assert(info2);

    
    *cmp_value = 0;

    
    H5VLcmp_connector_cls(cmp_value, info1->under_vol_id, info2->under_vol_id);
    if (*cmp_value != 0)
        return 0;

    
    H5VLcmp_connector_info(cmp_value, info1->under_vol_id, info1->under_vol_info, info2->under_vol_info);
    if (*cmp_value != 0)
        return 0;

    return 0;
} 

static herr_t
H5VL_pass_through_info_free(void *_info)
{
    H5VL_pass_through_info_t *info = (H5VL_pass_through_info_t *)_info;
    hid_t                     err_id;

#ifdef ENABLE_PASSTHRU_LOGGING
    Rprintf("------- PASS THROUGH VOL INFO Free\n");
#endif

    err_id = H5Eget_current_stack();

    
    if (info->under_vol_info)
        H5VLfree_connector_info(info->under_vol_id, info->under_vol_info);
    H5Idec_ref(info->under_vol_id);

    H5Eset_current_stack(err_id);

    
    free(info);

    return 0;
} 

static herr_t
H5VL_pass_through_info_to_str(const void *_info, char **str)
{
    const H5VL_pass_through_info_t *info              = (const H5VL_pass_through_info_t *)_info;
    H5VL_class_value_t              under_value       = (H5VL_class_value_t)-1;
    char                           *under_vol_string  = NULL;
    size_t                          under_vol_str_len = 0;

#ifdef ENABLE_PASSTHRU_LOGGING
    Rprintf("------- PASS THROUGH VOL INFO To String\n");
#endif

    
    H5VLget_value(info->under_vol_id, &under_value);
    H5VLconnector_info_to_str(info->under_vol_info, info->under_vol_id, &under_vol_string);

    
    if (under_vol_string)
        under_vol_str_len = strlen(under_vol_string);

    
    size_t strSize = 32 + under_vol_str_len;
    *str           = (char *)H5allocate_memory(strSize, (bool)0);
    assert(*str);

    
    snprintf(*str, strSize, "under_vol=%u;under_info={%s}", (unsigned)under_value,
             (under_vol_string ? under_vol_string : ""));

    return 0;
} 

static herr_t
H5VL_pass_through_str_to_info(const char *str, void **_info)
{
    H5VL_pass_through_info_t *info;
    unsigned                  under_vol_value;
    const char               *under_vol_info_start, *under_vol_info_end;
    hid_t                     under_vol_id;
    void                     *under_vol_info = NULL;

#ifdef ENABLE_PASSTHRU_LOGGING
    Rprintf("------- PASS THROUGH VOL INFO String To Info\n");
#endif

    
    sscanf(str, "under_vol=%u;", &under_vol_value);
    under_vol_id         = H5VLregister_connector_by_value((H5VL_class_value_t)under_vol_value, H5P_DEFAULT);
    under_vol_info_start = strchr(str, '{');
    under_vol_info_end   = strrchr(str, '}');
    assert(under_vol_info_end > under_vol_info_start);
    if (under_vol_info_end != (under_vol_info_start + 1)) {
        char *under_vol_info_str;

        under_vol_info_str = (char *)malloc((size_t)(under_vol_info_end - under_vol_info_start));
        memcpy(under_vol_info_str, under_vol_info_start + 1,
               (size_t)((under_vol_info_end - under_vol_info_start) - 1));
        *(under_vol_info_str + (under_vol_info_end - under_vol_info_start)) = '\0';

        H5VLconnector_str_to_info(under_vol_info_str, under_vol_id, &under_vol_info);

        free(under_vol_info_str);
    } 

    
    info                 = (H5VL_pass_through_info_t *)calloc(1, sizeof(H5VL_pass_through_info_t));
    info->under_vol_id   = under_vol_id;
    info->under_vol_info = under_vol_info;

    
    *_info = info;

    return 0;
} 

static void *
H5VL_pass_through_get_object(const void *obj)
{
    const H5VL_pass_through_t *o = (const H5VL_pass_through_t *)obj;

#ifdef ENABLE_PASSTHRU_LOGGING
    Rprintf("------- PASS THROUGH VOL Get object\n");
#endif

    return H5VLget_object(o->under_object, o->under_vol_id);
} 

static herr_t
H5VL_pass_through_get_wrap_ctx(const void *obj, void **wrap_ctx)
{
    const H5VL_pass_through_t    *o = (const H5VL_pass_through_t *)obj;
    H5VL_pass_through_wrap_ctx_t *new_wrap_ctx;

#ifdef ENABLE_PASSTHRU_LOGGING
    Rprintf("------- PASS THROUGH VOL WRAP CTX Get\n");
#endif

    
    new_wrap_ctx = (H5VL_pass_through_wrap_ctx_t *)calloc(1, sizeof(H5VL_pass_through_wrap_ctx_t));

    
    new_wrap_ctx->under_vol_id = o->under_vol_id;

    H5Iinc_ref(new_wrap_ctx->under_vol_id);

    H5VLget_wrap_ctx(o->under_object, o->under_vol_id, &new_wrap_ctx->under_wrap_ctx);

    
    *wrap_ctx = new_wrap_ctx;

    return 0;
} 

static void *
H5VL_pass_through_wrap_object(void *obj, H5I_type_t obj_type, void *_wrap_ctx)
{
    H5VL_pass_through_wrap_ctx_t *wrap_ctx = (H5VL_pass_through_wrap_ctx_t *)_wrap_ctx;
    H5VL_pass_through_t          *new_obj;
    void                         *under;

#ifdef ENABLE_PASSTHRU_LOGGING
    Rprintf("------- PASS THROUGH VOL WRAP Object\n");
#endif

    
    under = H5VLwrap_object(obj, obj_type, wrap_ctx->under_vol_id, wrap_ctx->under_wrap_ctx);
    if (under)
        new_obj = H5VL_pass_through_new_obj(under, wrap_ctx->under_vol_id);
    else
        new_obj = NULL;

    return new_obj;
} 

static void *
H5VL_pass_through_unwrap_object(void *obj)
{
    H5VL_pass_through_t *o = (H5VL_pass_through_t *)obj;
    void                *under;

#ifdef ENABLE_PASSTHRU_LOGGING
    Rprintf("------- PASS THROUGH VOL UNWRAP Object\n");
#endif

    
    under = H5VLunwrap_object(o->under_object, o->under_vol_id);

    if (under)
        H5VL_pass_through_free_obj(o);

    return under;
} 

static herr_t
H5VL_pass_through_free_wrap_ctx(void *_wrap_ctx)
{
    H5VL_pass_through_wrap_ctx_t *wrap_ctx = (H5VL_pass_through_wrap_ctx_t *)_wrap_ctx;
    hid_t                         err_id;

#ifdef ENABLE_PASSTHRU_LOGGING
    Rprintf("------- PASS THROUGH VOL WRAP CTX Free\n");
#endif

    err_id = H5Eget_current_stack();

    
    if (wrap_ctx->under_wrap_ctx)
        H5VLfree_wrap_ctx(wrap_ctx->under_wrap_ctx, wrap_ctx->under_vol_id);
    H5Idec_ref(wrap_ctx->under_vol_id);

    H5Eset_current_stack(err_id);

    
    free(wrap_ctx);

    return 0;
} 

static void *
H5VL_pass_through_attr_create(void *obj, const H5VL_loc_params_t *loc_params, const char *name, hid_t type_id,
                              hid_t space_id, hid_t acpl_id, hid_t aapl_id, hid_t dxpl_id, void **req)
{
    H5VL_pass_through_t *attr;
    H5VL_pass_through_t *o = (H5VL_pass_through_t *)obj;
    void                *under;

#ifdef ENABLE_PASSTHRU_LOGGING
    Rprintf("------- PASS THROUGH VOL ATTRIBUTE Create\n");
#endif

    under = H5VLattr_create(o->under_object, loc_params, o->under_vol_id, name, type_id, space_id, acpl_id,
                            aapl_id, dxpl_id, req);
    if (under) {
        attr = H5VL_pass_through_new_obj(under, o->under_vol_id);

        
        if (req && *req)
            *req = H5VL_pass_through_new_obj(*req, o->under_vol_id);
    } 
    else
        attr = NULL;

    return (void *)attr;
} 

static void *
H5VL_pass_through_attr_open(void *obj, const H5VL_loc_params_t *loc_params, const char *name, hid_t aapl_id,
                            hid_t dxpl_id, void **req)
{
    H5VL_pass_through_t *attr;
    H5VL_pass_through_t *o = (H5VL_pass_through_t *)obj;
    void                *under;

#ifdef ENABLE_PASSTHRU_LOGGING
    Rprintf("------- PASS THROUGH VOL ATTRIBUTE Open\n");
#endif

    under = H5VLattr_open(o->under_object, loc_params, o->under_vol_id, name, aapl_id, dxpl_id, req);
    if (under) {
        attr = H5VL_pass_through_new_obj(under, o->under_vol_id);

        
        if (req && *req)
            *req = H5VL_pass_through_new_obj(*req, o->under_vol_id);
    } 
    else
        attr = NULL;

    return (void *)attr;
} 

static herr_t
H5VL_pass_through_attr_read(void *attr, hid_t mem_type_id, void *buf, hid_t dxpl_id, void **req)
{
    H5VL_pass_through_t *o = (H5VL_pass_through_t *)attr;
    herr_t               ret_value;

#ifdef ENABLE_PASSTHRU_LOGGING
    Rprintf("------- PASS THROUGH VOL ATTRIBUTE Read\n");
#endif

    ret_value = H5VLattr_read(o->under_object, o->under_vol_id, mem_type_id, buf, dxpl_id, req);

    
    if (req && *req)
        *req = H5VL_pass_through_new_obj(*req, o->under_vol_id);

    return ret_value;
} 

static herr_t
H5VL_pass_through_attr_write(void *attr, hid_t mem_type_id, const void *buf, hid_t dxpl_id, void **req)
{
    H5VL_pass_through_t *o = (H5VL_pass_through_t *)attr;
    herr_t               ret_value;

#ifdef ENABLE_PASSTHRU_LOGGING
    Rprintf("------- PASS THROUGH VOL ATTRIBUTE Write\n");
#endif

    ret_value = H5VLattr_write(o->under_object, o->under_vol_id, mem_type_id, buf, dxpl_id, req);

    
    if (req && *req)
        *req = H5VL_pass_through_new_obj(*req, o->under_vol_id);

    return ret_value;
} 

static herr_t
H5VL_pass_through_attr_get(void *obj, H5VL_attr_get_args_t *args, hid_t dxpl_id, void **req)
{
    H5VL_pass_through_t *o = (H5VL_pass_through_t *)obj;
    herr_t               ret_value;

#ifdef ENABLE_PASSTHRU_LOGGING
    Rprintf("------- PASS THROUGH VOL ATTRIBUTE Get\n");
#endif

    ret_value = H5VLattr_get(o->under_object, o->under_vol_id, args, dxpl_id, req);

    
    if (req && *req)
        *req = H5VL_pass_through_new_obj(*req, o->under_vol_id);

    return ret_value;
} 

static herr_t
H5VL_pass_through_attr_specific(void *obj, const H5VL_loc_params_t *loc_params,
                                H5VL_attr_specific_args_t *args, hid_t dxpl_id, void **req)
{
    H5VL_pass_through_t *o = (H5VL_pass_through_t *)obj;
    herr_t               ret_value;

#ifdef ENABLE_PASSTHRU_LOGGING
    Rprintf("------- PASS THROUGH VOL ATTRIBUTE Specific\n");
#endif

    ret_value = H5VLattr_specific(o->under_object, loc_params, o->under_vol_id, args, dxpl_id, req);

    
    if (req && *req)
        *req = H5VL_pass_through_new_obj(*req, o->under_vol_id);

    return ret_value;
} 

static herr_t
H5VL_pass_through_attr_optional(void *obj, H5VL_optional_args_t *args, hid_t dxpl_id, void **req)
{
    H5VL_pass_through_t *o = (H5VL_pass_through_t *)obj;
    herr_t               ret_value;

#ifdef ENABLE_PASSTHRU_LOGGING
    Rprintf("------- PASS THROUGH VOL ATTRIBUTE Optional\n");
#endif

    ret_value = H5VLattr_optional(o->under_object, o->under_vol_id, args, dxpl_id, req);

    
    if (req && *req)
        *req = H5VL_pass_through_new_obj(*req, o->under_vol_id);

    return ret_value;
} 

static herr_t
H5VL_pass_through_attr_close(void *attr, hid_t dxpl_id, void **req)
{
    H5VL_pass_through_t *o = (H5VL_pass_through_t *)attr;
    herr_t               ret_value;

#ifdef ENABLE_PASSTHRU_LOGGING
    Rprintf("------- PASS THROUGH VOL ATTRIBUTE Close\n");
#endif

    ret_value = H5VLattr_close(o->under_object, o->under_vol_id, dxpl_id, req);

    
    if (req && *req)
        *req = H5VL_pass_through_new_obj(*req, o->under_vol_id);

    
    if (ret_value >= 0)
        H5VL_pass_through_free_obj(o);

    return ret_value;
} 

static void *
H5VL_pass_through_dataset_create(void *obj, const H5VL_loc_params_t *loc_params, const char *name,
                                 hid_t lcpl_id, hid_t type_id, hid_t space_id, hid_t dcpl_id, hid_t dapl_id,
                                 hid_t dxpl_id, void **req)
{
    H5VL_pass_through_t *dset;
    H5VL_pass_through_t *o = (H5VL_pass_through_t *)obj;
    void                *under;

#ifdef ENABLE_PASSTHRU_LOGGING
    Rprintf("------- PASS THROUGH VOL DATASET Create\n");
#endif

    under = H5VLdataset_create(o->under_object, loc_params, o->under_vol_id, name, lcpl_id, type_id, space_id,
                               dcpl_id, dapl_id, dxpl_id, req);
    if (under) {
        dset = H5VL_pass_through_new_obj(under, o->under_vol_id);

        
        if (req && *req)
            *req = H5VL_pass_through_new_obj(*req, o->under_vol_id);
    } 
    else
        dset = NULL;

    return (void *)dset;
} 

static void *
H5VL_pass_through_dataset_open(void *obj, const H5VL_loc_params_t *loc_params, const char *name,
                               hid_t dapl_id, hid_t dxpl_id, void **req)
{
    H5VL_pass_through_t *dset;
    H5VL_pass_through_t *o = (H5VL_pass_through_t *)obj;
    void                *under;

#ifdef ENABLE_PASSTHRU_LOGGING
    Rprintf("------- PASS THROUGH VOL DATASET Open\n");
#endif

    under = H5VLdataset_open(o->under_object, loc_params, o->under_vol_id, name, dapl_id, dxpl_id, req);
    if (under) {
        dset = H5VL_pass_through_new_obj(under, o->under_vol_id);

        
        if (req && *req)
            *req = H5VL_pass_through_new_obj(*req, o->under_vol_id);
    } 
    else
        dset = NULL;

    return (void *)dset;
} 

static herr_t
H5VL_pass_through_dataset_read(size_t count, void *dset[], hid_t mem_type_id[], hid_t mem_space_id[],
                               hid_t file_space_id[], hid_t plist_id, void *buf[], void **req)
{
    void  *obj_local;        
    void **obj = &obj_local; 
    size_t i;                
    herr_t ret_value;

#ifdef ENABLE_PASSTHRU_LOGGING
    Rprintf("------- PASS THROUGH VOL DATASET Read\n");
#endif

    
    if (count > 1)
        if (NULL == (obj = (void **)malloc(count * sizeof(void *))))
            return -1;

    
    for (i = 0; i < count; i++) {
        
        obj[i] = ((H5VL_pass_through_t *)dset[i])->under_object;

        
        if (((H5VL_pass_through_t *)dset[i])->under_vol_id != ((H5VL_pass_through_t *)dset[0])->under_vol_id)
            return -1;
    }

    ret_value = H5VLdataset_read(count, obj, ((H5VL_pass_through_t *)dset[0])->under_vol_id, mem_type_id,
                                 mem_space_id, file_space_id, plist_id, buf, req);

    
    if (req && *req)
        *req = H5VL_pass_through_new_obj(*req, ((H5VL_pass_through_t *)dset[0])->under_vol_id);

    
    if (obj != &obj_local)
        free(obj);

    return ret_value;
} 

static herr_t
H5VL_pass_through_dataset_write(size_t count, void *dset[], hid_t mem_type_id[], hid_t mem_space_id[],
                                hid_t file_space_id[], hid_t plist_id, const void *buf[], void **req)
{
    void  *obj_local;        
    void **obj = &obj_local; 
    size_t i;                
    herr_t ret_value;

#ifdef ENABLE_PASSTHRU_LOGGING
    Rprintf("------- PASS THROUGH VOL DATASET Write\n");
#endif

    
    if (count > 1)
        if (NULL == (obj = (void **)malloc(count * sizeof(void *))))
            return -1;

    
    for (i = 0; i < count; i++) {
        
        obj[i] = ((H5VL_pass_through_t *)dset[i])->under_object;

        
        if (((H5VL_pass_through_t *)dset[i])->under_vol_id != ((H5VL_pass_through_t *)dset[0])->under_vol_id)
            return -1;
    }

    ret_value = H5VLdataset_write(count, obj, ((H5VL_pass_through_t *)dset[0])->under_vol_id, mem_type_id,
                                  mem_space_id, file_space_id, plist_id, buf, req);

    
    if (req && *req)
        *req = H5VL_pass_through_new_obj(*req, ((H5VL_pass_through_t *)dset[0])->under_vol_id);

    
    if (obj != &obj_local)
        free(obj);

    return ret_value;
} 

static herr_t
H5VL_pass_through_dataset_get(void *dset, H5VL_dataset_get_args_t *args, hid_t dxpl_id, void **req)
{
    H5VL_pass_through_t *o = (H5VL_pass_through_t *)dset;
    herr_t               ret_value;

#ifdef ENABLE_PASSTHRU_LOGGING
    Rprintf("------- PASS THROUGH VOL DATASET Get\n");
#endif

    ret_value = H5VLdataset_get(o->under_object, o->under_vol_id, args, dxpl_id, req);

    
    if (req && *req)
        *req = H5VL_pass_through_new_obj(*req, o->under_vol_id);

    return ret_value;
} 

static herr_t
H5VL_pass_through_dataset_specific(void *obj, H5VL_dataset_specific_args_t *args, hid_t dxpl_id, void **req)
{
    H5VL_pass_through_t *o = (H5VL_pass_through_t *)obj;
    hid_t                under_vol_id;
    herr_t               ret_value;

#ifdef ENABLE_PASSTHRU_LOGGING
    Rprintf("------- PASS THROUGH VOL H5Dspecific\n");
#endif

    
    under_vol_id = o->under_vol_id;

    ret_value = H5VLdataset_specific(o->under_object, o->under_vol_id, args, dxpl_id, req);

    
    if (req && *req)
        *req = H5VL_pass_through_new_obj(*req, under_vol_id);

    return ret_value;
} 

static herr_t
H5VL_pass_through_dataset_optional(void *obj, H5VL_optional_args_t *args, hid_t dxpl_id, void **req)
{
    H5VL_pass_through_t *o = (H5VL_pass_through_t *)obj;
    herr_t               ret_value;

#ifdef ENABLE_PASSTHRU_LOGGING
    Rprintf("------- PASS THROUGH VOL DATASET Optional\n");
#endif

    ret_value = H5VLdataset_optional(o->under_object, o->under_vol_id, args, dxpl_id, req);

    
    if (req && *req)
        *req = H5VL_pass_through_new_obj(*req, o->under_vol_id);

    return ret_value;
} 

static herr_t
H5VL_pass_through_dataset_close(void *dset, hid_t dxpl_id, void **req)
{
    H5VL_pass_through_t *o = (H5VL_pass_through_t *)dset;
    herr_t               ret_value;

#ifdef ENABLE_PASSTHRU_LOGGING
    Rprintf("------- PASS THROUGH VOL DATASET Close\n");
#endif

    ret_value = H5VLdataset_close(o->under_object, o->under_vol_id, dxpl_id, req);

    
    if (req && *req)
        *req = H5VL_pass_through_new_obj(*req, o->under_vol_id);

    
    if (ret_value >= 0)
        H5VL_pass_through_free_obj(o);

    return ret_value;
} 

static void *
H5VL_pass_through_datatype_commit(void *obj, const H5VL_loc_params_t *loc_params, const char *name,
                                  hid_t type_id, hid_t lcpl_id, hid_t tcpl_id, hid_t tapl_id, hid_t dxpl_id,
                                  void **req)
{
    H5VL_pass_through_t *dt;
    H5VL_pass_through_t *o = (H5VL_pass_through_t *)obj;
    void                *under;

#ifdef ENABLE_PASSTHRU_LOGGING
    Rprintf("------- PASS THROUGH VOL DATATYPE Commit\n");
#endif

    under = H5VLdatatype_commit(o->under_object, loc_params, o->under_vol_id, name, type_id, lcpl_id, tcpl_id,
                                tapl_id, dxpl_id, req);
    if (under) {
        dt = H5VL_pass_through_new_obj(under, o->under_vol_id);

        
        if (req && *req)
            *req = H5VL_pass_through_new_obj(*req, o->under_vol_id);
    } 
    else
        dt = NULL;

    return (void *)dt;
} 

static void *
H5VL_pass_through_datatype_open(void *obj, const H5VL_loc_params_t *loc_params, const char *name,
                                hid_t tapl_id, hid_t dxpl_id, void **req)
{
    H5VL_pass_through_t *dt;
    H5VL_pass_through_t *o = (H5VL_pass_through_t *)obj;
    void                *under;

#ifdef ENABLE_PASSTHRU_LOGGING
    Rprintf("------- PASS THROUGH VOL DATATYPE Open\n");
#endif

    under = H5VLdatatype_open(o->under_object, loc_params, o->under_vol_id, name, tapl_id, dxpl_id, req);
    if (under) {
        dt = H5VL_pass_through_new_obj(under, o->under_vol_id);

        
        if (req && *req)
            *req = H5VL_pass_through_new_obj(*req, o->under_vol_id);
    } 
    else
        dt = NULL;

    return (void *)dt;
} 

static herr_t
H5VL_pass_through_datatype_get(void *dt, H5VL_datatype_get_args_t *args, hid_t dxpl_id, void **req)
{
    H5VL_pass_through_t *o = (H5VL_pass_through_t *)dt;
    herr_t               ret_value;

#ifdef ENABLE_PASSTHRU_LOGGING
    Rprintf("------- PASS THROUGH VOL DATATYPE Get\n");
#endif

    ret_value = H5VLdatatype_get(o->under_object, o->under_vol_id, args, dxpl_id, req);

    
    if (req && *req)
        *req = H5VL_pass_through_new_obj(*req, o->under_vol_id);

    return ret_value;
} 

static herr_t
H5VL_pass_through_datatype_specific(void *obj, H5VL_datatype_specific_args_t *args, hid_t dxpl_id, void **req)
{
    H5VL_pass_through_t *o = (H5VL_pass_through_t *)obj;
    hid_t                under_vol_id;
    herr_t               ret_value;

#ifdef ENABLE_PASSTHRU_LOGGING
    Rprintf("------- PASS THROUGH VOL DATATYPE Specific\n");
#endif

    
    under_vol_id = o->under_vol_id;

    ret_value = H5VLdatatype_specific(o->under_object, o->under_vol_id, args, dxpl_id, req);

    
    if (req && *req)
        *req = H5VL_pass_through_new_obj(*req, under_vol_id);

    return ret_value;
} 

static herr_t
H5VL_pass_through_datatype_optional(void *obj, H5VL_optional_args_t *args, hid_t dxpl_id, void **req)
{
    H5VL_pass_through_t *o = (H5VL_pass_through_t *)obj;
    herr_t               ret_value;

#ifdef ENABLE_PASSTHRU_LOGGING
    Rprintf("------- PASS THROUGH VOL DATATYPE Optional\n");
#endif

    ret_value = H5VLdatatype_optional(o->under_object, o->under_vol_id, args, dxpl_id, req);

    
    if (req && *req)
        *req = H5VL_pass_through_new_obj(*req, o->under_vol_id);

    return ret_value;
} 

static herr_t
H5VL_pass_through_datatype_close(void *dt, hid_t dxpl_id, void **req)
{
    H5VL_pass_through_t *o = (H5VL_pass_through_t *)dt;
    herr_t               ret_value;

#ifdef ENABLE_PASSTHRU_LOGGING
    Rprintf("------- PASS THROUGH VOL DATATYPE Close\n");
#endif

    assert(o->under_object);

    ret_value = H5VLdatatype_close(o->under_object, o->under_vol_id, dxpl_id, req);

    
    if (req && *req)
        *req = H5VL_pass_through_new_obj(*req, o->under_vol_id);

    
    if (ret_value >= 0)
        H5VL_pass_through_free_obj(o);

    return ret_value;
} 

static void *
H5VL_pass_through_file_create(const char *name, unsigned flags, hid_t fcpl_id, hid_t fapl_id, hid_t dxpl_id,
                              void **req)
{
    H5VL_pass_through_info_t *info;
    H5VL_pass_through_t      *file;
    hid_t                     under_fapl_id;
    void                     *under;

#ifdef ENABLE_PASSTHRU_LOGGING
    Rprintf("------- PASS THROUGH VOL FILE Create\n");
#endif

    
    H5Pget_vol_info(fapl_id, (void **)&info);

    
    if (!info)
        return NULL;

    
    under_fapl_id = H5Pcopy(fapl_id);

    
    H5Pset_vol(under_fapl_id, info->under_vol_id, info->under_vol_info);

    
    under = H5VLfile_create(name, flags, fcpl_id, under_fapl_id, dxpl_id, req);
    if (under) {
        file = H5VL_pass_through_new_obj(under, info->under_vol_id);

        
        if (req && *req)
            *req = H5VL_pass_through_new_obj(*req, info->under_vol_id);
    } 
    else
        file = NULL;

    
    H5Pclose(under_fapl_id);

    
    H5VL_pass_through_info_free(info);

    return (void *)file;
} 

static void *
H5VL_pass_through_file_open(const char *name, unsigned flags, hid_t fapl_id, hid_t dxpl_id, void **req)
{
    H5VL_pass_through_info_t *info;
    H5VL_pass_through_t      *file;
    hid_t                     under_fapl_id;
    void                     *under;

#ifdef ENABLE_PASSTHRU_LOGGING
    Rprintf("------- PASS THROUGH VOL FILE Open\n");
#endif

    
    H5Pget_vol_info(fapl_id, (void **)&info);

    
    if (!info)
        return NULL;

    
    under_fapl_id = H5Pcopy(fapl_id);

    
    H5Pset_vol(under_fapl_id, info->under_vol_id, info->under_vol_info);

    
    under = H5VLfile_open(name, flags, under_fapl_id, dxpl_id, req);
    if (under) {
        file = H5VL_pass_through_new_obj(under, info->under_vol_id);

        
        if (req && *req)
            *req = H5VL_pass_through_new_obj(*req, info->under_vol_id);
    } 
    else
        file = NULL;

    
    H5Pclose(under_fapl_id);

    
    H5VL_pass_through_info_free(info);

    return (void *)file;
} 

static herr_t
H5VL_pass_through_file_get(void *file, H5VL_file_get_args_t *args, hid_t dxpl_id, void **req)
{
    H5VL_pass_through_t *o = (H5VL_pass_through_t *)file;
    herr_t               ret_value;

#ifdef ENABLE_PASSTHRU_LOGGING
    Rprintf("------- PASS THROUGH VOL FILE Get\n");
#endif

    ret_value = H5VLfile_get(o->under_object, o->under_vol_id, args, dxpl_id, req);

    
    if (req && *req)
        *req = H5VL_pass_through_new_obj(*req, o->under_vol_id);

    return ret_value;
} 

static herr_t
H5VL_pass_through_file_specific(void *file, H5VL_file_specific_args_t *args, hid_t dxpl_id, void **req)
{
    H5VL_pass_through_t       *o = (H5VL_pass_through_t *)file;
    H5VL_pass_through_t       *new_o;
    H5VL_file_specific_args_t  my_args;
    H5VL_file_specific_args_t *new_args;
    H5VL_pass_through_info_t  *info         = NULL;
    hid_t                      under_vol_id = -1;
    herr_t                     ret_value;

#ifdef ENABLE_PASSTHRU_LOGGING
    Rprintf("------- PASS THROUGH VOL FILE Specific\n");
#endif

    if (args->op_type == H5VL_FILE_IS_ACCESSIBLE) {
        
        memcpy(&my_args, args, sizeof(my_args));

        
        H5Pget_vol_info(args->args.is_accessible.fapl_id, (void **)&info);

        
        if (!info)
            return (-1);

        
        under_vol_id = info->under_vol_id;

        
        my_args.args.is_accessible.fapl_id = H5Pcopy(args->args.is_accessible.fapl_id);

        
        H5Pset_vol(my_args.args.is_accessible.fapl_id, info->under_vol_id, info->under_vol_info);

        
        new_args = &my_args;

        
        new_o = NULL;
    } 
    else if (args->op_type == H5VL_FILE_DELETE) {
        
        memcpy(&my_args, args, sizeof(my_args));

        
        H5Pget_vol_info(args->args.del.fapl_id, (void **)&info);

        
        if (!info)
            return (-1);

        
        under_vol_id = info->under_vol_id;

        
        my_args.args.del.fapl_id = H5Pcopy(args->args.del.fapl_id);

        
        H5Pset_vol(my_args.args.del.fapl_id, info->under_vol_id, info->under_vol_info);

        
        new_args = &my_args;

        
        new_o = NULL;
    } 
    else {
        
        under_vol_id = o->under_vol_id;

        
        new_args = args;

        
        new_o = o->under_object;
    } 

    ret_value = H5VLfile_specific(new_o, under_vol_id, new_args, dxpl_id, req);

    
    if (req && *req)
        *req = H5VL_pass_through_new_obj(*req, under_vol_id);

    if (args->op_type == H5VL_FILE_IS_ACCESSIBLE) {
        
        H5Pclose(my_args.args.is_accessible.fapl_id);

        
        H5VL_pass_through_info_free(info);
    } 
    else if (args->op_type == H5VL_FILE_DELETE) {
        
        H5Pclose(my_args.args.del.fapl_id);

        
        H5VL_pass_through_info_free(info);
    } 
    else if (args->op_type == H5VL_FILE_REOPEN) {
        
        if (ret_value >= 0 && *args->args.reopen.file)
            *args->args.reopen.file = H5VL_pass_through_new_obj(*args->args.reopen.file, under_vol_id);
    } 

    return ret_value;
} 

static herr_t
H5VL_pass_through_file_optional(void *file, H5VL_optional_args_t *args, hid_t dxpl_id, void **req)
{
    H5VL_pass_through_t *o = (H5VL_pass_through_t *)file;
    herr_t               ret_value;

#ifdef ENABLE_PASSTHRU_LOGGING
    Rprintf("------- PASS THROUGH VOL File Optional\n");
#endif

    ret_value = H5VLfile_optional(o->under_object, o->under_vol_id, args, dxpl_id, req);

    
    if (req && *req)
        *req = H5VL_pass_through_new_obj(*req, o->under_vol_id);

    return ret_value;
} 

static herr_t
H5VL_pass_through_file_close(void *file, hid_t dxpl_id, void **req)
{
    H5VL_pass_through_t *o = (H5VL_pass_through_t *)file;
    herr_t               ret_value;

#ifdef ENABLE_PASSTHRU_LOGGING
    Rprintf("------- PASS THROUGH VOL FILE Close\n");
#endif

    ret_value = H5VLfile_close(o->under_object, o->under_vol_id, dxpl_id, req);

    
    if (req && *req)
        *req = H5VL_pass_through_new_obj(*req, o->under_vol_id);

    
    if (ret_value >= 0)
        H5VL_pass_through_free_obj(o);

    return ret_value;
} 

static void *
H5VL_pass_through_group_create(void *obj, const H5VL_loc_params_t *loc_params, const char *name,
                               hid_t lcpl_id, hid_t gcpl_id, hid_t gapl_id, hid_t dxpl_id, void **req)
{
    H5VL_pass_through_t *group;
    H5VL_pass_through_t *o = (H5VL_pass_through_t *)obj;
    void                *under;

#ifdef ENABLE_PASSTHRU_LOGGING
    Rprintf("------- PASS THROUGH VOL GROUP Create\n");
#endif

    under = H5VLgroup_create(o->under_object, loc_params, o->under_vol_id, name, lcpl_id, gcpl_id, gapl_id,
                             dxpl_id, req);
    if (under) {
        group = H5VL_pass_through_new_obj(under, o->under_vol_id);

        
        if (req && *req)
            *req = H5VL_pass_through_new_obj(*req, o->under_vol_id);
    } 
    else
        group = NULL;

    return (void *)group;
} 

static void *
H5VL_pass_through_group_open(void *obj, const H5VL_loc_params_t *loc_params, const char *name, hid_t gapl_id,
                             hid_t dxpl_id, void **req)
{
    H5VL_pass_through_t *group;
    H5VL_pass_through_t *o = (H5VL_pass_through_t *)obj;
    void                *under;

#ifdef ENABLE_PASSTHRU_LOGGING
    Rprintf("------- PASS THROUGH VOL GROUP Open\n");
#endif

    under = H5VLgroup_open(o->under_object, loc_params, o->under_vol_id, name, gapl_id, dxpl_id, req);
    if (under) {
        group = H5VL_pass_through_new_obj(under, o->under_vol_id);

        
        if (req && *req)
            *req = H5VL_pass_through_new_obj(*req, o->under_vol_id);
    } 
    else
        group = NULL;

    return (void *)group;
} 

static herr_t
H5VL_pass_through_group_get(void *obj, H5VL_group_get_args_t *args, hid_t dxpl_id, void **req)
{
    H5VL_pass_through_t *o = (H5VL_pass_through_t *)obj;
    herr_t               ret_value;

#ifdef ENABLE_PASSTHRU_LOGGING
    Rprintf("------- PASS THROUGH VOL GROUP Get\n");
#endif

    ret_value = H5VLgroup_get(o->under_object, o->under_vol_id, args, dxpl_id, req);

    
    if (req && *req)
        *req = H5VL_pass_through_new_obj(*req, o->under_vol_id);

    return ret_value;
} 

static herr_t
H5VL_pass_through_group_specific(void *obj, H5VL_group_specific_args_t *args, hid_t dxpl_id, void **req)
{
    H5VL_pass_through_t *o = (H5VL_pass_through_t *)obj;
    hid_t                under_vol_id;
    herr_t               ret_value;

#ifdef ENABLE_PASSTHRU_LOGGING
    Rprintf("------- PASS THROUGH VOL GROUP Specific\n");
#endif

    
    under_vol_id = o->under_vol_id;

    
    if (args->op_type == H5VL_GROUP_MOUNT) {
        H5VL_group_specific_args_t vol_cb_args; 

        
        vol_cb_args.op_type         = H5VL_GROUP_MOUNT;
        vol_cb_args.args.mount.name = args->args.mount.name;
        vol_cb_args.args.mount.child_file =
            ((H5VL_pass_through_t *)args->args.mount.child_file)->under_object;
        vol_cb_args.args.mount.fmpl_id = args->args.mount.fmpl_id;

        
        ret_value = H5VLgroup_specific(o->under_object, under_vol_id, &vol_cb_args, dxpl_id, req);
    } 
    else
        ret_value = H5VLgroup_specific(o->under_object, under_vol_id, args, dxpl_id, req);

    
    if (req && *req)
        *req = H5VL_pass_through_new_obj(*req, under_vol_id);

    return ret_value;
} 

static herr_t
H5VL_pass_through_group_optional(void *obj, H5VL_optional_args_t *args, hid_t dxpl_id, void **req)
{
    H5VL_pass_through_t *o = (H5VL_pass_through_t *)obj;
    herr_t               ret_value;

#ifdef ENABLE_PASSTHRU_LOGGING
    Rprintf("------- PASS THROUGH VOL GROUP Optional\n");
#endif

    ret_value = H5VLgroup_optional(o->under_object, o->under_vol_id, args, dxpl_id, req);

    
    if (req && *req)
        *req = H5VL_pass_through_new_obj(*req, o->under_vol_id);

    return ret_value;
} 

static herr_t
H5VL_pass_through_group_close(void *grp, hid_t dxpl_id, void **req)
{
    H5VL_pass_through_t *o = (H5VL_pass_through_t *)grp;
    herr_t               ret_value;

#ifdef ENABLE_PASSTHRU_LOGGING
    Rprintf("------- PASS THROUGH VOL H5Gclose\n");
#endif

    ret_value = H5VLgroup_close(o->under_object, o->under_vol_id, dxpl_id, req);

    
    if (req && *req)
        *req = H5VL_pass_through_new_obj(*req, o->under_vol_id);

    
    if (ret_value >= 0)
        H5VL_pass_through_free_obj(o);

    return ret_value;
} 

static herr_t
H5VL_pass_through_link_create(H5VL_link_create_args_t *args, void *obj, const H5VL_loc_params_t *loc_params,
                              hid_t lcpl_id, hid_t lapl_id, hid_t dxpl_id, void **req)
{
    H5VL_pass_through_t *o            = (H5VL_pass_through_t *)obj;
    hid_t                under_vol_id = -1;
    herr_t               ret_value;

#ifdef ENABLE_PASSTHRU_LOGGING
    Rprintf("------- PASS THROUGH VOL LINK Create\n");
#endif

    
    if (o)
        under_vol_id = o->under_vol_id;

    
    if (H5VL_LINK_CREATE_HARD == args->op_type) {
        void *cur_obj = args->args.hard.curr_obj;

        
        if (cur_obj) {
            
            if (under_vol_id < 0)
                under_vol_id = ((H5VL_pass_through_t *)cur_obj)->under_vol_id;

            
            args->args.hard.curr_obj = ((H5VL_pass_through_t *)cur_obj)->under_object;
        } 
    }     

    ret_value = H5VLlink_create(args, (o ? o->under_object : NULL), loc_params, under_vol_id, lcpl_id,
                                lapl_id, dxpl_id, req);

    
    if (req && *req)
        *req = H5VL_pass_through_new_obj(*req, under_vol_id);

    return ret_value;
} 

static herr_t
H5VL_pass_through_link_copy(void *src_obj, const H5VL_loc_params_t *loc_params1, void *dst_obj,
                            const H5VL_loc_params_t *loc_params2, hid_t lcpl_id, hid_t lapl_id, hid_t dxpl_id,
                            void **req)
{
    H5VL_pass_through_t *o_src        = (H5VL_pass_through_t *)src_obj;
    H5VL_pass_through_t *o_dst        = (H5VL_pass_through_t *)dst_obj;
    hid_t                under_vol_id = -1;
    herr_t               ret_value;

#ifdef ENABLE_PASSTHRU_LOGGING
    Rprintf("------- PASS THROUGH VOL LINK Copy\n");
#endif

    
    if (o_src)
        under_vol_id = o_src->under_vol_id;
    else if (o_dst)
        under_vol_id = o_dst->under_vol_id;
    assert(under_vol_id > 0);

    ret_value =
        H5VLlink_copy((o_src ? o_src->under_object : NULL), loc_params1, (o_dst ? o_dst->under_object : NULL),
                      loc_params2, under_vol_id, lcpl_id, lapl_id, dxpl_id, req);

    
    if (req && *req)
        *req = H5VL_pass_through_new_obj(*req, under_vol_id);

    return ret_value;
} 

static herr_t
H5VL_pass_through_link_move(void *src_obj, const H5VL_loc_params_t *loc_params1, void *dst_obj,
                            const H5VL_loc_params_t *loc_params2, hid_t lcpl_id, hid_t lapl_id, hid_t dxpl_id,
                            void **req)
{
    H5VL_pass_through_t *o_src        = (H5VL_pass_through_t *)src_obj;
    H5VL_pass_through_t *o_dst        = (H5VL_pass_through_t *)dst_obj;
    hid_t                under_vol_id = -1;
    herr_t               ret_value;

#ifdef ENABLE_PASSTHRU_LOGGING
    Rprintf("------- PASS THROUGH VOL LINK Move\n");
#endif

    
    if (o_src)
        under_vol_id = o_src->under_vol_id;
    else if (o_dst)
        under_vol_id = o_dst->under_vol_id;
    assert(under_vol_id > 0);

    ret_value =
        H5VLlink_move((o_src ? o_src->under_object : NULL), loc_params1, (o_dst ? o_dst->under_object : NULL),
                      loc_params2, under_vol_id, lcpl_id, lapl_id, dxpl_id, req);

    
    if (req && *req)
        *req = H5VL_pass_through_new_obj(*req, under_vol_id);

    return ret_value;
} 

static herr_t
H5VL_pass_through_link_get(void *obj, const H5VL_loc_params_t *loc_params, H5VL_link_get_args_t *args,
                           hid_t dxpl_id, void **req)
{
    H5VL_pass_through_t *o = (H5VL_pass_through_t *)obj;
    herr_t               ret_value;

#ifdef ENABLE_PASSTHRU_LOGGING
    Rprintf("------- PASS THROUGH VOL LINK Get\n");
#endif

    ret_value = H5VLlink_get(o->under_object, loc_params, o->under_vol_id, args, dxpl_id, req);

    
    if (req && *req)
        *req = H5VL_pass_through_new_obj(*req, o->under_vol_id);

    return ret_value;
} 

static herr_t
H5VL_pass_through_link_specific(void *obj, const H5VL_loc_params_t *loc_params,
                                H5VL_link_specific_args_t *args, hid_t dxpl_id, void **req)
{
    H5VL_pass_through_t *o = (H5VL_pass_through_t *)obj;
    herr_t               ret_value;

#ifdef ENABLE_PASSTHRU_LOGGING
    Rprintf("------- PASS THROUGH VOL LINK Specific\n");
#endif

    ret_value = H5VLlink_specific(o->under_object, loc_params, o->under_vol_id, args, dxpl_id, req);

    
    if (req && *req)
        *req = H5VL_pass_through_new_obj(*req, o->under_vol_id);

    return ret_value;
} 

static herr_t
H5VL_pass_through_link_optional(void *obj, const H5VL_loc_params_t *loc_params, H5VL_optional_args_t *args,
                                hid_t dxpl_id, void **req)
{
    H5VL_pass_through_t *o = (H5VL_pass_through_t *)obj;
    herr_t               ret_value;

#ifdef ENABLE_PASSTHRU_LOGGING
    Rprintf("------- PASS THROUGH VOL LINK Optional\n");
#endif

    ret_value = H5VLlink_optional(o->under_object, loc_params, o->under_vol_id, args, dxpl_id, req);

    
    if (req && *req)
        *req = H5VL_pass_through_new_obj(*req, o->under_vol_id);

    return ret_value;
} 

static void *
H5VL_pass_through_object_open(void *obj, const H5VL_loc_params_t *loc_params, H5I_type_t *opened_type,
                              hid_t dxpl_id, void **req)
{
    H5VL_pass_through_t *new_obj;
    H5VL_pass_through_t *o = (H5VL_pass_through_t *)obj;
    void                *under;

#ifdef ENABLE_PASSTHRU_LOGGING
    Rprintf("------- PASS THROUGH VOL OBJECT Open\n");
#endif

    under = H5VLobject_open(o->under_object, loc_params, o->under_vol_id, opened_type, dxpl_id, req);
    if (under) {
        new_obj = H5VL_pass_through_new_obj(under, o->under_vol_id);

        
        if (req && *req)
            *req = H5VL_pass_through_new_obj(*req, o->under_vol_id);
    } 
    else
        new_obj = NULL;

    return (void *)new_obj;
} 

static herr_t
H5VL_pass_through_object_copy(void *src_obj, const H5VL_loc_params_t *src_loc_params, const char *src_name,
                              void *dst_obj, const H5VL_loc_params_t *dst_loc_params, const char *dst_name,
                              hid_t ocpypl_id, hid_t lcpl_id, hid_t dxpl_id, void **req)
{
    H5VL_pass_through_t *o_src = (H5VL_pass_through_t *)src_obj;
    H5VL_pass_through_t *o_dst = (H5VL_pass_through_t *)dst_obj;
    herr_t               ret_value;

#ifdef ENABLE_PASSTHRU_LOGGING
    Rprintf("------- PASS THROUGH VOL OBJECT Copy\n");
#endif

    ret_value =
        H5VLobject_copy(o_src->under_object, src_loc_params, src_name, o_dst->under_object, dst_loc_params,
                        dst_name, o_src->under_vol_id, ocpypl_id, lcpl_id, dxpl_id, req);

    
    if (req && *req)
        *req = H5VL_pass_through_new_obj(*req, o_src->under_vol_id);

    return ret_value;
} 

static herr_t
H5VL_pass_through_object_get(void *obj, const H5VL_loc_params_t *loc_params, H5VL_object_get_args_t *args,
                             hid_t dxpl_id, void **req)
{
    H5VL_pass_through_t *o = (H5VL_pass_through_t *)obj;
    herr_t               ret_value;

#ifdef ENABLE_PASSTHRU_LOGGING
    Rprintf("------- PASS THROUGH VOL OBJECT Get\n");
#endif

    ret_value = H5VLobject_get(o->under_object, loc_params, o->under_vol_id, args, dxpl_id, req);

    
    if (req && *req)
        *req = H5VL_pass_through_new_obj(*req, o->under_vol_id);

    return ret_value;
} 

static herr_t
H5VL_pass_through_object_specific(void *obj, const H5VL_loc_params_t *loc_params,
                                  H5VL_object_specific_args_t *args, hid_t dxpl_id, void **req)
{
    H5VL_pass_through_t *o = (H5VL_pass_through_t *)obj;
    hid_t                under_vol_id;
    herr_t               ret_value;

#ifdef ENABLE_PASSTHRU_LOGGING
    Rprintf("------- PASS THROUGH VOL OBJECT Specific\n");
#endif

    
    under_vol_id = o->under_vol_id;

    ret_value = H5VLobject_specific(o->under_object, loc_params, o->under_vol_id, args, dxpl_id, req);

    
    if (req && *req)
        *req = H5VL_pass_through_new_obj(*req, under_vol_id);

    return ret_value;
} 

static herr_t
H5VL_pass_through_object_optional(void *obj, const H5VL_loc_params_t *loc_params, H5VL_optional_args_t *args,
                                  hid_t dxpl_id, void **req)
{
    H5VL_pass_through_t *o = (H5VL_pass_through_t *)obj;
    herr_t               ret_value;

#ifdef ENABLE_PASSTHRU_LOGGING
    Rprintf("------- PASS THROUGH VOL OBJECT Optional\n");
#endif

    ret_value = H5VLobject_optional(o->under_object, loc_params, o->under_vol_id, args, dxpl_id, req);

    
    if (req && *req)
        *req = H5VL_pass_through_new_obj(*req, o->under_vol_id);

    return ret_value;
} 

herr_t
H5VL_pass_through_introspect_get_conn_cls(void *obj, H5VL_get_conn_lvl_t lvl, const H5VL_class_t **conn_cls)
{
    H5VL_pass_through_t *o = (H5VL_pass_through_t *)obj;
    herr_t               ret_value;

#ifdef ENABLE_PASSTHRU_LOGGING
    Rprintf("------- PASS THROUGH VOL INTROSPECT GetConnCls\n");
#endif

    
    if (H5VL_GET_CONN_LVL_CURR == lvl) {
        *conn_cls = &H5VL_pass_through_g;
        ret_value = 0;
    } 
    else
        ret_value = H5VLintrospect_get_conn_cls(o->under_object, o->under_vol_id, lvl, conn_cls);

    return ret_value;
} 

herr_t
H5VL_pass_through_introspect_get_cap_flags(const void *_info, uint64_t *cap_flags)
{
    const H5VL_pass_through_info_t *info = (const H5VL_pass_through_info_t *)_info;
    herr_t                          ret_value;

#ifdef ENABLE_PASSTHRU_LOGGING
    Rprintf("------- PASS THROUGH VOL INTROSPECT GetCapFlags\n");
#endif

    
    if (!info) {
        Rprintf("\nH5VLpassthru.c line %d in %s: info for pass-through VOL can't be null\n", __LINE__,
               __func__);
        return -1;
    }

    if (H5Iis_valid(info->under_vol_id) <= 0) {
        Rprintf("\nH5VLpassthru.c line %d in %s: not a valid underneath VOL ID for pass-through VOL\n",
               __LINE__, __func__);
        return -1;
    }

    
    ret_value = H5VLintrospect_get_cap_flags(info->under_vol_info, info->under_vol_id, cap_flags);

    
    if (ret_value >= 0)
        *cap_flags |= H5VL_pass_through_g.cap_flags;

    return ret_value;
} 

herr_t
H5VL_pass_through_introspect_opt_query(void *obj, H5VL_subclass_t cls, int opt_type, uint64_t *flags)
{
    H5VL_pass_through_t *o = (H5VL_pass_through_t *)obj;
    herr_t               ret_value;

#ifdef ENABLE_PASSTHRU_LOGGING
    Rprintf("------- PASS THROUGH VOL INTROSPECT OptQuery\n");
#endif

    ret_value = H5VLintrospect_opt_query(o->under_object, o->under_vol_id, cls, opt_type, flags);

    return ret_value;
} 

static herr_t
H5VL_pass_through_request_wait(void *obj, uint64_t timeout, H5VL_request_status_t *status)
{
    H5VL_pass_through_t *o = (H5VL_pass_through_t *)obj;
    herr_t               ret_value;

#ifdef ENABLE_PASSTHRU_LOGGING
    Rprintf("------- PASS THROUGH VOL REQUEST Wait\n");
#endif

    ret_value = H5VLrequest_wait(o->under_object, o->under_vol_id, timeout, status);

    if (ret_value >= 0 && *status != H5VL_REQUEST_STATUS_IN_PROGRESS)
        H5VL_pass_through_free_obj(o);

    return ret_value;
} 

static herr_t
H5VL_pass_through_request_notify(void *obj, H5VL_request_notify_t cb, void *ctx)
{
    H5VL_pass_through_t *o = (H5VL_pass_through_t *)obj;
    herr_t               ret_value;

#ifdef ENABLE_PASSTHRU_LOGGING
    Rprintf("------- PASS THROUGH VOL REQUEST Notify\n");
#endif

    ret_value = H5VLrequest_notify(o->under_object, o->under_vol_id, cb, ctx);

    if (ret_value >= 0)
        H5VL_pass_through_free_obj(o);

    return ret_value;
} 

static herr_t
H5VL_pass_through_request_cancel(void *obj, H5VL_request_status_t *status)
{
    H5VL_pass_through_t *o = (H5VL_pass_through_t *)obj;
    herr_t               ret_value;

#ifdef ENABLE_PASSTHRU_LOGGING
    Rprintf("------- PASS THROUGH VOL REQUEST Cancel\n");
#endif

    ret_value = H5VLrequest_cancel(o->under_object, o->under_vol_id, status);

    if (ret_value >= 0)
        H5VL_pass_through_free_obj(o);

    return ret_value;
} 

static herr_t
H5VL_pass_through_request_specific(void *obj, H5VL_request_specific_args_t *args)
{
    H5VL_pass_through_t *o         = (H5VL_pass_through_t *)obj;
    herr_t               ret_value = -1;

#ifdef ENABLE_PASSTHRU_LOGGING
    Rprintf("------- PASS THROUGH VOL REQUEST Specific\n");
#endif

    ret_value = H5VLrequest_specific(o->under_object, o->under_vol_id, args);

    return ret_value;
} 

static herr_t
H5VL_pass_through_request_optional(void *obj, H5VL_optional_args_t *args)
{
    H5VL_pass_through_t *o = (H5VL_pass_through_t *)obj;
    herr_t               ret_value;

#ifdef ENABLE_PASSTHRU_LOGGING
    Rprintf("------- PASS THROUGH VOL REQUEST Optional\n");
#endif

    ret_value = H5VLrequest_optional(o->under_object, o->under_vol_id, args);

    return ret_value;
} 

static herr_t
H5VL_pass_through_request_free(void *obj)
{
    H5VL_pass_through_t *o = (H5VL_pass_through_t *)obj;
    herr_t               ret_value;

#ifdef ENABLE_PASSTHRU_LOGGING
    Rprintf("------- PASS THROUGH VOL REQUEST Free\n");
#endif

    ret_value = H5VLrequest_free(o->under_object, o->under_vol_id);

    if (ret_value >= 0)
        H5VL_pass_through_free_obj(o);

    return ret_value;
} 

herr_t
H5VL_pass_through_blob_put(void *obj, const void *buf, size_t size, void *blob_id, void *ctx)
{
    H5VL_pass_through_t *o = (H5VL_pass_through_t *)obj;
    herr_t               ret_value;

#ifdef ENABLE_PASSTHRU_LOGGING
    Rprintf("------- PASS THROUGH VOL BLOB Put\n");
#endif

    ret_value = H5VLblob_put(o->under_object, o->under_vol_id, buf, size, blob_id, ctx);

    return ret_value;
} 

herr_t
H5VL_pass_through_blob_get(void *obj, const void *blob_id, void *buf, size_t size, void *ctx)
{
    H5VL_pass_through_t *o = (H5VL_pass_through_t *)obj;
    herr_t               ret_value;

#ifdef ENABLE_PASSTHRU_LOGGING
    Rprintf("------- PASS THROUGH VOL BLOB Get\n");
#endif

    ret_value = H5VLblob_get(o->under_object, o->under_vol_id, blob_id, buf, size, ctx);

    return ret_value;
} 

herr_t
H5VL_pass_through_blob_specific(void *obj, void *blob_id, H5VL_blob_specific_args_t *args)
{
    H5VL_pass_through_t *o = (H5VL_pass_through_t *)obj;
    herr_t               ret_value;

#ifdef ENABLE_PASSTHRU_LOGGING
    Rprintf("------- PASS THROUGH VOL BLOB Specific\n");
#endif

    ret_value = H5VLblob_specific(o->under_object, o->under_vol_id, blob_id, args);

    return ret_value;
} 

herr_t
H5VL_pass_through_blob_optional(void *obj, void *blob_id, H5VL_optional_args_t *args)
{
    H5VL_pass_through_t *o = (H5VL_pass_through_t *)obj;
    herr_t               ret_value;

#ifdef ENABLE_PASSTHRU_LOGGING
    Rprintf("------- PASS THROUGH VOL BLOB Optional\n");
#endif

    ret_value = H5VLblob_optional(o->under_object, o->under_vol_id, blob_id, args);

    return ret_value;
} 

static herr_t
H5VL_pass_through_token_cmp(void *obj, const H5O_token_t *token1, const H5O_token_t *token2, int *cmp_value)
{
    H5VL_pass_through_t *o = (H5VL_pass_through_t *)obj;
    herr_t               ret_value;

#ifdef ENABLE_PASSTHRU_LOGGING
    Rprintf("------- PASS THROUGH VOL TOKEN Compare\n");
#endif

    
    assert(obj);
    assert(token1);
    assert(token2);
    assert(cmp_value);

    ret_value = H5VLtoken_cmp(o->under_object, o->under_vol_id, token1, token2, cmp_value);

    return ret_value;
} 

static herr_t
H5VL_pass_through_token_to_str(void *obj, H5I_type_t obj_type, const H5O_token_t *token, char **token_str)
{
    H5VL_pass_through_t *o = (H5VL_pass_through_t *)obj;
    herr_t               ret_value;

#ifdef ENABLE_PASSTHRU_LOGGING
    Rprintf("------- PASS THROUGH VOL TOKEN To string\n");
#endif

    
    assert(obj);
    assert(token);
    assert(token_str);

    ret_value = H5VLtoken_to_str(o->under_object, obj_type, o->under_vol_id, token, token_str);

    return ret_value;
} 

static herr_t
H5VL_pass_through_token_from_str(void *obj, H5I_type_t obj_type, const char *token_str, H5O_token_t *token)
{
    H5VL_pass_through_t *o = (H5VL_pass_through_t *)obj;
    herr_t               ret_value;

#ifdef ENABLE_PASSTHRU_LOGGING
    Rprintf("------- PASS THROUGH VOL TOKEN From string\n");
#endif

    
    assert(obj);
    assert(token);
    assert(token_str);

    ret_value = H5VLtoken_from_str(o->under_object, obj_type, o->under_vol_id, token_str, token);

    return ret_value;
} 

herr_t
H5VL_pass_through_optional(void *obj, H5VL_optional_args_t *args, hid_t dxpl_id, void **req)
{
    H5VL_pass_through_t *o = (H5VL_pass_through_t *)obj;
    herr_t               ret_value;

#ifdef ENABLE_PASSTHRU_LOGGING
    Rprintf("------- PASS THROUGH VOL generic Optional\n");
#endif

    ret_value = H5VLoptional(o->under_object, o->under_vol_id, args, dxpl_id, req);

    return ret_value;
} 
