% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/attributes.R
\name{igraph-attribute-combination}
\alias{igraph-attribute-combination}
\alias{attribute.combination}
\title{How igraph functions handle attributes when the graph changes}
\description{
Many times, when the structure of a graph is modified, vertices/edges map of
the original graph map to vertices/edges in the newly created (modified)
graph. For example \code{\link[=simplify]{simplify()}} maps multiple edges to single
edges. igraph provides a flexible mechanism to specify what to do with the
vertex/edge attributes in these cases.
}
\details{
The functions that support the combination of attributes have one or two
extra arguments called \code{vertex.attr.comb} and/or \code{edge.attr.comb}
that specify how to perform the mapping of the attributes. E.g.
\code{\link[=contract]{contract()}} contracts many vertices into a single one, the
attributes of the vertices can be combined and stores as the vertex
attributes of the new graph.

The specification of the combination of (vertex or edge) attributes can be
given as \enumerate{
\item a character scalar,
\item a function object or
\item a list of character scalars and/or function objects.
}

If it is a character scalar, then it refers to one of the predefined
combinations, see their list below.

If it is a function, then the given function is expected to perform the
combination. It will be called once for each new vertex/edge in the graph,
with a single argument: the attribute values of the vertices that map to
that single vertex.

The third option, a list can be used to specify different combination
methods for different attributes. A named entry of the list corresponds to
the attribute with the same name. An unnamed entry (i.e. if the name is the
empty string) of the list specifies the default combination method. I.e.
\preformatted{list(weight="sum", "ignore")} specifies that the weight of the
new edge should be sum of the weights of the corresponding edges in the old
graph; and that the rest of the attributes should be ignored (=dropped).
}
\section{Predefined combination functions}{
 The following combination
behaviors are predefined:
\describe{
\item{"ignore"}{
The attribute is ignored and dropped.
}
\item{"sum"}{
The sum of the attributes is calculated.
This does not work for character attributes and works for complex attributes
only if they have a \code{sum} generic defined.
(E.g. it works for sparse matrices from the \code{Matrix} package,
because they have a \code{sum} method.)
}
\item{"prod"}{
The product of the attributes is calculated.
This does not work for character attributes and works for complex attributes
only if they have a \code{prod} function defined.
}
\item{"min"}{
The minimum of the attributes is calculated and returned.
For character and complex attributes the standard R \code{min} function is used.
}
\item{"max"}{
The maximum of the attributes is calculated and returned.
For character and complex attributes the standard R \code{max} function is used.
}
\item{"random"}{
Chooses one of the supplied attribute values, uniformly randomly.
For character and complex attributes this is implemented by calling \code{sample}.
}
\item{"first"}{
Always chooses the first attribute value.
It is implemented by calling the \code{\link[=head]{head()}} function.
}
\item{"last"}{
Always chooses the last attribute value.
It is implemented by calling the \code{\link[=tail]{tail()}} function.
}
\item{"mean"}{
The mean of the attributes is calculated and returned.
For character and complex attributes this simply calls the \code{\link[=mean]{mean()}} function.
}
\item{"median"}{
The median of the attributes is selected.
Calls the R \code{\link[=median]{median()}} function for all attribute types.
}
\item{"concat"}{
Concatenate the attributes, using the \code{\link[=c]{c()}} function.
This results almost always a complex attribute.
}
}
}

\examples{

g <- make_graph(c(1, 2, 1, 2, 1, 2, 2, 3, 3, 4))
E(g)$weight <- 1:5

## print attribute values with the graph
igraph_options(print.graph.attributes = TRUE)
igraph_options(print.vertex.attributes = TRUE)
igraph_options(print.edge.attributes = TRUE)

## new attribute is the sum of the old ones
simplify(g, edge.attr.comb = "sum")

## collect attributes into a string
simplify(g, edge.attr.comb = toString)

## concatenate them into a vector, this creates a complex
## attribute
simplify(g, edge.attr.comb = "concat")

E(g)$name <- letters[seq_len(ecount(g))]

## both attributes are collected into strings
simplify(g, edge.attr.comb = toString)

## harmonic average of weights, names are dropped
simplify(g, edge.attr.comb = list(
  weight = function(x) length(x) / sum(1 / x),
  name = "ignore"
))
}
\seealso{
\code{\link[=graph_attr]{graph_attr()}}, \code{\link[=vertex_attr]{vertex_attr()}},
\code{\link[=edge_attr]{edge_attr()}} on how to use graph/vertex/edge attributes in
general. \code{\link[=igraph_options]{igraph_options()}} on igraph parameters.

Vertex, edge and graph attributes
\code{\link{delete_edge_attr}()},
\code{\link{delete_graph_attr}()},
\code{\link{delete_vertex_attr}()},
\code{\link{edge_attr}()},
\code{\link{edge_attr<-}()},
\code{\link{edge_attr_names}()},
\code{\link{graph_attr}()},
\code{\link{graph_attr<-}()},
\code{\link{graph_attr_names}()},
\code{\link{igraph-dollar}},
\code{\link{igraph-vs-attributes}},
\code{\link{set_edge_attr}()},
\code{\link{set_graph_attr}()},
\code{\link{set_vertex_attr}()},
\code{\link{set_vertex_attrs}()},
\code{\link{vertex_attr}()},
\code{\link{vertex_attr<-}()},
\code{\link{vertex_attr_names}()}
}
\author{
Gabor Csardi \email{csardi.gabor@gmail.com}
}
\concept{attributes}
\keyword{graphs}
