test_that("use_extendr() sets up extendr files correctly", {
  skip_if_not_installed("usethis")
  skip_on_cran()

  path <- local_package("testpkg")
  # capture setup messages
  withr::local_options(usethis.quiet = FALSE)
  use_extendr()

  # DESCRITION file
  version_in_desc <- stringi::stri_trim_both(desc::desc_get(
    "Config/rextendr/version",
    path
  )[[1]])
  sysreq_in_desc <- stringi::stri_trim_both(desc::desc_get(
    "SystemRequirements",
    path
  )[[1]])
  expect_identical(version_in_desc, as.character(packageVersion("rextendr")))
  expect_identical(
    sysreq_in_desc,
    "Cargo (Rust's package manager), rustc >= 1.65.0, xz"
  )

  # directory structure
  expect_true(dir.exists("src"))
  expect_true(dir.exists("tools"))
  expect_true(dir.exists(file.path("src", "rust")))
  expect_true(dir.exists(file.path("src", "rust", "src")))

  # ensure all files generated by rextendr are present
  expect_snapshot(cat_file(".gitignore"))
  expect_snapshot(cat_file(".Rbuildignore"))
  expect_snapshot(cat_file("configure"))
  expect_snapshot(cat_file("configure.win"))
  expect_snapshot(cat_file("tools", "msrv.R"))
  expect_snapshot(cat_file("R", "extendr-wrappers.R"))
  expect_snapshot(cat_file("src", ".gitignore"))
  expect_snapshot(cat_file("src", "Makevars.in"))
  expect_snapshot(cat_file("src", "entrypoint.c"))
  expect_snapshot(cat_file("src", "Makevars.win.in"))
  expect_snapshot(cat_file("src", "testpkg-win.def"))
  expect_snapshot(cat_file("src", "rust", "Cargo.toml"))
  expect_snapshot(cat_file("src", "rust", "src", "lib.rs"))
})

test_that("use_extendr() quiet if quiet=TRUE", {
  skip_if_not_installed("usethis")
  skip_on_cran()

  path <- local_package("quiet")
  expect_snapshot(use_extendr(quiet = TRUE))
})

test_that("use_extendr() skip pre-existing files in non-interactive sessions", {
  skip_if_not_installed("usethis")
  skip_on_cran()

  path <- local_package("testpkg.wrap")
  use_extendr(quiet = FALSE)
  withr::local_options(usethis.quiet = FALSE)
  expect_snapshot(use_extendr())
})

test_that("use_extendr() can overwrite files in non-interactive sessions", {
  skip_if_not_installed("usethis")
  skip_on_cran()

  path <- local_package("testpkg")
  use_extendr()
  withr::local_options(usethis.quiet = FALSE)
  expect_snapshot(use_extendr(
    crate_name = "foo",
    lib_name = "bar",
    overwrite = TRUE
  ))
  expect_snapshot(cat_file("src", "rust", "Cargo.toml"))
})

test_that("use_rextendr_template() works when usethis not available", {
  skip_if_not_installed("usethis")
  skip_on_cran()

  path <- local_package("testpkg.wrap")
  # mock that usethis installed
  with_mocked_bindings(
    {
      use_rextendr_template(
        "_gitignore",
        save_as = file.path("installed")
      )
    },
    is_installed = function(...) TRUE
  )

  # mock that usethis not installed
  with_mocked_bindings(
    {
      use_rextendr_template(
        "_gitignore",
        save_as = file.path("not_installed")
      )
    },
    is_installed = function(...) FALSE
  )

  expect_identical(
    brio::read_file(file.path("installed")),
    brio::read_file(file.path("not_installed"))
  )
})

test_that("use_rextendr_template() can overwrite existing files", {
  skip_if_not_installed("usethis")
  skip_on_cran()

  path <- local_package("testpkg.wrap")
  dir.create("src")
  file_path <- file.path("src", "Makevars.in")

  use_rextendr_template(
    "Makevars.in",
    save_as = file_path,
    quiet = TRUE,
    data = list(lib_name = "foo")
  )
  use_rextendr_template(
    "Makevars.in",
    save_as = file_path,
    quiet = TRUE,
    overwrite = TRUE,
    data = list(lib_name = "bar")
  )

  expect_snapshot(cat_file("src", "Makevars.in"))
})

# Check that {rextendr} works in packages containing dots in their names.
# The check is performed by compiling the sample package and checking that
# `hello_world()` template function is available and works.
test_that("use_extendr() handles R packages with dots in the name", {
  skip_if_not_installed("usethis")
  skip_if_not_installed("devtools")
  skip_on_cran()
  skip_if_cargo_unavailable()

  path <- local_package("a.b.c")
  use_extendr()
  document()
  devtools::load_all()
  expect_identical(hello_world(), "Hello world!")
})

# Specify crate name and library names explicitly
test_that("use_extendr() handles R package name, crate name and library name separately", {
  skip_if_not_installed("usethis")
  skip_if_not_installed("devtools")
  skip_on_cran()
  skip_if_cargo_unavailable()

  path <- local_package("testPackage")
  use_extendr(crate_name = "crate_name", lib_name = "lib_name")
  document()
  devtools::load_all()
  expect_identical(hello_world(), "Hello world!")
})

# Pass unsupported values to `crate_name` and `lib_name` and expect errors.
test_that("use_extendr() does not allow invalid rust names", {
  skip_if_not_installed("usethis")
  skip_on_cran()

  path <- local_package("testPackage")
  expect_rextendr_error(use_extendr(crate_name = "22unsupported"))
  expect_rextendr_error(use_extendr(lib_name = "@unsupported"))
})

test_that("R/ folder is created when not present", {
  skip_if_not_installed("usethis")
  skip_on_cran()

  path <- local_temp_dir("my.pkg")
  usethis::proj_set(path, force = TRUE)
  usethis::use_description()

  expect_false(dir.exists("R/"))

  # expect no error
  expect_rextendr_error(use_extendr(), regexp = NA)
})

test_that("Message if the SystemRequirements field is already set.", {
  skip_if_not_installed("usethis")
  skip_on_cran()

  path <- local_package("testpkg")
  sys_req <- "testreq"

  desc::desc_set("SystemRequirements", sys_req)

  withr::local_options(usethis.quiet = FALSE)
  expect_message(
    created <- use_extendr(),
    "Please update it manually if needed"
  )

  expect_true(created)
  expect_identical(desc::desc_get("SystemRequirements")[[1]], sys_req)
})

test_that("`use_extendr()` works correctly when path is specified explicitly", {
  skip_if_not_installed("usethis")
  skip_on_cran()

  local_temp_dir("temp_dir")
  usethis::create_package("testpkg")

  use_extendr(path = "testpkg")
  succeed()
})


test_that("`use_extendr()` passes R CMD check", {
  skip_if_not_installed("usethis")
  skip_if_not_installed("rcmdcheck")
  skip_on_cran()

  path <- local_package("testpkg")
  # write the license file to pass R CMD check
  usethis::use_mit_license()
  usethis::use_test("dummy", FALSE)
  use_extendr()
  vendor_pkgs()
  document()

  # store results
  res <- rcmdcheck::rcmdcheck(
    args = c("--no-manual", "--no-tests"),
    libpath = rev(.libPaths())
  )

  # check the output
  expect_length(res$errors, 0)
  expect_length(res$warnings, 0)
})
