#' Pre-processes a data frame of deaths, exposed to risk or times of exposition to be used by the `dw_crude_mx` function
#'
#' @description This function pre-processes a data frame (or matrix) of deaths,
#'              exposed to risk or times of exposition to be used by the
#'              `dw_crude_mx` function when the data frame has a variable of ages
#'              in the first column and likely data are non-available for all the
#'              sequence of ages desired by the user. It creates a new data frame by
#'              inserting zeroes in the cells corresponding to the missing ages.
#'
#' @author Jose M. Pavia \email{pavia@@uv.es}
#'
#' @param x A data frame (or matrix) of deaths, exposed to risk or times of exposition with ages as first column.
#' @param age.min A non-negative integer indicating the minimum age for the output object. Default, 0.
#' @param age.max A non-negative integer indicating the maximum age for the output object. Default, 120.
#
#' @return
#' A data.frame containing the same columns as `x` except for the first one, as outlined in the function's description.
#' @export
#'
#' @examples
#'
#' deaths <- structure(list(ages = c(67, 69:78),
#'                          Year2017 = c(144, 102, 113, 122, 156, 110, 126, 132, 120, 172, 110),
#'                          Year2018 = c(111, 122, 109, 116, 162, 154, 115, 146, 100, 169, 146),
#'                          Year2019 = c(100, 123, 113, 151, 122, 110, 137, 175, 137, 110, 155)),
#'                          class = "data.frame", row.names = 1:11)
#'
#' example <- dw_preprocess(x = deaths)
#'

dw_preprocess <- function(x,
                        age.min = 0,
                        age.max = 120){

  x <- as.matrix(x)
  nombres <- colnames(x)
  y <- x[, 1L]
  if (any((y - floor(y)) > 0) | any(y < 0))
    stop("Only non-negative integers are allowed for the first column of x.")
  x <- x[x[, 1] >= age.min & x[, 1] <= age.max, ]
  falta <- age.min:age.max
  falta <- falta[!(falta %in% x[, 1L])]
  if (length(falta) > 0){
    x <- rbind(x, cbind(falta, matrix(0, nrow = length(falta),
                                      ncol = ncol(x) - 1)))
  }
  x <- x[order(x[, 1]), ]
  rownames(x) <- paste("Age", x[, 1], sep =".")
  x <- x[, -1]
  colnames(x) <- nombres[-1]
  x <- as.data.frame(x)
  return(x)
}
