% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/pop_fis.R
\name{pop_fis}
\alias{pop_fis}
\title{Compute population specific FIS}
\usage{
pop_fis(
  .x,
  method = c("Nei87", "WG17"),
  by_locus = FALSE,
  include_global = FALSE,
  allele_sharing_mat = NULL
)
}
\arguments{
\item{.x}{a grouped \code{\link{gen_tibble}} (as obtained by using \code{\link[dplyr:group_by]{dplyr::group_by()}})}

\item{method}{one of "Nei87" (based on Nei 1987, eqn 7.41) or "WG17" (for
Weir and Goudet 2017) to compute FIS}

\item{by_locus}{boolean, determining whether FIS should be returned by
locus(TRUE), or as a single genome wide value (FALSE, the default). Note
that this is only relevant for "Nei87", as "WG17" always returns a single
value.}

\item{include_global}{boolean determining whether, besides the population
specific estimates, a global estimate should be appended. Note that this
will return a vector of n populations plus 1 (the global value), or a
matrix with n+1 columns if \code{by_locus=TRUE}.}

\item{allele_sharing_mat}{optional and only relevant for "WG17", the matrix
of Allele Sharing returned by \code{\link[=pairwise_allele_sharing]{pairwise_allele_sharing()}} with
\code{as_matrix=TRUE}. As a number of statistics can be derived from the Allele
Sharing matrix, it is sometimes more efficient to pre-compute this matrix.}
}
\value{
a vector of population specific fis (plus the global value if
\code{include_global=TRUE})
}
\description{
This function computes population specific FIS, using either the approach of
Nei 1987 (with an algorithm equivalent to the one used by
\code{hierfstat::basic.stats()}) or of Weir and Goudet 2017 (with an algorithm
equivalent to the one used by \code{hierfstat::fis.dosage()}).
}
\examples{
\dontshow{if (all(rlang::is_installed(c("RhpcBLASctl", "data.table")))) withAutoprint(\{ # examplesIf}
\dontshow{
data.table::setDTthreads(2)
RhpcBLASctl::blas_set_num_threads(2)
RhpcBLASctl::omp_set_num_threads(2)
}
example_gt <- load_example_gt("grouped_gen_tbl")

# Compute FIS using Nei87
example_gt \%>\% pop_fis(method = "Nei87")

# Compute FIS using WG17
example_gt \%>\% pop_fis(method = "WG17")

# To include the global FIS, set include_global = TRUE
example_gt \%>\% pop_fis(method = "Nei87", include_global = TRUE)

# To return FIS by locus, set by_locus = TRUE
example_gt \%>\% pop_fis(method = "Nei87", by_locus = TRUE)

# To calculate from a pre-computed allele sharing matrix:
allele_sharing_mat <- pairwise_allele_sharing(example_gt, as_matrix = TRUE)
example_gt \%>\% pop_fis(
  method = "WG17",
  allele_sharing_mat = allele_sharing_mat
)
\dontshow{\}) # examplesIf}
}
\references{
Nei M. (1987) Molecular Evolutionary Genetics. Columbia
University Press
Weir, BS and Goudet J (2017) A Unified Characterization of
Population Structure and Relatedness. Genetics (2017) 206:2085
}
\seealso{
\code{\link[hierfstat:basic.stats]{hierfstat::basic.stats()}} \code{\link[hierfstat:fs.dosage]{hierfstat::fis.dosage()}}
}
