











#![allow(clippy::match_wild_err_arm)]
use alloc::vec;
#[cfg(not(feature = "std"))]
use alloc::vec::Vec;
#[cfg(feature = "std")]
use num_traits::Float;
use num_traits::{One, Zero};
use std::mem;
use std::mem::MaybeUninit;

use crate::dimension::offset_from_low_addr_ptr_to_logical_ptr;
use crate::dimension::{self, CanIndexCheckMode};
use crate::error::{self, ShapeError};
use crate::extension::nonnull::nonnull_from_vec_data;
use crate::imp_prelude::*;
use crate::indexes;
use crate::indices;
#[cfg(feature = "std")]
use crate::iterators::to_vec;
use crate::iterators::to_vec_mapped;
use crate::iterators::TrustedIterator;
use crate::StrideShape;
#[cfg(feature = "std")]
use crate::{geomspace, linspace, logspace};
#[allow(unused_imports)]
use rawpointer::PointerExt;







impl<S, A> ArrayBase<S, Ix1>
where S: DataOwned<Elem = A>
{









    pub fn from_vec(v: Vec<A>) -> Self
    {
        if mem::size_of::<A>() == 0 {
            assert!(
                v.len() <= isize::MAX as usize,
                "Length must fit in `isize`.",
            );
        }
        unsafe { Self::from_shape_vec_unchecked(v.len() as Ix, v) }
    }










    #[allow(clippy::should_implement_trait)]
    pub fn from_iter<I: IntoIterator<Item = A>>(iterable: I) -> Self
    {
        Self::from_vec(iterable.into_iter().collect())
    }


















    #[cfg(feature = "std")]
    pub fn linspace(start: A, end: A, n: usize) -> Self
    where A: Float
    {
        Self::from(to_vec(linspace::linspace(start, end, n)))
    }












    #[cfg(feature = "std")]
    pub fn range(start: A, end: A, step: A) -> Self
    where A: Float
    {
        Self::from(to_vec(linspace::range(start, end, step)))
    }






















    #[cfg(feature = "std")]
    pub fn logspace(base: A, start: A, end: A, n: usize) -> Self
    where A: Float
    {
        Self::from(to_vec(logspace::logspace(base, start, end, n)))
    }




























    #[cfg(feature = "std")]
    pub fn geomspace(start: A, end: A, n: usize) -> Option<Self>
    where A: Float
    {
        Some(Self::from(to_vec(geomspace::geomspace(start, end, n)?)))
    }
}


impl<S, A> ArrayBase<S, Ix2>
where S: DataOwned<Elem = A>
{



    pub fn eye(n: Ix) -> Self
    where
        S: DataMut,
        A: Clone + Zero + One,
    {
        let mut eye = Self::zeros((n, n));
        for a_ii in eye.diag_mut() {
            *a_ii = A::one();
        }
        eye
    }












    pub fn from_diag<S2>(diag: &ArrayBase<S2, Ix1>) -> Self
    where
        A: Clone + Zero,
        S: DataMut,
        S2: Data<Elem = A>,
    {
        let n = diag.len();
        let mut arr = Self::zeros((n, n));
        arr.diag_mut().assign(diag);
        arr
    }












    pub fn from_diag_elem(n: usize, elem: A) -> Self
    where
        S: DataMut,
        A: Clone + Zero,
    {
        let mut eye = Self::zeros((n, n));
        for a_ii in eye.diag_mut() {
            *a_ii = elem.clone();
        }
        eye
    }
}

#[cfg(not(debug_assertions))]
#[allow(clippy::match_wild_err_arm)]
macro_rules! size_of_shape_checked_unwrap {
    ($dim:expr) => {
        match dimension::size_of_shape_checked($dim) {
            Ok(sz) => sz,
            Err(_) => {
                panic!("ndarray: Shape too large, product of non-zero axis lengths overflows isize")
            }
        }
    };
}

#[cfg(debug_assertions)]
macro_rules! size_of_shape_checked_unwrap {
    ($dim:expr) => {
        match dimension::size_of_shape_checked($dim) {
            Ok(sz) => sz,
            Err(_) => panic!(
                "ndarray: Shape too large, product of non-zero axis lengths \
                 overflows isize in shape {:?}",
                $dim
            ),
        }
    };
}

















impl<S, A, D> ArrayBase<S, D>
where
    S: DataOwned<Elem = A>,
    D: Dimension,
{




















    pub fn from_elem<Sh>(shape: Sh, elem: A) -> Self
    where
        A: Clone,
        Sh: ShapeBuilder<Dim = D>,
    {
        let shape = shape.into_shape_with_order();
        let size = size_of_shape_checked_unwrap!(&shape.dim);
        let v = vec![elem; size];
        unsafe { Self::from_shape_vec_unchecked(shape, v) }
    }




    pub fn zeros<Sh>(shape: Sh) -> Self
    where
        A: Clone + Zero,
        Sh: ShapeBuilder<Dim = D>,
    {
        Self::from_elem(shape, A::zero())
    }




    pub fn ones<Sh>(shape: Sh) -> Self
    where
        A: Clone + One,
        Sh: ShapeBuilder<Dim = D>,
    {
        Self::from_elem(shape, A::one())
    }




    pub fn default<Sh>(shape: Sh) -> Self
    where
        A: Default,
        Sh: ShapeBuilder<Dim = D>,
    {
        Self::from_shape_simple_fn(shape, A::default)
    }











    pub fn from_shape_simple_fn<Sh, F>(shape: Sh, mut f: F) -> Self
    where
        Sh: ShapeBuilder<Dim = D>,
        F: FnMut() -> A,
    {
        let shape = shape.into_shape_with_order();
        let len = size_of_shape_checked_unwrap!(&shape.dim);
        let v = to_vec_mapped(0..len, move |_| f());
        unsafe { Self::from_shape_vec_unchecked(shape, v) }
    }





















    pub fn from_shape_fn<Sh, F>(shape: Sh, f: F) -> Self
    where
        Sh: ShapeBuilder<Dim = D>,
        F: FnMut(D::Pattern) -> A,
    {
        let shape = shape.into_shape_with_order();
        let _ = size_of_shape_checked_unwrap!(&shape.dim);
        if shape.is_c() {
            let v = to_vec_mapped(indices(shape.dim.clone()).into_iter(), f);
            unsafe { Self::from_shape_vec_unchecked(shape, v) }
        } else {
            let dim = shape.dim.clone();
            let v = to_vec_mapped(indexes::indices_iter_f(dim), f);
            unsafe { Self::from_shape_vec_unchecked(shape, v) }
        }
    }


































    pub fn from_shape_vec<Sh>(shape: Sh, v: Vec<A>) -> Result<Self, ShapeError>
    where Sh: Into<StrideShape<D>>
    {

        Self::from_shape_vec_impl(shape.into(), v)
    }

    fn from_shape_vec_impl(shape: StrideShape<D>, v: Vec<A>) -> Result<Self, ShapeError>
    {
        let dim = shape.dim;
        let is_custom = shape.strides.is_custom();
        dimension::can_index_slice_with_strides(&v, &dim, &shape.strides, dimension::CanIndexCheckMode::OwnedMutable)?;
        if !is_custom && dim.size() != v.len() {
            return Err(error::incompatible_shapes(&Ix1(v.len()), &dim));
        }
        let strides = shape.strides.strides_for_dim(&dim);
        unsafe { Ok(Self::from_vec_dim_stride_unchecked(dim, strides, v)) }
    }

























    pub unsafe fn from_shape_vec_unchecked<Sh>(shape: Sh, v: Vec<A>) -> Self
    where Sh: Into<StrideShape<D>>
    {
        let shape = shape.into();
        let dim = shape.dim;
        let strides = shape.strides.strides_for_dim(&dim);
        Self::from_vec_dim_stride_unchecked(dim, strides, v)
    }

    unsafe fn from_vec_dim_stride_unchecked(dim: D, strides: D, mut v: Vec<A>) -> Self
    {

        debug_assert!(dimension::can_index_slice(&v, &dim, &strides, CanIndexCheckMode::OwnedMutable).is_ok());

        let ptr = nonnull_from_vec_data(&mut v).add(offset_from_low_addr_ptr_to_logical_ptr(&dim, &strides));
        ArrayBase::from_data_ptr(DataOwned::new(v), ptr).with_strides_dim(strides, dim)
    }







    pub(crate) unsafe fn from_shape_trusted_iter_unchecked<Sh, I, F>(shape: Sh, iter: I, map: F) -> Self
    where
        Sh: Into<StrideShape<D>>,
        I: TrustedIterator + ExactSizeIterator,
        F: FnMut(I::Item) -> A,
    {
        let shape = shape.into();
        let dim = shape.dim;
        let strides = shape.strides.strides_for_dim(&dim);
        let v = to_vec_mapped(iter, map);
        Self::from_vec_dim_stride_unchecked(dim, strides, v)
    }














































    pub fn uninit<Sh>(shape: Sh) -> ArrayBase<S::MaybeUninit, D>
    where Sh: ShapeBuilder<Dim = D>
    {
        unsafe {
            let shape = shape.into_shape_with_order();
            let size = size_of_shape_checked_unwrap!(&shape.dim);
            let mut v = Vec::with_capacity(size);
            v.set_len(size);
            ArrayBase::from_shape_vec_unchecked(shape, v)
        }
    }






















    pub fn build_uninit<Sh, F>(shape: Sh, builder: F) -> ArrayBase<S::MaybeUninit, D>
    where
        Sh: ShapeBuilder<Dim = D>,
        F: FnOnce(ArrayViewMut<MaybeUninit<A>, D>),
    {
        let mut array = Self::uninit(shape);

        unsafe {
            builder(array.raw_view_mut_unchecked().deref_into_view_mut());
        }
        array
    }
}
