use std::{
    env,
    io,
    path::{
        Path,
        PathBuf,
    }
};
use anyhow::{anyhow, Context, Result};
use clap::Parser;
use once_cell::sync::OnceCell;

use sequoia_openpgp::{
    self as openpgp,
    Cert,
    KeyHandle,
    parse::Parse,
};
use sequoia_cert_store::{
    CertStore,
    Store,
};
use sequoia_git::*;

#[macro_use]
mod macros;

mod cli;
use cli::CertArg;

mod commands;
mod output;
#[allow(dead_code)]
mod utils;

impl CertArg {
    fn get(&self, config: &Config) -> Result<Cert> {
        let filename;
        let r: Result<(&Path, Vec<u8>), KeyHandle> = if let Some(value) = &self.value {
            // First try to open as a file.  Only if the file does not
            // exist, interpret the value as a key handle.

            filename = PathBuf::from(value);
            match std::fs::read(&filename) {
                Ok(contents) => {
                    Ok((&filename, contents))
                }
                Err(err) => {
                    if err.kind() == std::io::ErrorKind::NotFound {
                        match value.parse::<KeyHandle>() {
                            Ok(kh) => Err(kh),
                            Err(err) => return Err(
                                err.context(
                                    format!("File {} does not exist, \
                                             and is not a valid fingerprint \
                                             or Key ID",
                                            filename.display()))),
                        }
                    } else {
                        return Err(anyhow::Error::from(err).context(
                            format!("Opening {}", filename.display())));
                    }
                }
            }
        } else if let Some(kh) = &self.cert_handle {
            Err(kh.clone())
        } else if let Some(filename) = &self.cert_file {
            let content = std::fs::read(&filename)
                .with_context(|| {
                    format!("Opening {}", filename.display())
                })?;
            Ok((filename, content))
        } else {
            unreachable!("clap ensures that one argument is set")
        };

        match r {
            Ok((filename, content)) => {
                // Parse content as a Cert and make sure content only
                // contains a single certificate.
                Cert::from_bytes(&content)
                    .with_context(|| {
                        format!("Parsing {}", filename.display())
                    })
            }
            Err(kh) => {
                let certs = config.cert_store()?.lookup_by_key(&kh)?;

                let cert = match certs.len() {
                    0 => return Err(anyhow!("Key {} not found", kh)),
                    1 => certs[0].to_cert()?.clone(),
                    n => return Err(anyhow!(
                        "Key {} is part of {} certs, use cert \
                         fingerprint to resolve",
                        kh, n)),
                };

                Ok(cert)
            }
        }
    }
}

pub struct Config<'a> {
    policy_file: Option<&'a Path>,
    output_format: output::Format,
    no_cert_store: bool,
    cert_store: Option<PathBuf>,
    cert_store_instance: OnceCell<CertStore<'a>>,
}

impl<'a> Config<'a> {
    // Returns the cert store.
    //
    // If it is not yet open, opens it.
    //
    // If it does not exist, issues a warning and returns an empty
    // cert store.
    fn cert_store(&self) -> Result<&CertStore<'a>> {
        self.cert_store_instance.get_or_try_init(|| {
            if self.no_cert_store {
                Ok(CertStore::empty())
            } else {
                let path = self.cert_store.clone()
                    .unwrap_or_else(|| cli::cert_store_base().into());

                match path.metadata() {
                    Ok(metadata) => {
                        if metadata.is_dir() {
                            CertStore::open(&path)
                        } else {
                            Err(anyhow::anyhow!(
                                "Not a certificate directory"))
                        }
                    }
                    Err(err) => {
                        if err.kind() == std::io::ErrorKind::NotFound {
                            eprintln!("Warning: no certificate \
                                       store found at {}",
                                      path.display());
                            Ok(CertStore::empty())
                        } else {
                            Err(err.into())
                        }
                    }
                }
                .with_context(|| {
                    format!("While opening the certificate store at {:?}",
                            &path)
                })
            }
        })
    }

    fn read_policy(&self) -> Result<Policy> {
        if let Some(path) = self.policy_file {
            Policy::read(path)
                .with_context(|| {
                    format!("Reading specified policy file: {}",
                            path.display())
                })
        } else {
            Policy::read_from_working_dir()
                .with_context(|| {
                    format!("Reading default policy file")
                })
        }
    }

    fn write_policy(&self, p: &Policy) -> Result<()> {
        if let Some(path) = self.policy_file {
            p.write(path)
                .with_context(|| {
                    format!("Updating the specified policy file: {}",
                            path.display())
                })
        } else {
            p.write_to_working_dir()
                .with_context(|| {
                    format!("Updating default policy file")
                })
        }
    }
}

// Returns the current git repository.
fn git_repo() -> Result<git2::Repository> {
    let cwd = env::current_dir()
        .context("Getting current working directory")?;
    let repo = git2::Repository::discover(&cwd)
        .with_context(|| {
            format!("Looking for git repository under {}",
                    cwd.display())
        })?;

    Ok(repo)
}

fn main() -> anyhow::Result<()> {
    let policy = openpgp::policy::StandardPolicy::new(); // XXX
    let cli = cli::Cli::parse();

    let config = Config {
        policy_file: cli.policy_file.as_deref(),
        output_format: cli.output_format.parse()?,
        no_cert_store: cli.no_cert_store,
        cert_store: cli.cert_store,
        cert_store_instance: Default::default(),
    };

    let commit_by_symbolic_name = |git: &git2::Repository, name: &str,
                                   trust_root: bool|
        -> Result<git2::Oid>
    {
        // Allow the zero oid.
        if let Ok(oid) = git2::Oid::from_str(name) {
            if oid.is_zero() {
                return Ok(oid);
            }
        }

        let (object, reference) = git.revparse_ext(name)
            .with_context(|| {
                format!("Looking up {:?}.", name)
            })?;

        // We won't get a reference if we are passed an OID.
        if let Some(reference) = reference {
            if trust_root {
                if reference.is_tag() {
                    eprintln!("Warning: using a tag as the trust root \
                               could allow the remote repository to \
                               manipulate your trust root.");
                } else if reference.is_remote() {
                    eprintln!("Warning: using a remote branch as the \
                               trust root could allow the remote repository \
                               to manipulate your trust root.");
                }
            }

            let commit = reference.peel_to_commit()
                .with_context(|| {
                    format!("{:?} is not a commit", name)
                })?;

            Ok(commit.id())
        } else {
            if let Ok(commit) = object.into_commit() {
                Ok(commit.id())
            } else {
                Err(anyhow::anyhow!("{:?} is not a commit", name))
            }
        }
    };

    let lookup_trust_root = |git: &git2::Repository,
                             trust_root: Option<&str>|
        -> Result<git2::Oid>
    {
        if let Some(trust_root) = trust_root {
            return commit_by_symbolic_name(git, trust_root, true);
        }

        // We only look in the repository's configuration file.
        let config = git.config()?
            .open_level(git2::ConfigLevel::Local)?
            .snapshot()?;
        let trust_root = match config.get_str("sequoia.trust-root") {
            Ok(trust_root) => trust_root,
            Err(err) => {
                if err.code() == git2::ErrorCode::NotFound {
                    eprintln!("Warning: no trust root specified.  Either \
                               pass the '--trust-root' option, or set \
                               the 'sequoia.trust-root' configuration \
                               key in your repository's local git config \
                               to reference a commit.");
                }

                return Err(anyhow::Error::from(err).context(
                    "Reading 'sequoia.trust-root' from the repository's \
                     git config."));
            }
        };

        commit_by_symbolic_name(git, trust_root, true)
    };

    match cli.subcommand {
        cli::Subcommand::Init(command) => {
            commands::init::dispatch(command)?;
        }

        cli::Subcommand::Policy { command } => {
            commands::policy::dispatch(&config, command)?;
        }

        cli::Subcommand::Log {
            trust_root,
            keep_going,
            prune_certs,
            commit_range,
        } => {
            if prune_certs && commit_range.is_some() && cli.policy_file.is_none() {
                return Err(anyhow!("--prune-certs can only modify \
                                    HEAD or a shadow policy"));
            }

            let git = git_repo()?;
            let trust_root = lookup_trust_root(&git, trust_root.as_deref())?;

            let shadow_p = if let Some(s) = &cli.policy_file {
                Some(std::fs::read(s)?)
            } else {
                None
            };
            let shadow_p = shadow_p.as_deref();

            let head = git.head()?.target().unwrap();
            let (start, target) = if let Some(commit_range) = commit_range {
                let mut s = commit_range.splitn(2, "..");
                let first = s.next().expect("always one component");
                if let Some(second) = s.next() {
                    if second.is_empty() {
                        (commit_by_symbolic_name(&git, first, false)?, head)
                    } else {
                        (commit_by_symbolic_name(&git, first, false)?,
                         commit_by_symbolic_name(&git, second, false)?)
                    }
                } else {
                    (trust_root, commit_by_symbolic_name(&git, first, false)?)
                }
            } else {
                (trust_root, head)
            };

            let mut cache = VerificationCache::new()?;

            let mut vresults = VerificationResult::default();
            let result = match config.output_format {
                output::Format::HumanReadable => {
                    verify(&git, trust_root, shadow_p,
                           (start, target),
                           &mut vresults,
                           keep_going,
                           |oid, parent_oid, result| {
                               output::Commit::new(
                                   &git, oid, parent_oid, &cli.policy_file, result)?
                                   .describe(&mut io::stdout())?;
                               Ok(())
                           },
                           &mut cache,
                    )
                },
                output::Format::Json => {
                    use serde::ser::{Serializer, SerializeSeq};
                    let mut serializer = serde_json::ser::Serializer::pretty(
                        std::io::stdout());
                    let mut seq = serializer.serialize_seq(None)?;
                    let r =
                        verify(&git, trust_root, shadow_p,
                               (start, target),
                               &mut vresults,
                               keep_going,
                               |oid, parent_oid, result| {
                                   seq.serialize_element(
                                       &output::Commit::new(
                                           &git, oid, parent_oid, &cli.policy_file, result)?
                                   ).map_err(anyhow::Error::from)?;
                                   Ok(())
                               },
                               &mut cache,
                        );
                    seq.end()?;
                    r
                },
            };

            if prune_certs {
                let mut p = config.read_policy()?;

                for a in p.authorization.values_mut() {
                    let certs =
                        a.certs()?
                        .map(|r| r.and_then(Cert::try_from))
                        .collect::<Result<Vec<_>>>()?;

                    a.set_certs_filter(
                        certs,
                        // Keep all subkeys that made a signature, and
                        // those that are alive now.
                        |sk| {
                            let fp = sk.fingerprint();
                            vresults.signer_keys.contains(&fp)
                                || {
                                    // Slightly awkward, because we
                                    // cannot use sk.with_policy.
                                    let c = sk.cert();

                                    c.with_policy(&policy, None)
                                        .map(|vka| vka.keys().key_handle(fp)
                                             .next().is_some())
                                        .unwrap_or(false)
                                }
                        },
                        // Keep all user IDs that were primary user
                        // IDs when a signature was made, and the ones
                        // that are the primary userid now.
                        |uid| vresults.primary_uids.contains(&uid)
                            || {
                                // Slightly awkward, because we
                                // cannot use sk.with_policy.
                                let c = uid.cert();

                                c.with_policy(&policy, None)
                                    .and_then(|vka| vka.primary_userid())
                                    .map(|u| u.userid() == uid.userid())
                                    .unwrap_or(false)
                            }
                    )?;
                }

                config.write_policy(&p)?;
            }

            let _ = cache.persist();
            result?;
        },

        cli::Subcommand::Verify {
            trust_root,
            signature,
            archive,
        } => {
            let git = git_repo()?;
            let policy = if let Some(s) = &cli.policy_file {
                Policy::read(s)
                    .with_context(|| {
                        format!("Reading specified policy file: {}",
                                s.display())
                    })?
            } else {
                let trust_root = lookup_trust_root(
                    &git, trust_root.as_deref())?;

                Policy::read_from_commit(&git, &trust_root)
                    .with_context(|| {
                        format!("Reading policy from commit {}",
                                trust_root)
                    })?
            };

            // XXX: In the future, mmap the data.
            let signature = std::fs::read(&signature)
                .with_context(|| {
                    format!("Reading signature data from {}",
                            signature.display())
                })?;
            let archive = std::fs::read(&archive)
                .with_context(|| {
                    format!("Reading archive data from {}",
                            archive.display())
                })?;

            let r = policy.verify_archive(signature, archive);
            let o = output::Archive::new(r)?;
            match config.output_format {
                output::Format::HumanReadable =>
                    o.describe(&mut io::stdout())?,
                output::Format::Json =>
                    serde_json::to_writer_pretty(io::stdout(), &o)?,
            }
        },

        cli::Subcommand::UpdateHook {
            trust_root,
            ref_name: _,
            old_object,
            new_object,
        } => {
            let git = git_repo()?;
            let trust_root = commit_by_symbolic_name(&git, &trust_root, true)
                .with_context(|| {
                    format!("Looking up specified trust root ({})",
                            trust_root)
                })?;
            let new_object = commit_by_symbolic_name(&git, &new_object, false)
                .with_context(|| {
                    format!("Looking up new object ({})",
                            new_object)
                })?;
            let old_object = commit_by_symbolic_name(&git, &old_object, false)
                .with_context(|| {
                    format!("Looking up old object ({})",
                            old_object)
                })?;

            // Fall back to the trust root if this is a new branch.
            let start = if let Err(err)
                = git_is_ancestor(&git, old_object, new_object)
            {
                if let Some(e) = err.downcast_ref::<sequoia_git::Error>() {
                    if matches!(e, Error::NoPathConnecting(_, _)) {
                        // There's no path from old object to new
                        // object.  Use the trust root.
                        trust_root
                    } else {
                        // Some other error: abort.
                        return Err(err);
                    }
                } else {
                    // There's a path from old object to new object.
                    old_object
                }
            } else {
                trust_root
            };

            let mut cache = VerificationCache::new()?;
            let mut vresults = VerificationResult::default();
            let result = verify(&git, trust_root,
                   None,
                   (start, new_object),
                   &mut vresults,
                   false,
                   |oid, parent_oid, result| {
                       output::Commit::new(
                           &git, oid, parent_oid, &cli.policy_file, result)?
                           .describe(&mut io::stdout())?;
                       Ok(())
                   },
                   &mut cache,
            );

            let _ = cache.persist();
            result?;
        },
    }
    Ok(())
}
