/*
 **************************************************************************
 *                                                                        *
 *                            N O T I C E                                 *
 *                                                                        *
 *               Copyright Abandoned, 1987, Fred Fish                     *
 *                                                                        *
 *                                                                        *
 * This previously copyrighted work has been placed into the  public      *
 * domain  by  the  author  and  may be freely used for any purpose,      *
 * private or commercial.                                                 *
 *                                                                        *
 * Because of the number of inquiries I was receiving about the  use      *
 * of this product in commercially developed works I have decided to      *
 * simply make it public domain to further its unrestricted use.   I      *
 * specifically  would  be  most happy to see this material become a      *
 * part of the standard Unix distributions by AT&T and the  Berkeley      *
 * Computer  Science  Research Group, and a standard part of the GNU      *
 * system from the Free Software Foundation.                              *
 *                                                                        *
 * I would appreciate it, as a courtesy, if this notice is  left  in      *
 * all copies and derivative works.  Thank you.                           *
 *                                                                        *
 * The author makes no warranty of any kind  with  respect  to  this      *
 * product  and  explicitly disclaims any implied warranties of mer-      *
 * chantability or fitness for any particular purpose.                    *
 *                                                                        *
 **************************************************************************
 */

/*
 *  FILE
 *
 * dbug.c   runtime support routines for dbug package
 *
 *  RCS
 *
 * $Id: dbug.c 1.19 2000/11/15 23:01:28 federico Exp $
 *
 * $Log: dbug.c $
 * Revision 1.19  2000/11/15 23:01:28  federico
 * Removed vmprintf.c: it was an unused static function.
 * Removed a wrong #endif
 *
 * Revision 1.18  2000/11/15 22:59:17  federico
 * .
 *
 * Revision 1.17  2000/11/14 21:52:13  federico
 * When pushing a state with the 'g' character set for profiling,
 * check if profiling was already enabled in order to keep
 * the profile produced so far.
 * Moreover I've added fflush calls after writing to the profile
 * log.
 * case 'O' lacked the break statement in _Db_push_:fixed.
 * removed a call to IDBUG_FREE in _db_push
 * removed a call to IDBUG_FREE in FreeList
 * added an fflusu call in OpenProfile.
 *
 * Revision 1.16  2000/10/30 23:00:54  federico
 * Fixed db_init_as: now the security field is copied.
 *
 * Revision 1.15  2000/10/30 22:52:31  federico
 * .
 *
 * Revision 1.14  2000/10/24 21:59:17  federico
 * Added a default sanity function (DefaultSanity)
 * Added a correct initialization for the sanity field.
 * StrDup doesn't duplicate the string anymore.
 * Delay arg doesn't divide by 10 anymore the value passed.
 * Fixed implementation of dbug_delay.
 *
 * Revision 1.13  2000/10/08 11:30:42  federico
 * Cosmetics and a little fix to _db_push_
 *
 * Revision 1.12  2000/08/09 11:57:54  federico
 * Added int2str and restored DBUG_DUMP functionality.
 *
 * Revision 1.11  2000/08/09 10:52:19  federico
 * Changed DelayArg
 * Now aborts on failed malloc.
 * Cosmetics.
 *
 * Revision 1.10  2000/08/07 18:09:05  federico
 * Mainly cosmetic changes apart from a fix in _db_init_
 * Renamed _db_clone_... into _db_init_as_
 *
 * Revision 1.9  2000/08/06 00:46:43  federico
 * Added idbug_*_t for thread management,
 * and ?.
 *
 * Revision 1.8  2000/07/23 19:51:12  federico
 * Fixed division by zero in Delay.
 * Removed initialization code from global variables (they default to zero
 * as they are globals).
 *
 * Revision 1.7  2000/07/23 11:47:57  federico
 * Apparently fixed bug regarding sanity checking.
 *
 * Revision 1.6  2000/07/21 21:35:56  federico
 * Writable returns IDBUG_TRUE always.
 * Changed indentation.
 *
 * Revision 1.5  2000/07/16 10:45:20  federico
 * Added additional idbug_t parameter to all the neede function.
 * Still uses the old globals.
 * Some #if/#ifdef still remain.
 * Ciao.
 *
 * Revision 1.2  2000/07/12 22:56:53  federico
 * NOT WORKING.
 *
 *
 *  DESCRIPTION
 *
 * These are the runtime support routines for the dbug package.
 * The dbug package has two main components; the user include
 * file containing various macro definitions, and the runtime
 * support routines which are called from the macro expansions.
 *
 * Externally visible functions in the runtime support module
 * use the naming convention pattern "_db_xx...xx_", thus
 * they are unlikely to collide with user defined function names.
 *
 *  AUTHOR(S)
 *
 * Fred Fish  (base code)
 * Enhanced Software Technologies, Tempe, AZ
 * asuvax!mcdphx!estinc!fnf
 *
 * Binayak Banerjee (profiling enhancements)
 * seismo!bpa!sjuvax!bbanerje
 *
 * Michael Widenius:
 * DBUG_DUMP  - To dump a pice of memory.
 * PUSH_FLAG "O" - To be used insted of "o" if we don't
 *     want flushing (for slow systems)
 * PUSH_FLAG "S" - check of malloc on entry/exit
 *
 */

/*
 * Federico Spinazzi's own notes:
 * NOTES:
 * - threads:
 *  - add a specific field to idbug_t
 *  - no profiling;
 *  - a thread-specific key for the state !!!;
 *  - a thread identifier;
 *  - locking (where?: befor output function, usually lock released
 *    in dbug_flush)
 * TEST:
 * - TEST: longjmp/setjmp: the DBUG_ macros should work: nude
 *   setjmp/longjmp don't
 * - TEST: S debugging flags
 * - indipendently test multiple threads on the same output file
 * TODO:
 * - update _db_mprintf with the one in the mprintf directory ...
 * - UNICODE
 * - document that DBUG_PUSH allocates memory and DBUG_POP release it
 * - OS specific flags ?
 * - make UNIX_SECURITY (SUID, ...), WIN32_SECURITY, NO_SECURITY (default)
 *   maybe that could be a target for a symlink attack (I really don't
 *   understand well why there is a ChangeOwner function ... )?
 * - UNIX/DOS/WIN directory separator
 * - do platform-dependent : exists, writable, changeower
 * - getpid upon init
 * - setuid manipulation in ChangeOwner ? Add a security field to reproduce
 *   UNIX chown behaviour ?
 */

#ifdef IDBUG_OFF
#	undef IDBUG_OFF
#endif

#define IDBUG_IMPLEMENTATION

#include "dbug.h"

#define _db_code_state(a) ((a)->state)

#ifndef max
#define max(a,b) ((a) > (b) ? (a) : (b))
#endif

#ifndef min
#define min(a,b) ((a) < (b) ? (a) : (b))
#endif

#if defined (IDBUG_OS_W32) || defined (IDBUG_OS_MSDOS)
#	define FN_LIBCHAR '\\'
#else
#	if defined (IDBUG_OS_UNIX)
# define FN_LIBCHAR '/'
# endif
#endif


#if defined(NOT_ACCESS)
#	define EXISTS(pathname) (FALSE)  /* Assume no existance */
#	define Writable(name) (TRUE)
#else
#	define EXISTS(pathname)	 (access (pathname, F_OK) == 0)
#	define WRITABLE(pathname)	 (access (pathname, W_OK) == 0)
#endif /* !defined(NOT_ACCESS) */

/*
 *       Manifest constants that should not require any changes.
 */

char *_DBUG_START_CONDITION_ = "";


/*
 *       Manifest constants which may be "tuned" if desired.
 */

enum
  {
    PRINTBUF = 1024,            /* Print buffer size */
    INDENT = 2,                 /* Indentation per trace level */
    MAXDEPTH = 200              /* Maximum trace depth default */
  };

/*
 * The following flags are used to determine which
 * capabilities the user has enabled with the state
 * push macro.
 */


#define TRACING (idbug->stack -> flags & _IDBUG_TRACE_ON)
#define DEBUGGING (idbug->stack -> flags & _IDBUG_DEBUG_ON)
#define PROFILING (idbug->stack -> flags & _IDBUG_PROFILE_ON)
#define STREQ(a,b) (strcmp(a,b) == 0)

static const char EOS = '\000';

unsigned long int
_db_ansi_process_id (void * foo)
{
  return 1;
}

/*
 * GLOBAL INTERFACE
 */
static idbug_t _g_dbug_state;
idbug_t *g_idbug_state = &_g_dbug_state;
idbug_thread_params_t idbug_no_thread_params;
idbug_process_params_t idbug_ansi_process_params =
	{ "", _db_ansi_process_id, NULL };

/*
 * The default file for profiling.  Could also add another flag
 * (G?) which allowed the user to specify this.
 *
 * If the automatic variables get allocated on the stack in
 * reverse order from their declarations, then define AUTOS_REVERSE.
 * This is used by the code that keeps track of stack usage.  For
 * forward allocation, the difference in the dbug frame pointers
 * represents stack used by the callee function.  For reverse allocation,
 * the difference represents stack used by the caller function.
 *
 */

static const char *PROF_FILE = "dbugmon.out";
static const char *PROF_EFMT = "E\t%ld\t%s\n";
static const char *PROF_SFMT = "S\t%p\t%lx\t%s\n";
static const char *PROF_XFMT = "X\t%ld\t%s\n";


/*
 * The user may specify a list of functions to trace or
 * debug.  These lists are kept in a linear linked list,
 * a very simple implementation.
 */

struct link
  {
    char *str;                  /* Pointer to link's contents */
    struct link *next_link;     /* Pointer to the next link */
  };

/*
 * Debugging states can be pushed or popped off of a
 * stack which is implemented as a linked list.  Note
 * that the head of the list is the current state and the
 * stack is pushed by adding a new state to the head of the
 * list or popped by removing the first link.
 */

 /*
  *  ONLY IF THREADED
  */
 /* Open profile output stream */
static FILE *OpenProfile (idbug_t * idbug, const char *name);
 /* Profile if asked for it */
static int DoProfile (idbug_t * idbug);
 /* Return current user time (ms) */
static unsigned long Clock (void);
 /* Parse a debug command string */
static struct link *ListParse (idbug_t * idbug, char *ctlp);
 /* Make a fresh copy of a string */
static char *StrDup (idbug_t * idbug, char *str);
 /* Open debug output stream */
static void DBUGOpenFile (idbug_t * idbug, char *name);
 /* Close debug output stream */
static void CloseFile (idbug_t * idbug, FILE * fp);
 /* Push current debug state */
static void PushState (idbug_t * idbug);
 /* Test for tracing enabled */
static int DoTrace (idbug_t * idbug, idbug_code_state_t * state);
 /* Remove leading pathname components */
static char *BaseName (char *pathname);
static int DoPrefix (idbug_t * idbug, unsigned int line);
static void FreeList (idbug_t * idbug, struct link *linkp);
static void Indent (idbug_t * idbug, int indent);
static int InList (struct link *linkp, char *cp);
static void dbug_flush (idbug_t * idbug);
static int dbug_delay (int ticks);
static int DelayArg (int value);
static int Writable (idbug_t * idbug, const char *pathname);
static void ChangeOwner (idbug_t * idbug, const char *pathname);
static int Exists (const char *filename);
static char *int2str(long int val, char *dst, int radix);

/*
 * Miscellaneous printf format strings.
 */
static const char *ERR_MISSING_RETURN = "%s: missing DBUG_RETURN or DBUG_VOID_RETURN macro in function \"%s\"\n";
static const char *ERR_OPEN = "%s: can't open debug output stream \"%s\": ";
static const char *ERR_CLOSE = "%s: can't close debug file: ";
static const char *ERR_CHOWN = "%s: can't change owner/group of \"%s\": ";
/*
 * static const char *ERR_ABORT = "%s: debugger aborting because %s\n";
 */

/*
 * #define _db_code_state() (&static_code_state)
 * const static code_state_t static_code_state =
 * {0, 0, "?func", "?file", NULL, 0, NULL, NULL, 0, "?"};
 */


/*
 * reentrant strtok: are we sure it is what is needed ?
 */
/*
 *  FUNCTION
 *
 *  dbug_strtok - reentrant strtok
 *
 *  SYNOPSIS
 *
 *  static char * dbug_strtok (_s, delim)
 *  char ** _s
 *  char delim
 *
 *  DESCRIPTION
 *
 *  like strtok, but reentrant
 *
 *  Example:
 *  while ((p = dbug_strtok (&string, ':') != NULL)
 *    puts (p);
 *
 */
static char *
dbug_strtok (char **_s, char delim)
{
  char *s = *_s;
  char *tok;

  if (s == NULL)
    return NULL;

  /*
   * Skip (span) leading delimiters
   */
  if (*s == delim)
    do
      {
        s++;
      }
    while (*s != EOS && *s == delim);

  if (*s == EOS)
    return NULL;

  tok = s;
  /*
   * Skip non-delimiters
   */
  while (*s != EOS && *s++ != delim)
    ;

  if (*s != EOS)
    {
      *_s = s;
      s[-1] = EOS;
    }
  else
    {
      *_s = NULL;
      if (s[-1] == delim)
        s[-1] = EOS;
    }
  return tok;
}

static void
dbug_free (idbug_t * idbug, void *p, const char *file, size_t line)
{
  free (p);
}

static void *
dbug_malloc (idbug_t * idbug, size_t s, const char *file, size_t line)
{
  void *p = NULL;

  p = malloc (s);

  if (p == NULL)
    {
      fprintf (idbug->_db_fp_, "\ndbug: out of memory: exiting\n");
      fprintf (idbug->_db_fp_, "file \"%s\", line %ld\n", file, line);
      exit (EXIT_FAILURE);
    }
  return p;
}


static int
DefaultSanity (const char *str, unsigned int u, void *ptr)
{
  fputs ("dbug: DefaultSanity called\n", stderr);
  return 0;
}

/*
   code_state_t *
   _db_code_state (idbug_t * idbug)
   {
   return idbug->state;
   }
 */
/*
 *  FUNCTION
 *
 *  _db_init initialize dbug library
 *
 *  SYNOPSIS
 *
 *  void _db_init_ (idbug_t * idbug, unsigned int flags)
 *
 *  DESCRIPTION
 *
 *  initialize some variables needed by dbug library:
 *  set NULL values if the fields are unused.
 *  Always returns IDBUG_TRUE
 *
 *  FLAGS:
 *   DBUG_THREADED
 *   DBUG_SANITY
 *
 */

EXPORT int
_db_init_ (idbug_t * idbug, unsigned int flags,
           const idbug_process_params_t * process_params,
           const idbug_sanity_t sanity, void *opaque)
{

  void idbug_init_security (idbug_t *);

  assert (idbug != NULL);
  assert (process_params != NULL);
  assert (idbug_no_thread_params.mutex == NULL);
  assert (idbug_no_thread_params.lock_mutex == NULL);
  assert (idbug_no_thread_params.unlock_mutex == NULL);
  assert (idbug_no_thread_params.id == NULL);

  idbug->_db_fp_ = stderr;
  idbug->_db_process_ = process_params->name;

  idbug->_db_pfp_ = NULL;
  idbug->_db_on_ = IDBUG_FALSE;
  idbug->_db_pon_ = IDBUG_FALSE;
  idbug->_no_db_ = IDBUG_FALSE;


  idbug->stack = &idbug->_stack;
  idbug->state = &idbug->_state;

  /*
   * init state
   */
  idbug->stack->flags = 0;
  idbug->stack->delay = 0;
  idbug->stack->maxdepth = MAXDEPTH;
  idbug->stack->sub_level = 0;
  idbug->stack->out_file = stderr;
  idbug->stack->prof_file = (FILE *) 0;
  idbug->stack->functions = NULL;
  idbug->stack->p_functions = NULL;
  idbug->stack->keywords = NULL;
  idbug->stack->processes = NULL;

  /*
   * what is used for ?
   */
  idbug->get_process_id = process_params->id;

  /*
   * only with threads
   */
  if (flags & IDBUG_THREADED)
    {
      idbug->mutex = process_params->thread_params->mutex;
      idbug->lock_mutex = process_params->thread_params->lock_mutex;
      idbug->unlock_mutex = process_params->thread_params->unlock_mutex;

      idbug->get_thread_id = process_params->thread_params->id;
    }
  else
    {
      idbug->mutex = idbug->lock_mutex = idbug->unlock_mutex = NULL;
      idbug->get_thread_id = NULL;
    }


  /*
   * whenever entering a DBUG call
   */
  if (flags & IDBUG_SANITY)
    {
      assert (sanity != NULL);
      idbug->sanity = sanity;
    }
  else
    {
      idbug->sanity = DefaultSanity;
    }


  idbug->opaque = opaque;

  idbug->flags = flags;

  idbug_init_security (idbug);

  idbug->init_done = IDBUG_TRUE;

  return IDBUG_TRUE;
}


/*
 *  FUNCTION
 *
 * _db_push_ push current debugger state and set up new one
 *
 *  SYNOPSIS
 *
 * VOID _db_push_ (control)
 * char *control;
 *
 *  DESCRIPTION
 *
 * Given pointer to a debug control string in "control", pushes
 * the current debug state, parses the control string, and sets
 * up a new debug state.
 *
 * The only attribute of the new state inherited from the previous
 * state is the current function nesting level.  This can be
 * overridden by using the "r" flag in the control string.
 *
 * The debug control string is a sequence of colon separated fields
 * as follows:
 *
 *  <field_1>:<field_2>:...:<field_N>
 *
 * Each field consists of a mandatory flag character followed by
 * an optional "," and comma separated list of modifiers:
 *
 *  flag[,modifier,modifier,...,modifier]
 *
 * The currently recognized flag characters are:
 *
 *  d Enable output from DBUG_<N> macros for
 *   for the current state.  May be followed
 *   by a list of keywords which selects output
 *   only for the DBUG macros with that keyword.
 *   A null list of keywords implies output for
 *   all macros.
 *
 *  D Delay after each debugger output line.
 *   The argument is the number of tenths of seconds
 *   to delay, subject to machine capabilities.
 *   I.E.  -#D,20 is delay two seconds.
 *
 *  f Limit debugging and/or tracing, and profiling to the
 *   list of named functions.  Note that a null list will
 *   disable all functions.  The appropriate "d" or "t"
 *   flags must still be given, this flag only limits their
 *   actions if they are enabled.
 *
 *  F Identify the source file name for each
 *   line of debug or trace output.
 *
 *  i Identify the process with the pid for each line of
 *   debug or trace output.
 *
 *  g Enable profiling.  Create a file called 'dbugmon.out'
 *   containing information that can be used to profile
 *   the program.  May be followed by a list of keywords
 *   that select profiling only for the functions in that
 *   list.  A null list implies that all functions are
 *   considered.
 *
 *  L Identify the source file line number for
 *   each line of debug or trace output.
 *
 *  n Print the current function nesting depth for
 *   each line of debug or trace output.
 *
 *  N Number each line of dbug output.
 *
 *  o Redirect the debugger output stream to the
 *   specified file.  The default output is stderr.
 *
 *  O As O but the file is really flushed between each
 *   write. When neaded the file is closed and reopened
 *   between each write.
 *
 *  p Limit debugger actions to specified processes.
 *   A process must be identified with the
 *   DBUG_PROCESS macro and match one in the list
 *   for debugger actions to occur.
 *
 *  P Print the current process name for each
 *   line of debug or trace output.
 *
 *  r When pushing a new state, do not inherit
 *   the previous state's function nesting level.
 *   Useful when the output is to start at the
 *   left margin.
 *
 *  S Do function _sanity(_file_,_line_) at each
 *   debugged function until _sanity() returns
 *   something that differs from 0.
 *   (Moustly used with safemalloc)
 *
 *  t Enable function call/exit trace lines.
 *   May be followed by a list (containing only
 *   one modifier) giving a numeric maximum
 *   trace level, beyond which no output will
 *   occur for either debugging or tracing
 *   macros.  The default is a compile time
 *   option.
 *
 * Some examples of debug control strings which might appear
 * on a shell command line (the "-#" is typically used to
 * introduce a control string to an application program) are:
 *
 *  -#d:t
 *  -#d:f,main,subr1:F:L:t,20
 *  -#d,input,output,files:n
 *
 * For convenience, any leading "-#" is stripped off.
 *
 */

EXPORT int
_db_push_ (idbug_t * idbug, char *_control)
{
  char *scan;
  struct link *temp;
  idbug_code_state_t *state;
  char *new_str;
  char *control = _control;

  if (!idbug->_db_fp_)
    idbug->_db_fp_ = stderr;    /* Output stream, default stderr */

  if (control != NULL && *control == '-')
    {
      if (*++control == '#')
        control++;
    }

  new_str = StrDup (idbug, control);

  if (new_str == NULL)
    {
      return IDBUG_FALSE;
    }

  if (*control)
    idbug->_no_db_ = 0;         /* We are using dbug after all */

  PushState (idbug);
  state = _db_code_state (idbug);

  scan = dbug_strtok (&new_str, ':');
  for (; scan != NULL; scan = dbug_strtok (&new_str, ':'))
    {
      switch (*scan++)
        {
        case 'd':
          idbug->_db_on_ = IDBUG_TRUE;
          idbug->stack->flags |= _IDBUG_DEBUG_ON;
          if (*scan++ == ',')
            {
              idbug->stack->keywords = ListParse (idbug, scan);
            }
          break;
        case 'D':
          idbug->stack->delay = 0;
          if (*scan++ == ',')
            {
              temp = ListParse (idbug, scan);
              idbug->stack->delay = DelayArg (atoi (temp->str));
              FreeList (idbug, temp);
            }
          break;
        case 'f':
          if (*scan++ == ',')
            {
              idbug->stack->functions = ListParse (idbug, scan);
            }
          break;
        case 'F':
          idbug->stack->flags |= _IDBUG_FILE_ON;
          break;
        case 'i':
          idbug->stack->flags |= _IDBUG_PID_ON;
          break;
        case 'g':
          if (!(idbug->flags & IDBUG_THREADED))
            {
              if (idbug->_db_pon_ == IDBUG_FALSE)
                {
                  idbug->_db_pon_ = IDBUG_TRUE;
                  if (OpenProfile (idbug, PROF_FILE))
                    {
                      idbug->stack->flags |= _IDBUG_PROFILE_ON;
                      if (*scan++ == ',')
                        idbug->stack->p_functions = ListParse (idbug, scan);
                    }
                }
            }
          break;
        case 'L':
          idbug->stack->flags |= _IDBUG_LINE_ON;
          break;
        case 'n':
          idbug->stack->flags |= _IDBUG_DEPTH_ON;
          break;
        case 'N':
          idbug->stack->flags |= _IDBUG_NUMBER_ON;
          break;
        case 'O':
          idbug->stack->flags |= _IDBUG_FLUSH_ON_WRITE;
          break;
        case 'o':
          if (*scan++ == ',')
            {
              temp = ListParse (idbug, scan);
              DBUGOpenFile (idbug, temp->str);
              FreeList (idbug, temp);
            }
          else
            {
              DBUGOpenFile (idbug, "-");
            }
          break;
        case 'p':
          if (*scan++ == ',')
            {
              idbug->stack->processes = ListParse (idbug, scan);
            }
          break;
        case 'P':
          idbug->stack->flags |= _IDBUG_PROCESS_ON;
          break;
        case 'r':
          idbug->stack->sub_level = state->level;
          break;
        case 't':
          idbug->stack->flags |= _IDBUG_TRACE_ON;
          if (*scan++ == ',')
            {
              temp = ListParse (idbug, scan);
              idbug->stack->maxdepth = atoi (temp->str);
              FreeList (idbug, temp);
            }
          break;
        case 'S':
          idbug->stack->flags |= IDBUG_SANITY;
          break;
        }
    }

  return IDBUG_TRUE;
}


/*
 *  FUNCTION
 *
 * _db_pop_    pop the debug idbug->stack
 *
 *  DESCRIPTION
 *
 * Pops the debug idbug->stack, returning the debug state to its
 * condition prior to the most recent _db_push_ invocation.
 * Note that the pop will fail if it would remove the last
 * valid state from the idbug->stack.  This prevents user errors
 * in the push/pop sequence from screwing up the debugger.
 * Maybe there should be some kind of warning printed if the
 * user tries to pop too many states.
 *
 */

EXPORT void
_db_pop_ (idbug_t * idbug)
{
  struct idbug_state *discard;
  discard = idbug->stack;
  if (discard != NULL && discard->next_state != NULL)
    {
      idbug->stack = discard->next_state;
      idbug->_db_fp_ = idbug->stack->out_file;
      idbug->_db_pfp_ = idbug->stack->prof_file;
      if (discard->keywords != NULL)
        {
          FreeList (idbug, discard->keywords);
        }
      if (discard->functions != NULL)
        {
          FreeList (idbug, discard->functions);
        }
      if (discard->processes != NULL)
        {
          FreeList (idbug, discard->processes);
        }
      if (discard->p_functions != NULL)
        {
          FreeList (idbug, discard->p_functions);
        }
      CloseFile (idbug, discard->out_file);
      if (discard->prof_file)
        CloseFile (idbug, discard->prof_file);
      IDBUG_FREE (idbug, (char *) discard);
    }
}


/*
 *  FUNCTION
 *
 * _db_enter_    process entry point to user function
 *
 *  SYNOPSIS
 *
 * VOID _db_enter_ (_func_, _file_, _line_,
 *    _sfunc_, _sfile_, _slevel_, _sframep_)
 * char *_func_;  points to current function name
 * char *_file_;  points to current file name
 * int _line_;  called from source line number
 * char **_sfunc_;  save previous _func_
 * char **_sfile_;  save previous _file_
 * int *_slevel_;  save previous nesting level
 * char ***_sframep_; save previous frame pointer
 *
 *  DESCRIPTION
 *
 * Called at the beginning of each user function to tell
 * the debugger that a new function has been entered.
 * Note that the pointers to the previous user function
 * name and previous user file name are stored on the
 * caller's stack (this is why the ENTER macro must be
 * the first "executable" code in a function, since it
 * allocates these storage locations).  The previous nesting
 * level is also stored on the callers stack for internal
 * self consistency checks.
 *
 * Also prints a trace line if tracing is enabled and
 * increments the current function nesting depth.
 *
 * Note that this mechanism allows the debugger to know
 * what the current user function is at all times, without
 * maintaining an internal stack for the function names.
 *
 */

EXPORT void
_db_enter_ (idbug_t * idbug,
            const char *_func_, const char *_file_, unsigned int _line_,
            char **_sfunc_, char **_sfile_, unsigned int *_slevel_,
            char ***_sframep_)
{
  idbug_code_state_t *state;

  if (!idbug->_no_db_)
    {
      if (!idbug->init_done)
        _db_push_ (idbug, _DBUG_START_CONDITION_);
      state = _db_code_state (idbug);

      *_sfunc_ = (char *) state->func;
      *_sfile_ = state->file;
      state->func = (char *) _func_;
      state->file = (char *) _file_;  /* BaseName takes time !! */
      *_slevel_ = ++state->level;

      if (!(idbug->flags & IDBUG_THREADED))
        {
          *_sframep_ = state->framep;
          state->framep = (char **) _sframep_;
          if (DoProfile (idbug))
            {
              long stackused;
              if (*state->framep == NULL)
                {
                  stackused = 0;
                }
              else
                {
                  stackused = ((long) (*state->framep))
                    - ((long) (state->framep));
                  stackused = stackused > 0 ? stackused : -stackused;
                }
              (void) fprintf (idbug->_db_pfp_, PROF_EFMT, Clock (),
                              state->func);
              (void) fflush (idbug->_db_pfp_);

              if (idbug->flags & IDBUG_AUTOS_REVERSE)
                {
                  (void) fprintf (idbug->_db_pfp_, PROF_SFMT, state->framep,
                                  stackused, *_sfunc_);
                }
              else
                {
                  (void) fprintf (idbug->_db_pfp_, PROF_SFMT, state->framep,
                                  stackused, state->func);
                }
              (void) fflush (idbug->_db_pfp_);
            }
        }
      if (DoTrace (idbug, state))
        {
          if (idbug->flags & IDBUG_THREADED)
            idbug->lock_mutex (idbug->mutex);

          DoPrefix (idbug, _line_);
          Indent (idbug, state->level);
          (void) fprintf (idbug->_db_fp_, ">%s\n", state->func);
          /* This does a unlock */
          dbug_flush (idbug);
        }
      if (idbug->stack->flags & IDBUG_SANITY)
        {
          if (idbug->sanity (_file_, _line_, idbug->opaque))
            idbug->stack->flags &= ~IDBUG_SANITY;
        }
    }
}

/*
 *  FUNCTION
 *
 * _db_return_    process exit from user function
 *
 *  SYNOPSIS
 *
 * VOID _db_return_ (_line_, _sfunc_, _sfile_, _slevel_)
 * int _line_;  current source line number
 * char **_sfunc_;  where previous _func_ is to be retrieved
 * char **_sfile_;  where previous _file_ is to be retrieved
 * int *_slevel_;  where previous level was stashed
 *
 *  DESCRIPTION
 *
 * Called just before user function executes an explicit or implicit
 * return.  Prints a trace line if trace is enabled, decrements
 * the current nesting level, and restores the current function and
 * file names from the defunct function's stack.
 *
 */

EXPORT void
_db_return_ (idbug_t * idbug,
             unsigned int _line_, char **_sfunc_, char **_sfile_,
             unsigned int *_slevel_)
{
  idbug_code_state_t *state;

  if (!idbug->_no_db_)
    {
      if (!idbug->init_done)
        _db_push_ (idbug, "");

      state = _db_code_state (idbug);
      assert (state != NULL);

      if (idbug->stack->flags & (_IDBUG_TRACE_ON | _IDBUG_DEBUG_ON | _IDBUG_PROFILE_ON))
        {
          if (idbug->flags & IDBUG_THREADED)
            idbug->lock_mutex (idbug->mutex);

          if (state->level != (int) *_slevel_)
            (void) fprintf (idbug->_db_fp_, ERR_MISSING_RETURN,
                            idbug->_db_process_, state->func);
          else
            {
              if (idbug->stack->flags & IDBUG_SANITY)
                if (idbug->sanity (*_sfile_, _line_, idbug->opaque))
                  idbug->stack->flags &= ~IDBUG_SANITY;

              if (!(idbug->flags & IDBUG_THREADED))
                {
                  if (DoProfile (idbug))
                    {
                      (void) fprintf (idbug->_db_pfp_, PROF_XFMT,
                                      Clock (), state->func);
                      (void) fflush (idbug->_db_pfp_);
                    }
                }
              if (DoTrace (idbug, state))
                {
                  DoPrefix (idbug, _line_);
                  Indent (idbug, state->level);
                  (void) fprintf (idbug->_db_fp_, "<%s\n", state->func);
                }
            }
          dbug_flush (idbug);
        }
      state->level = *_slevel_ - 1;
      state->func = *_sfunc_;
      state->file = *_sfile_;
      if (!(idbug->flags & IDBUG_THREADED))
        {
          if (state->framep != NULL)
            state->framep = (char **) *state->framep;
        }
    }
}


/*
 *  FUNCTION
 *
 * _db_pargs_    log arguments for subsequent use by _db_doprnt_()
 *
 *  SYNOPSIS
 *
 * VOID _db_pargs_ (_line_, keyword)
 * int _line_;
 * char *keyword;
 *
 *  DESCRIPTION
 *
 * The new universal printing macro DBUG_PRINT, which replaces
 * all forms of the DBUG_N macros, needs two calls to runtime
 * support routines.  The first, this function, remembers arguments
 * that are used by the subsequent call to _db_doprnt_().
 *
 */

EXPORT void
_db_pargs_ (idbug_t * idbug, unsigned int _line_, const char *keyword)
{
  idbug_code_state_t *state = _db_code_state (idbug);
  state->u_line = _line_;
  state->u_keyword = (char *) keyword;
}


/*
 *  FUNCTION
 *
 * _db_doprnt_    handle print of debug lines
 *
 *  SYNOPSIS
 *
 * VOID _db_doprnt_ (format, va_alist)
 * char *format;
 * va_dcl;
 *
 *  DESCRIPTION
 *
 * When invoked via one of the DBUG macros, tests the current keyword
 * set by calling _db_pargs_() to see if that macro has been selected
 * for processing via the debugger control string, and if so, handles
 * printing of the arguments via the format string.  The line number
 * of the DBUG macro in the source is found in u_line.
 *
 * Note that the format string SHOULD NOT include a terminating
 * newline, this is supplied automatically.
 *
 */

EXPORT void
_db_doprnt_ (idbug_t * idbug, const char *format,...)
{
  va_list args;
  idbug_code_state_t *state;
  state = _db_code_state (idbug);

  va_start (args, format);

  if (_db_keyword_ (idbug, state->u_keyword))
    {
      if (idbug->flags & IDBUG_THREADED)
        idbug->lock_mutex (idbug->mutex);

      DoPrefix (idbug, state->u_line);
      if (TRACING)
        {
          Indent (idbug, state->level + 1);
        }
      else
        {
          (void) fprintf (idbug->_db_fp_, "%s: ", state->func);
        }
      (void) fprintf (idbug->_db_fp_, "%s: ", state->u_keyword);
      (void) vfprintf (idbug->_db_fp_, format, args);
      va_end (args);
      (void) fputc ('\n', idbug->_db_fp_);
      dbug_flush (idbug);
    }
  va_end (args);
}


/*
 *  FUNCTION
 *
 *       _db_dump_    dump a string until '\0' is found
 *
 *  SYNOPSIS
 *
 *       void _db_dump_ (_line_,keyword,memory,length)
 *       int _line_;  current source line number
 *       char *keyword;
 *       char *memory;  Memory to print
 *       int length;  Bytes to print
 *
 *  DESCRIPTION
 *  Dump N characters in a binary array.
 *  Is used to examine corrputed memory or arrays.
 */


EXPORT void
_db_dump_ (idbug_t * idbug, unsigned int _line_, const char *keyword,
           const char *memory, unsigned int length)
{
  int pos;
  char dbuff[90], *strpos;
  idbug_code_state_t *state;
  state = _db_code_state (idbug);

  if (_db_keyword_ (idbug, (char *) keyword))
    {
      if (idbug->flags & IDBUG_THREADED)
        idbug->lock_mutex (idbug->mutex);

      pos = DoPrefix (idbug, _line_);
      if (TRACING)
        {
          Indent (idbug, state->level + 1);
          pos += min (
              min (state->level - idbug->stack->sub_level, 0) * INDENT, 80);
        }
      else
        {
          pos += fprintf (idbug->_db_fp_, "%s: ", state->func);
        }

      sprintf (dbuff, "%s: Memory: %p  Bytes: ", keyword, memory);
      pos += strlen (dbuff);

      (void) fputs (dbuff, idbug->_db_fp_);

      while (length-- > 0)
        {
          strpos = int2str ((long) *((unsigned char *) memory++), dbuff, 10);
          *strpos++ = ' ';
          *strpos = 0;
          if ((pos += (int) (strpos - dbuff)) >= 80)
            {
              pos = (int) (strpos - dbuff);
              fputc ('\n', idbug->_db_fp_);
            }

          (void) fputs (dbuff, idbug->_db_fp_);
        }
      (void) fputc ('\n', idbug->_db_fp_);
      dbug_flush (idbug);       /* unlock_mutex */
    }
}


/*
 *  FUNCTION
 *
 * ListParse    parse list of modifiers in debug control string
 *
 *  SYNOPSIS
 *
 * static struct link *ListParse (ctlp)
 * char *ctlp;
 *
 *  DESCRIPTION
 *
 * Given pointer to a comma separated list of strings in "cltp",
 * parses the list, building a list and returning a pointer to it.
 * The original comma separated list is modified in the process of
 * building the linked list, thus it had better be a duplicate
 * if it is important.
 *
 * Note that since each link is added at the head of the list,
 * the final list will be in "reverse order", which is not
 * significant for our usage here.
 *
 */

static struct link *
ListParse (idbug_t * idbug, char *ctlp)
{
  char *start;
  struct link *new;
  struct link *head;

  head = NULL;
  while (*ctlp != EOS)
    {
      start = ctlp;
      while (*ctlp != EOS && *ctlp != ',')
        {
          ctlp++;
        }
      if (*ctlp == ',')
        {
          *ctlp++ = EOS;
        }
      new = (struct link *) IDBUG_MALLOC (idbug, sizeof (struct link));
      new->str = StrDup (idbug, start);
      new->next_link = head;
      head = new;
    }
  return head;
}

/*
 *  FUNCTION
 *
 * InList    test a given string for member of a given list
 *
 *  SYNOPSIS
 *
 * static int InList (linkp, cp)
 * struct link *linkp;
 * char *cp;
 *
 *  DESCRIPTION
 *
 * Tests the string pointed to by "cp" to determine if it is in
 * the list pointed to by "linkp".  Linkp points to the first
 * link in the list.  If linkp is NULL then the string is treated
 * as if it is in the list (I.E all strings are in the null list).
 * This may seem rather strange at first but leads to the desired
 * operation if no list is given.  The net effect is that all
 * strings will be accepted when there is no list, and when there
 * is a list, only those strings in the list will be accepted.
 *
 */

static int
InList (struct link *linkp, char *cp)
{
  register struct link *scan;
  register int accept;

  if (linkp == NULL)
    {
      accept = IDBUG_TRUE;
    }
  else
    {
      accept = IDBUG_FALSE;
      for (scan = linkp; scan != NULL; scan = scan->next_link)
        {
          if (STREQ (scan->str, cp))
            {
              accept = IDBUG_TRUE;
              break;
            }
        }
    }
  return (accept);
}


/*
 *  FUNCTION
 *
 * PushState    push current state onto stack and set up new one
 *
 *  SYNOPSIS
 *
 * static VOID PushState ()
 *
 *  DESCRIPTION
 *
 * Pushes the current state on the state stack, and initializes
 * a new state.  The only parameter inherited from the previous
 * state is the function nesting level.  This action can be
 * inhibited if desired, via the "r" flag.
 *
 * The state stack is a linked list of states, with the new
 * state added at the head.  This allows the stack to grow
 * to the limits of memory if necessary.
 *
 */

static void
PushState (idbug_t * idbug)
{
  struct idbug_state *new_state;

  new_state = (struct idbug_state *) IDBUG_MALLOC (idbug, sizeof (*new_state));
  new_state->flags = 0;
  new_state->delay = 0;
  new_state->maxdepth = MAXDEPTH;
  new_state->sub_level = 0;
  new_state->out_file = stderr;
  new_state->prof_file = (FILE *) 0;
  new_state->functions = NULL;
  new_state->p_functions = NULL;
  new_state->keywords = NULL;
  new_state->processes = NULL;
  new_state->next_state = idbug->stack;
  idbug->stack = new_state;
}


/*
 *  FUNCTION
 *
 * DoTrace    check to see if tracing is current enabled
 *
 *  SYNOPSIS
 *
 * static int DoTrace (stack)
 *
 *  DESCRIPTION
 *
 * Checks to see if tracing is enabled based on whether the
 * user has specified tracing, the maximum trace depth has
 * not yet been reached, the current function is selected,
 * and the current process is selected.  Returns TRUE if
 * tracing is enabled, FALSE otherwise.
 *
 */

static int
DoTrace (idbug_t * idbug, idbug_code_state_t * state)
{
  int trace = IDBUG_FALSE;

  if (TRACING &&
      state->level <= idbug->stack->maxdepth &&
      InList (idbug->stack->functions, state->func) &&
      InList (idbug->stack->processes, idbug->_db_process_))
    trace = IDBUG_TRUE;
  return trace;
}


/*
 *  FUNCTION
 *
 * DoProfile    check to see if profiling is current enabled
 *
 *  SYNOPSIS
 *
 * static int DoProfile ()
 *
 *  DESCRIPTION
 *
 * Checks to see if profiling is enabled based on whether the
 * user has specified profiling, the maximum trace depth has
 * not yet been reached, the current function is selected,
 * and the current process is selected.  Returns TRUE if
 * profiling is enabled, FALSE otherwise.
 *
 */

static int
DoProfile (idbug_t * idbug)
{
  register int profile;
  idbug_code_state_t *state;

  assert (!(idbug->flags & IDBUG_THREADED));

  state = _db_code_state (idbug);

  profile = IDBUG_FALSE;
  if (PROFILING &&
      state->level <= idbug->stack->maxdepth &&
      InList (idbug->stack->p_functions, state->func) &&
      InList (idbug->stack->processes, idbug->_db_process_))
    profile = IDBUG_TRUE;
  return (profile);
}


/*
 *  FUNCTION
 *
 * _db_keyword_    test keyword for member of keyword list
 *
 *  SYNOPSIS
 *
 * int _db_keyword_ (keyword)
 * char *keyword;
 *
 *  DESCRIPTION
 *
 * Test a keyword to determine if it is in the currently active
 * keyword list.  As with the function list, a keyword is accepted
 * if the list is null, otherwise it must match one of the list
 * members.  When debugging is not on, no keywords are accepted.
 * After the maximum trace level is exceeded, no keywords are
 * accepted (this behavior subject to change).  Additionally,
 * the current function and process must be accepted based on
 * their respective lists.
 *
 * Returns TRUE if keyword accepted, FALSE otherwise.
 *
 */

EXPORT int
_db_keyword_ (idbug_t * idbug, char *keyword)
{
  int accept;
  idbug_code_state_t *state;

  if (!idbug->init_done)
    _db_push_ (idbug, "");
  state = _db_code_state (idbug);
  accept = IDBUG_FALSE;
  if (DEBUGGING &&
      state->level <= idbug->stack->maxdepth &&
      InList (idbug->stack->functions, state->func) &&
      InList (idbug->stack->keywords, keyword) &&
      InList (idbug->stack->processes, idbug->_db_process_))
    accept = IDBUG_TRUE;
  return (accept);
}

/*
 *  FUNCTION
 *
 * Indent    indent a line to the given indentation level
 *
 *  SYNOPSIS
 *
 * static VOID Indent (indent)
 * int indent;
 *
 *  DESCRIPTION
 *
 * Indent a line to the given level.  Note that this is
 * a simple minded but portable implementation.
 * There are better ways.
 *
 * Also, the indent must be scaled by the compile time option
 * of character positions per nesting level.
 *
 */

static void
Indent (idbug_t * idbug, int indent)
{
  register int count;

  indent = max (indent - 1 - idbug->stack->sub_level, 0) * INDENT;
  for (count = 0; count < indent; count++)
    {
      if ((count % INDENT) == 0)
        fputc ('|', idbug->_db_fp_);
      else
        fputc (' ', idbug->_db_fp_);
    }
}


/*
 *  FUNCTION
 *
 * FreeList    free all memory associated with a linked list
 *
 *  SYNOPSIS
 *
 * static VOID FreeList (linkp)
 * struct link *linkp;
 *
 *  DESCRIPTION
 *
 * Given pointer to the head of a linked list, frees all
 * memory held by the list and the members of the list.
 *
 */

static void
FreeList (idbug_t * idbug, struct link *linkp)
{
  register struct link *old;

  while (linkp != NULL)
    {
      old = linkp;
      linkp = linkp->next_link;
      /*
         * the string is no more allocated on the heap
         if (old->str != NULL)
         {
         IDBUG_FREE (idbug, old->str);
         }
       */
      IDBUG_FREE (idbug, (char *) old);
    }
}


/*
 *  FUNCTION
 *
 * StrDup   make a duplicate of a string in new memory
 *
 *  SYNOPSIS
 *
 * static char *StrDup (string)
 * char *string;
 *
 *  DESCRIPTION
 *
 * Given pointer to a string, allocates sufficient memory to make
 * a duplicate copy, and copies the string to the newly allocated
 * memory.  Failure to allocated sufficient memory is immediately
 * fatal.
 *
 */


static char *
StrDup (idbug_t * idbug, char *str)
{
  return str;
}


/*
 *  FUNCTION
 *
 * DoPrefix    print debugger line prefix prior to indentation
 *
 *  SYNOPSIS
 *
 * static VOID DoPrefix (_line_)
 * int _line_;
 *
 *  DESCRIPTION
 *
 * Print prefix common to all debugger output lines, prior to
 * doing indentation if necessary.  Print such information as
 * current process name, current source file name and line number,
 * and current function nesting depth.
 *
 */

static int
DoPrefix (idbug_t * idbug, unsigned int _line_)
{
  idbug_code_state_t *state;
  int written = 0;
  state = _db_code_state (idbug);

  state->lineno++;
  if (idbug->stack->flags & _IDBUG_PID_ON)
    {
      if (idbug->flags & IDBUG_THREADED)
        written += fprintf (idbug->_db_fp_, "%5lx: ",
                            idbug->get_thread_id (idbug->opaque));
      else
        written += fprintf (idbug->_db_fp_, "%5ld: ",
                            idbug->get_process_id (idbug->opaque));
    }
  if (idbug->stack->flags & _IDBUG_NUMBER_ON)
    {
      written += fprintf (idbug->_db_fp_, "%5d: ", state->lineno);
    }
  if (idbug->stack->flags & _IDBUG_PROCESS_ON)
    {
      written += fprintf (idbug->_db_fp_, "%s: ", idbug->_db_process_);
    }
  if (idbug->stack->flags & _IDBUG_FILE_ON)
    {
      written += fprintf (idbug->_db_fp_, "%14s: ", BaseName (state->file));
    }
  if (idbug->stack->flags & _IDBUG_LINE_ON)
    {
      written += fprintf (idbug->_db_fp_, "%5d: ", _line_);
    }
  if (idbug->stack->flags & _IDBUG_DEPTH_ON)
    {
      written += fprintf (idbug->_db_fp_, "%4d: ", state->level);
    }

  return written;
}


/*
 *  FUNCTION
 *
 * DBUGOpenFile    open new output stream for debugger output
 *
 *  SYNOPSIS
 *
 * static VOID DBUGOpenFile (name)
 * char *name;
 *
 *  DESCRIPTION
 *
 * Given name of a new file (or "-" for stdout) opens the file
 * and sets the output stream to the new file.
 * stdout stream
 *
 */

static void
DBUGOpenFile (idbug_t * idbug, char *name)
{
  register FILE *fp;
  register int newfile;

  DEBUGGING_IDBUG (
                    fprintf (stderr, "dbug: writing log to file %s\n", name);
    );

  if (name != NULL && strcmp (name, "") != 0)
    {
      strcpy (idbug->stack->name, name);
      if (strcmp (name, "-") == 0)
        {
          idbug->_db_fp_ = stdout;
          idbug->stack->out_file = idbug->_db_fp_;
          idbug->stack->flags |= _IDBUG_FLUSH_ON_WRITE;
        }
      else
        {
          if (!Writable (idbug, name))
            {
              (void) fprintf (stderr, ERR_OPEN, idbug->_db_process_, name);
              perror ("");
              fflush (stderr);
            }
          else
            {
          newfile = !Exists (name);
          if (NULL == (fp = fopen (name, "w")))
            {
                  (void) fprintf (stderr, ERR_OPEN, idbug->_db_process_, name);
              perror ("");
              fflush (stderr);
            }
          else
            {
              idbug->_db_fp_ = fp;
              idbug->stack->out_file = fp;
              if (newfile)
                {
                  ChangeOwner (idbug, name);
                }
            }
        }
    }
    }
}

/*
 *  FUNCTION
 *
 * OpenProfile    open new output stream for profiler output
 *
 *  SYNOPSIS
 *
 * static FILE *OpenProfile (name)
 * char *name;
 *
 *  DESCRIPTION
 *
 * Given name of a new file, opens the file
 * and sets the profiler output stream to the new file.
 *
 * It is currently unclear whether the prefered behavior is
 * to truncate any existing file, or simply append to it.
 * The latter behavior would be desirable for collecting
 * accumulated runtime history over a number of separate
 * runs.  It might take some changes to the analyzer program
 * though, and the notes that Binayak sent with the profiling
 * diffs indicated that append was the normal mode, but this
 * does not appear to agree with the actual code. I haven't
 * investigated at this time [fnf; 24-Jul-87].
 */

static FILE *
OpenProfile (idbug_t * idbug, const char *name)
{
  register FILE *fp;
  register int newfile;

  assert (!(idbug->flags & IDBUG_THREADED));

  fp = NULL;

  if (idbug->_db_fp_ != NULL)
    fflush (idbug->_db_fp_);

  if (!Writable (idbug, name))
    {
      (void) fprintf (idbug->_db_fp_, ERR_OPEN, idbug->_db_process_, name);
      perror ("");
      dbug_flush (idbug);
      (void) dbug_delay (idbug->stack->delay);
    }
  else
    {
  newfile = !Exists (name);
  if (NULL == (fp = fopen (name, "w")))
    {
      (void) fprintf (idbug->_db_fp_, ERR_OPEN, idbug->_db_process_, name);
      perror ("");
      dbug_flush (idbug);
    }
  else
    {
      idbug->_db_pfp_ = fp;
      idbug->stack->prof_file = fp;
      if (newfile)
        {
          ChangeOwner (idbug, name);
        }
    }
    }
  return fp;
}

/*
 *  FUNCTION
 *
 * CloseFile    close the debug output stream
 *
 *  SYNOPSIS
 *
 * static VOID CloseFile (fp)
 * FILE *fp;
 *
 *  DESCRIPTION
 *
 * Closes the debug output stream unless it is standard output
 * or standard error.
 *
 */

static void
CloseFile (idbug_t * idbug, FILE * fp)
{
  if (fp != stderr && fp != stdout)
    {
      if (fclose (fp) == EOF)
        {
          if (idbug->flags & IDBUG_THREADED)
            idbug->lock_mutex (idbug->mutex);

          (void) fprintf (idbug->_db_fp_, ERR_CLOSE, idbug->_db_process_);
          perror ("");
          dbug_flush (idbug);
        }
    }
}


/*
 *  FUNCTION
 *
 * DbugMalloc    allocate memory for debugger runtime support
 *
 *  SYNOPSIS
 *
 * static long *DbugMalloc (size)
 * int size;
 *
 *  DESCRIPTION
 *
 * Allocate more memory for debugger runtime support functions.
 * Failure to to allocate the requested number of bytes is
 * immediately fatal to the current process.  This may be
 * rather unfriendly behavior.  It might be better to simply
 * print a warning message, freeze the current debugger state,
 * and continue execution.
 *
 */

/*
 *  FUNCTION
 *
 * BaseName    strip leading pathname components from name
 *
 *  SYNOPSIS
 *
 * static char *BaseName (pathname)
 * char *pathname;
 *
 *  DESCRIPTION
 *
 * Given pointer to a complete pathname, locates the base file
 * name at the end of the pathname and returns a pointer to
 * it.
 *
 */

static char *
BaseName (char *pathname)
{
  register char *base;

  base = strrchr (pathname, FN_LIBCHAR);
  if (base++ == '\0')
    base = pathname;
  return (base);
}

static int
Exists (const char *filename)
{
  FILE *fp = fopen (filename, "r");

  if (fp == NULL)
    return IDBUG_FALSE;

  fclose (fp);

  return IDBUG_TRUE;
}

/*
 * Is the directory writable by the real-user ?
 */
static int
Writable (idbug_t * idbug, const char *pathname)
{
  extern idbug_writable (idbug_t *, const char *);

  return idbug_writable (idbug, pathname);
}

/*
 *  FUNCTION
 *
 * ChangeOwner    change owner to real user for suid programs
 *
 *  SYNOPSIS
 *
 * static VOID ChangeOwner (pathname)
 *
 *  DESCRIPTION
 *
 * For unix systems, change the owner of the newly created debug
 * file to the real owner.  This is strictly for the benefit of
 * programs that are running with the set-user-id bit set.
 *
 * Note that at this point, the fact that pathname represents
 * a newly created file has already been established.  If the
 * program that the debugger is linked to is not running with
 * the suid bit set, then this operation is redundant (but
 * harmless).
 *
 */

static void
ChangeOwner (idbug_t * idbug, const char *pathname)
{
  if (!(idbug->flags & IDBUG_SUID))
    return;

  if (idbug->security.pchown (pathname, idbug->security.pgetuid (),
                              idbug->security.pgetgid ()) == -1)
    {
      (void) fprintf (stderr, ERR_CHOWN, idbug->_db_process_, pathname);
      perror ("");
      (void) fflush (stderr);
    }
}


/*
 *  FUNCTION
 *
 * _db_setjmp_    save debugger environment
 *
 *  SYNOPSIS
 *
 * VOID _db_setjmp_ ()
 *
 *  DESCRIPTION
 *
 * Invoked as part of the user's DBUG_SETJMP macro to save
 * the debugger environment in parallel with saving the user's
 * environment.
 *
 */


EXPORT void
_db_setjmp_ (idbug_t * idbug)
{
  idbug_code_state_t *state;
  state = _db_code_state (idbug);

  state->jmplevel = state->level;
  state->jmpfunc = state->func;
  state->jmpfile = state->file;
}

/*
 *  FUNCTION
 *
 * _db_clone_idbug_for_thread_
 *
 *  SYNOPSIS
 *
 * void _db_clone_idbug_for_thread_ (dest, source)
 * idbug_t * dest
 * idbug_t * source
 *
 *  DESCRIPTION
 *
 * Clone the idbug_t pointed by source into the memory pointed by dest
 * The idbug_code_state_t state is memset to (0)
 * The main use is with multithreading applications, in order to provide
 * a thread specific interface.
 *
 * Notice that after this call dest will differ from source only about
 * its idbug_code_state_t state, as each thread has its own stack, local
 * variables and so on: this seems to be consistent for me.
 *
 */

EXPORT void
_db_init_as_ (idbug_t * dest, idbug_t * source)
{
  assert (source != NULL);
  assert (dest != NULL);

  dest->_db_fp_ = source->_db_fp_;
  dest->_db_process_ = source->_db_process_;

  dest->_db_on_ = source->_db_on_;
  dest->_no_db_ = source->_no_db_;

  /* profile stream */
  dest->_db_pfp_ = source->_db_pfp_;
  dest->_db_pon_ = source->_db_pon_;

  dest->stack = &dest->_stack;
  dest->state = &dest->_state;

  memcpy (dest->_db_jmp_buf, source->_db_jmp_buf,
          sizeof (dest->_db_jmp_buf));

  dest->get_process_id = source->get_process_id;

  dest->mutex = source->mutex;
  dest->lock_mutex = source->lock_mutex;
  dest->unlock_mutex = source->unlock_mutex;
  dest->get_thread_id = source->get_thread_id;
  dest->sanity = source->sanity;
  dest->opaque = source->opaque;

  dest->flags = source->flags;

  memset (dest->state, 0, sizeof (*dest->state));

  /* TODO: security fields */
  memcpy (&dest->security, &source->security, sizeof (idbug_security_t));


  dest->stack->flags = source->stack->flags;
  dest->stack->maxdepth = source->stack->maxdepth;
  dest->stack->delay = source->stack->delay;
  dest->stack->sub_level = source->stack->sub_level;
  dest->stack->out_file = source->stack->out_file;
  dest->stack->prof_file = source->stack->prof_file;
  strcpy (dest->stack->name, source->stack->name);
  dest->stack->functions = source->stack->functions;
  dest->stack->p_functions = source->stack->p_functions;
  dest->stack->keywords = source->stack->keywords;
  dest->stack->processes = source->stack->processes;
  dest->stack->next_state = source->stack->next_state;
}

/*
 *  FUNCTION
 *
 * _db_longjmp_    restore previously saved debugger environment
 *
 *  SYNOPSIS
 *
 * VOID _db_longjmp_ ()
 *
 *  DESCRIPTION
 *
 * Invoked as part of the user's DBUG_LONGJMP macro to restore
 * the debugger environment in parallel with restoring the user's
 * previously saved environment.
 *
 */

EXPORT void
_db_longjmp_ (idbug_t * idbug)
{
  idbug_code_state_t *state;
  state = _db_code_state (idbug);

  state->level = state->jmplevel;
  if (state->jmpfunc)
    {
      state->func = state->jmpfunc;
    }
  if (state->jmpfile)
    {
      state->file = state->jmpfile;
    }
}

/*
 *  FUNCTION
 *
 * DelayArg   convert D flag argument to appropriate value
 *
 *  SYNOPSIS
 *
 * static int DelayArg (value)
 * int value;
 *
 *  DESCRIPTION
 *
 * Converts delay argument, given in tenths of a second, to the
 * appropriate numerical argument used by the system to delay
 * that that many tenths of a second.  For example, on the
 * amiga, there is a system call "Delay()" which takes an
 * argument in ticks (50 per second).  On unix, the sleep
 * command takes seconds.  Thus a value of "10", for one
 * second of delay, gets converted to 50 on the amiga, and 1
 * on unix.  Other systems will need to use a timing loop.
 *
 */

#ifdef AMIGA
#define HZ (50)                 /* Probably in some header somewhere */
#endif

static int
DelayArg (int value)
{
  unsigned int delayarg = 0;

  delayarg = value;             /* Delay is in seconds for sleep () */

  DEBUGGING_IDBUG (
              fprintf (stderr, "dbug: DelayArg: delayarg = %u\n", delayarg);
    )

    return delayarg;
}


/*
 * A dummy delay stub for systems that do not support delays.
 * With a little work, this can be turned into a timing loop.
 */

static int
dbug_delay (int ticks)
{
  /*
   * wait ticks seconds:
   * - do one cycle and get elapsed seconds, divide ticks by elapsed
   *   then, do the cycle for that number of times
   */

  double time1;

 	time1 = (double) clock();

  DEBUGGING_IDBUG (
                    fprintf (stderr, "dbug: dbug_delay %d ticks\n", ticks);
    )

    time1 = time1 / (double) CLOCKS_PER_SEC;

	while (((double) clock() / (double) CLOCKS_PER_SEC - time1) * 10 < ticks)
    ;

  return ticks;
}



 /*
  * TODO:
  * flush dbug-stream, free mutex lock & wait delay
  * This is because some systems (MSDOS!!) dosn't flush fileheader
  * and dbug-file isn't readable after a system crash !!
  */

static void
dbug_flush (idbug_t * idbug)
{
  assert (idbug != NULL);

  if (idbug->flags & IDBUG_THREADED)
    assert (idbug->mutex != NULL);

  if (idbug->flags & IDBUG_THREADED
      || idbug->stack->flags & _IDBUG_FLUSH_ON_WRITE)
    {

#if defined(DBUG_OS_MSDOS) || defined(DBUG_OS_W32)
      if (idbug->_db_fp_ != stdout && idbug->_db_fp_ != stderr)
        {
          if (!(freopen (idbug->stack->name, "a", idbug->_db_fp_)))
            {
              (void) fprintf (stderr, ERR_OPEN, idbug->_db_process_,
                              idbug->stack->name);
              fflush (stderr);
              idbug->_db_fp_ = stdout;
              idbug->stack->out_file = idbug->_db_fp_;
              idbug->stack->flags |= _IDBUG_FLUSH_ON_WRITE;
            }
        }
      else
#endif
        {
          (void) fflush (idbug->_db_fp_);
          (void) dbug_delay (idbug->stack->delay);
        }
    }

  if (idbug->flags & IDBUG_THREADED)
    {
      assert (idbug->mutex != NULL);
      idbug->unlock_mutex (idbug->mutex);
    }
}                               /* dbug_flush */


/*
 * Here we need the definitions of the clock routine.  Add your
 * own for whatever system that you have.
 */

/*
 * Returns the user time in milliseconds used by this process so far.
 */

static unsigned long
Clock (void)
{
  return (unsigned long) ((double) clock () * (1000.0 / (double) CLOCKS_PER_SEC));
}

/* The following copyright notice applies to code generated by
   ** "mktclapp".  The "mktclapp" program itself is covered by the
   ** GNU Public License.
   **
   ** Copyright (c) 1998 D. Richard Hipp
   **
   ** The author hereby grants permission to use, copy, modify, distribute,
   ** and license this software and its documentation for any purpose, provided
   ** that existing copyright notices are retained in all copies and that this
   ** notice is included verbatim in any distributions. No written agreement,
   ** license, or royalty fee is required for any of the authorized uses.
   ** Modifications to this software may be copyrighted by their authors
   ** and need not follow the licensing terms described here, provided that
   ** the new terms are clearly indicated on the first page of each file where
   ** they apply.
   **
   ** In no event shall the author or the distributors be liable to any party
   ** for direct, indirect, special, incidental, or consequential damages
   ** arising out of the use of this software, its documentation, or any
   ** derivatives thereof, even if the author has been advised of the
   ** possibility of such damage.  The author and distributors specifically
   ** disclaim any warranties, including but not limited to the implied
   ** warranties of merchantability, fitness for a particular purpose, and
   ** non-infringment.  This software is provided at no fee on an
   ** "as is" basis.  The author and/or distritutors have no obligation
   ** to provide maintenance, support, updates, enhancements and/or
   ** modifications.
   **
   ** GOVERNMENT USE: If you are acquiring this software on behalf of the
   ** U.S. government, the Government shall have only "Restricted Rights"
   ** in the software and related documentation as defined in the Federal
   ** Acquisition Regulations (FARs) in Clause 52.227.19 (c) (2).  If you
   ** are acquiring the software on behalf of the Department of Defense, the
   ** software shall be classified as "Commercial Computer Software" and the
   ** Government shall have only "Restricted Rights" as defined in Clause
   ** 252.227-7013 (c) (1) of DFARs.  Notwithstanding the foregoing, the
   ** author grants the U.S. Government and others acting in its behalf
   ** permission to use and distribute the software in accordance with the
   ** terms specified in this license.
 */
/*
   ** The following modules is an enhanced replacement for the "printf" programs
   ** found in the standard library.  The following enhancements are
   ** supported:
   **
   **      +  Additional functions.  The standard set of "printf" functions
   **         includes printf, fprintf, sprintf, vprintf, vfprintf, and
   **         vsprintf.  This module adds the following:
   **
   **           *  mprintf --  Similar to sprintf.  Writes output to memory
   **                          obtained from malloc.
   **
   **           *  A v- version (ex: vmprintf) of every function is also
   **              supplied.
   **
   **      +  A few extensions to the formatting notation are supported:
   **
   **           *  The "=" flag (similar to "-") causes the output to be
   **              be centered in the appropriately sized field.
   **
   **           *  The %b field outputs an integer in binary notation.
   **
   **           *  The %c field now accepts a precision.  The character output
   **              is repeated by the number of times the precision specifies.
   **
   **           *  The %' field works like %c, but takes as its character the
   **              next character of the format string, instead of the next
   **              argument.  For example,  printf("%.78'-")  prints 78 minus
   **              signs, the same as  printf("%.78c",'-').
   **
   **      +  When compiled using GCC on a SPARC, this version of printf is
   **         faster than the library printf for SUN OS 4.1.
   **
   **      +  All functions are fully reentrant.
   **
 */
/*
   ** Undefine COMPATIBILITY to make some slight changes in the way things
   ** work.  I think the changes are an improvement, but they are not
   ** backwards compatible.
 */
/* #define COMPATIBILITY       / * Compatible with SUN OS 4.1 */
/*
   ** Characters that need to be escaped inside a TCL string.
 */
static const char NeedEsc[] =
{
  1, 1, 1, 1, 1, 1, 1, 1, 'b', 't', 'n', 1, 'f', 'r', 1, 1,
  1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1,
  0, 0, '"', 0, '$', 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
  0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
  0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
  0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, '[', '\\', ']', 0, 0,
  0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
  0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 0, 1, 0, 1,
  1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1,
  1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1,
  1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1,
  1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1,
  1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1,
  1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1,
  1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1,
  1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1,
};

/*
   ** Conversion types fall into various categories as defined by the
   ** following enumeration.
 */

enum e_type
  {                             /* The type of the format field */
    RADIX,                      /* Integer types.  %d, %x, %o, and so forth */
    FLOAT,                      /* Floating point.  %f */
    EXP,                        /* Exponentional notation. %e and %E */
    GENERIC,                    /* Floating or exponential, depending on exponent. %g */
    SIZE,                       /* Return number of characters processed so far. %n */
    STRING,                     /* Strings. %s */
    PERCENT,                    /* Percent symbol. %% */
    CHARX,                      /* Characters. %c */
    ERROR,                      /* Used to indicate no such conversion type */
/* The rest are extensions, not normally found in printf() */
    CHARLIT,                    /* Literal characters.  %' */
    TCLESCAPE,                  /* Strings with special characters escaped.  %q */
    MEM_STRING,                 /* A string which should be deleted after use. %z */
    ORDINAL                     /* 1st, 2nd, 3rd and so forth */
  };

/*
   ** Each builtin conversion character (ex: the 'd' in "%d") is described
   ** by an instance of the following structure
 */
typedef struct s_info
  {                             /* Information about each format field */
    int fmttype;                /* The format field code letter */
    int base;                   /* The base for radix conversion */
    char *charset;              /* The character set for conversion */
    int flag_signed;            /* Is the quantity signed? */
    char *prefix;               /* Prefix on non-zero values in alt format */
    enum e_type type;           /* Conversion paradigm */
  }
info;

/*
   ** The following table is searched linearly, so it is good to put the
   ** most frequently used conversion types first.
 */
static const info fmtinfo[] =
{
  {'d', 10, "0123456789", 1, 0, RADIX,},
  {'s', 0, 0, 0, 0, STRING,},
  {'q', 0, 0, 0, 0, TCLESCAPE,},
  {'z', 0, 0, 0, 0, MEM_STRING,},
  {'c', 0, 0, 0, 0, CHARX,},
  {'o', 8, "01234567", 0, "0", RADIX,},
  {'u', 10, "0123456789", 0, 0, RADIX,},
  {'x', 16, "0123456789abcdef", 0, "x0", RADIX,},
  {'X', 16, "0123456789ABCDEF", 0, "X0", RADIX,},
  {'r', 10, "0123456789", 0, 0, ORDINAL,},
  {'f', 0, 0, 1, 0, FLOAT,},
  {'e', 0, "e", 1, 0, EXP,},
  {'E', 0, "E", 1, 0, EXP,},
  {'g', 0, "e", 1, 0, GENERIC,},
  {'G', 0, "E", 1, 0, GENERIC,},
  {'i', 10, "0123456789", 1, 0, RADIX,},
  {'n', 0, 0, 0, 0, SIZE,},
  {'%', 0, 0, 0, 0, PERCENT,},
  {'b', 2, "01", 0, "b0", RADIX,},  /* Binary notation */
  {'p', 10, "0123456789", 0, 0, RADIX,},  /* Pointers */
  {'\'', 0, 0, 0, 0, CHARLIT,}, /* Literal char */
};

#define NINFO  (sizeof(fmtinfo)/sizeof(info))  /* Size of the fmtinfo table */

/*
   ** If NOFLOATINGPOINT is defined, then none of the floating point
   ** conversions will work.
 */
#ifndef NOFLOATINGPOINT
/*
   ** "*val" is a double such that 0.1 <= *val < 10.0
   ** Return the ascii code for the leading digit of *val, then
   ** multiply "*val" by 10.0 to renormalize.
   **
   ** Example:
   **     input:     *val = 3.14159
   **     output:    *val = 1.4159    function return = '3'
   **
   ** The counter *cnt is incremented each time.  After counter exceeds
   ** 16 (the number of significant digits in a 64-bit float) '0' is
   ** always returned.
 */
static int
getdigit (double *val, int *cnt)
{
  int digit;
  double d;
  if ((*cnt)++ >= 16)
    return '0';
  digit = (int) *val;
  d = digit;
  digit += '0';
  *val = (*val - d) * 10.0;
  return digit;
}
#endif

#define BUFSIZE 1000            /* Size of the output buffer */

/*
   ** The root program.  All variations call this core.
   **
   ** INPUTS:
   **   func   This is a pointer to a function taking three arguments
   **            1. A pointer to the list of characters to be output
   **               (Note, this list is NOT null terminated.)
   **            2. An integer number of characters to be output.
   **               (Note: This number might be zero.)
   **            3. A pointer to anything.  Same as the "arg" parameter.
   **
   **   arg    This is the pointer to anything which will be passed as the
   **          third argument to "func".  Use it for whatever you like.
   **
   **   fmt    This is the format string, as in the usual print.
   **
   **   ap     This is a pointer to a list of arguments.  Same as in
   **          vfprint.
   **
   ** OUTPUTS:
   **          The return value is the total number of characters sent to
   **          the function "func".  Returns -1 on a error.
   **
   ** Note that the order in which automatic variables are declared below
   ** seems to make a big difference in determining how fast this beast
   ** will run.
 */
static int
vxprintf (void (*func) (const char *, int, void *),
          void *arg,
          const char *format,
          va_list ap)
{
  register const char *fmt;     /* The format string. */
  register int c;               /* Next character in the format string */
  register char *bufpt;         /* Pointer to the conversion buffer */
  register int precision;       /* Precision of the current field */
  register int length;          /* Length of the field */
  register int idx;             /* A general purpose loop counter */
  int count;                    /* Total number of characters output */
  int width;                    /* Width of the current field */
  int flag_leftjustify;         /* True if "-" flag is present */
  int flag_plussign;            /* True if "+" flag is present */
  int flag_blanksign;           /* True if " " flag is present */
  int flag_alternateform;       /* True if "#" flag is present */
  int flag_zeropad;             /* True if field width constant starts with zero */
  int flag_long;                /* True if "l" flag is present */
  int flag_center;              /* True if "=" flag is present */
  unsigned long longvalue;      /* Value for integer types */
  double realvalue;             /* Value for real types */
  const info *infop;            /* Pointer to the appropriate info structure */
  char buf[BUFSIZE + 1];        /* Conversion buffer */
  char prefix;                  /* Prefix character.  "+" or "-" or " " or '\0'. */
  int errorflag = 0;            /* True if an error is encountered */
  enum e_type xtype;            /* Conversion paradigm */
  char *zMem;                   /* String to be freed */
  char *zExtra;                 /* Extra memory used for TCLESCAPE conversions */
  const static char spaces[] =
	"                                                  "
  "                                                                      ";
#define SPACESIZE (sizeof(spaces)-1)
#ifndef NOFLOATINGPOINT
  int exp;                      /* exponent of real numbers */
  double rounder;               /* Used for rounding floating point values */
  int flag_dp;                  /* True if decimal point should be shown */
  int flag_rtz;                 /* True if trailing zeros should be removed */
  int flag_exp;                 /* True to force display of the exponent */
  int nsd;                      /* Number of significant digits returned */
#endif

  fmt = format;                 /* Put in a register for speed */
  count = length = 0;
  bufpt = 0;
  for (; (c = (*fmt)) != 0; ++fmt)
    {
      if (c != '%')
        {
          register int amt;
          bufpt = (char *) fmt;
          amt = 1;
          while ((c = (*++fmt)) != '%' && c != 0)
            amt++;
          (*func) (bufpt, amt, arg);
          count += amt;
          if (c == 0)
            break;
        }
      if ((c = (*++fmt)) == 0)
        {
          errorflag = 1;
          (*func) ("%", 1, arg);
          count++;
          break;
        }
      /* Find out what flags are present */
      flag_leftjustify = flag_plussign = flag_blanksign =
        flag_alternateform = flag_zeropad = flag_center = 0;
      do
        {
          switch (c)
            {
            case '-':
              flag_leftjustify = 1;
              c = 0;
              break;
            case '+':
              flag_plussign = 1;
              c = 0;
              break;
            case ' ':
              flag_blanksign = 1;
              c = 0;
              break;
            case '#':
              flag_alternateform = 1;
              c = 0;
              break;
            case '0':
              flag_zeropad = 1;
              c = 0;
              break;
            case '=':
              flag_center = 1;
              c = 0;
              break;
            default:
              break;
            }
        }
      while (c == 0 && (c = (*++fmt)) != 0);
      if (flag_center)
        flag_leftjustify = 0;
      /* Get the field width */
      width = 0;
      if (c == '*')
        {
          width = va_arg (ap, int);
          if (width < 0)
            {
              flag_leftjustify = 1;
              width = -width;
            }
          c = *++fmt;
        }
      else
        {
          while (isdigit (c))
            {
              width = width * 10 + c - '0';
              c = *++fmt;
            }
        }
      if (width > BUFSIZE - 10)
        {
          width = BUFSIZE - 10;
        }
      /* Get the precision */
      if (c == '.')
        {
          precision = 0;
          c = *++fmt;
          if (c == '*')
            {
              precision = va_arg (ap, int);
#ifndef COMPATIBILITY
              /* This is sensible, but SUN OS 4.1 doesn't do it. */
              if (precision < 0)
                precision = -precision;
#endif
              c = *++fmt;
            }
          else
            {
              while (isdigit (c))
                {
                  precision = precision * 10 + c - '0';
                  c = *++fmt;
                }
            }
          /* Limit the precision to prevent overflowing buf[] during conversion */
          if (precision > BUFSIZE - 40)
            precision = BUFSIZE - 40;
        }
      else
        {
          precision = -1;
        }
      /* Get the conversion type modifier */
      if (c == 'l')
        {
          flag_long = 1;
          c = *++fmt;
        }
      else
        {
          flag_long = 0;
        }
      /* Fetch the info entry for the field */
      infop = 0;
      for (idx = 0; idx < NINFO; idx++)
        {
          if (c == fmtinfo[idx].fmttype)
            {
              infop = &fmtinfo[idx];
              break;
            }
        }
      /* No info entry found.  It must be an error. */
      if (infop == 0)
        {
          xtype = ERROR;
        }
      else
        {
          xtype = infop->type;
        }
      zExtra = 0;

      /*
         ** At this point, variables are initialized as follows:
         **
         **   flag_alternateform          TRUE if a '#' is present.
         **   flag_plussign               TRUE if a '+' is present.
         **   flag_leftjustify            TRUE if a '-' is present or if the
         **                               field width was negative.
         **   flag_zeropad                TRUE if the width began with 0.
         **   flag_long                   TRUE if the letter 'l' (ell) prefixed
         **                               the conversion character.
         **   flag_blanksign              TRUE if a ' ' is present.
         **   width                       The specified field width.  This is
         **                               always non-negative.  Zero is the default.
         **   precision                   The specified precision.  The default
         **                               is -1.
         **   xtype                       The class of the conversion.
         **   infop                       Pointer to the appropriate info struct.
       */
      switch (xtype)
        {
        case ORDINAL:
        case RADIX:
          if (flag_long)
            longvalue = va_arg (ap, long);
          else
            longvalue = va_arg (ap, int);
#ifdef COMPATIBILITY
          /* For the format %#x, the value zero is printed "0" not "0x0".
             ** I think this is stupid. */
          if (longvalue == 0)
            flag_alternateform = 0;
#else
          /* More sensible: turn off the prefix for octal (to prevent "00"),
             ** but leave the prefix for hex. */
          if (longvalue == 0 && infop->base == 8)
            flag_alternateform = 0;
#endif
          if (infop->flag_signed)
            {
              if (*(long *) &longvalue < 0)
                {
                  longvalue = -*(long *) &longvalue;
                  prefix = '-';
                }
              else if (flag_plussign)
                prefix = '+';
              else if (flag_blanksign)
                prefix = ' ';
              else
                prefix = 0;
            }
          else
            prefix = 0;
          if (flag_zeropad && precision < width - (prefix != 0))
            {
              precision = width - (prefix != 0);
            }
          bufpt = &buf[BUFSIZE];
          if (xtype == ORDINAL)
            {
              long a, b;
              a = longvalue % 10;
              b = longvalue % 100;
              bufpt -= 2;
              if (a == 0 || a > 3 || (b > 10 && b < 14))
                {
                  bufpt[0] = 't';
                  bufpt[1] = 'h';
                }
              else if (a == 1)
                {
                  bufpt[0] = 's';
                  bufpt[1] = 't';
                }
              else if (a == 2)
                {
                  bufpt[0] = 'n';
                  bufpt[1] = 'd';
                }
              else if (a == 3)
                {
                  bufpt[0] = 'r';
                  bufpt[1] = 'd';
                }
            }
          {
            register char *cset;  /* Use registers for speed */
            register int base;
            cset = infop->charset;
            base = infop->base;
            do
              {                 /* Convert to ascii */
                *(--bufpt) = cset[longvalue % base];
                longvalue = longvalue / base;
              }
            while (longvalue > 0);
          }
          length = (long) &buf[BUFSIZE] - (long) bufpt;
          for (idx = precision - length; idx > 0; idx--)
            {
              *(--bufpt) = '0'; /* Zero pad */
            }
          if (prefix)
            *(--bufpt) = prefix;  /* Add sign */
          if (flag_alternateform && infop->prefix)
            {                   /* Add "0" or "0x" */
              char *pre, x;
              pre = infop->prefix;
              if (*bufpt != pre[0])
                {
                  for (pre = infop->prefix; (x = (*pre)) != 0; pre++)
                    *(--bufpt) = x;
                }
            }
          length = (long) &buf[BUFSIZE] - (long) bufpt;
          break;
        case FLOAT:
        case EXP:
        case GENERIC:
          realvalue = va_arg (ap, double);
#ifndef NOFLOATINGPOINT
          if (precision < 0)
            precision = 6;      /* Set default precision */
          if (precision > BUFSIZE - 10)
            precision = BUFSIZE - 10;
          if (realvalue < 0.0)
            {
              realvalue = -realvalue;
              prefix = '-';
            }
          else
            {
              if (flag_plussign)
                prefix = '+';
              else if (flag_blanksign)
                prefix = ' ';
              else
                prefix = 0;
            }
          if (infop->type == GENERIC && precision > 0)
            precision--;
          rounder = 0.0;
#ifdef COMPATIBILITY
          /* Rounding works like BSD when the constant 0.4999 is used.  Wierd! */
          for (idx = precision, rounder = 0.4999; idx > 0; idx--, rounder *= 0.1);
#else
          /* It makes more sense to use 0.5 */
          for (idx = precision, rounder = 0.5; idx > 0; idx--, rounder *= 0.1);
#endif
          if (infop->type == FLOAT)
            realvalue += rounder;
          /* Normalize realvalue to within 10.0 > realvalue >= 1.0 */
          exp = 0;
          if (realvalue > 0.0)
            {
              int k = 0;
              while (realvalue >= 1e8 && k++ < 100)
                {
                  realvalue *= 1e-8;
                  exp += 8;
                }
              while (realvalue >= 10.0 && k++ < 100)
                {
                  realvalue *= 0.1;
                  exp++;
                }
              while (realvalue < 1e-8 && k++ < 100)
                {
                  realvalue *= 1e8;
                  exp -= 8;
                }
              while (realvalue < 1.0 && k++ < 100)
                {
                  realvalue *= 10.0;
                  exp--;
                }
              if (k >= 100)
                {
                  bufpt = "NaN";
                  length = 3;
                  break;
                }
            }
          bufpt = buf;
          /*
             ** If the field type is GENERIC, then convert to either EXP
             ** or FLOAT, as appropriate.
           */
          flag_exp = xtype == EXP;
          if (xtype != FLOAT)
            {
              realvalue += rounder;
              if (realvalue >= 10.0)
                {
                  realvalue *= 0.1;
                  exp++;
                }
            }
          if (xtype == GENERIC)
            {
              flag_rtz = !flag_alternateform;
              if (exp < -4 || exp > precision)
                {
                  xtype = EXP;
                }
              else
                {
                  precision = precision - exp;
                  xtype = FLOAT;
                }
            }
          else
            {
              flag_rtz = 0;
            }
          /*
             ** The "exp+precision" test causes output to be of type EXP if
             ** the precision is too large to fit in buf[].
           */
          nsd = 0;
          if (xtype == FLOAT && exp + precision < BUFSIZE - 30)
            {
              flag_dp = (precision > 0 || flag_alternateform);
              if (prefix)
                *(bufpt++) = prefix;  /* Sign */
              if (exp < 0)
                *(bufpt++) = '0';  /* Digits before "." */
              else
                for (; exp >= 0; exp--)
                  *(bufpt++) = getdigit (&realvalue, &nsd);
              if (flag_dp)
                *(bufpt++) = '.';  /* The decimal point */
              for (exp++; exp < 0 && precision > 0; precision--, exp++)
                {
                  *(bufpt++) = '0';
                }
              while ((precision--) > 0)
                *(bufpt++) = getdigit (&realvalue, &nsd);
              *(bufpt--) = 0;   /* Null terminate */
              if (flag_rtz && flag_dp)
                {               /* Remove trailing zeros and "." */
                  while (bufpt >= buf && *bufpt == '0')
                    *(bufpt--) = 0;
                  if (bufpt >= buf && *bufpt == '.')
                    *(bufpt--) = 0;
                }
              bufpt++;          /* point to next free slot */
            }
          else
            {                   /* EXP or GENERIC */
              flag_dp = (precision > 0 || flag_alternateform);
              if (prefix)
                *(bufpt++) = prefix;  /* Sign */
              *(bufpt++) = getdigit (&realvalue, &nsd);  /* First digit */
              if (flag_dp)
                *(bufpt++) = '.';  /* Decimal point */
              while ((precision--) > 0)
                *(bufpt++) = getdigit (&realvalue, &nsd);
              bufpt--;          /* point to last digit */
              if (flag_rtz && flag_dp)
                {               /* Remove tail zeros */
                  while (bufpt >= buf && *bufpt == '0')
                    *(bufpt--) = 0;
                  if (bufpt >= buf && *bufpt == '.')
                    *(bufpt--) = 0;
                }
              bufpt++;          /* point to next free slot */
              if (exp || flag_exp)
                {
                  *(bufpt++) = infop->charset[0];
                  if (exp < 0)
                    {
                      *(bufpt++) = '-';
                      exp = -exp;
                    }           /* sign of exp */
                  else
                    {
                      *(bufpt++) = '+';
                    }
                  if (exp >= 100)
                    {
                      *(bufpt++) = (exp / 100) + '0';  /* 100's digit */
                      exp %= 100;
                    }
                  *(bufpt++) = exp / 10 + '0';  /* 10's digit */
                  *(bufpt++) = exp % 10 + '0';  /* 1's digit */
                }
            }
          /* The converted number is in buf[] and zero terminated. Output it.
             ** Note that the number is in the usual order, not reversed as with
             ** integer conversions. */
          length = (long) bufpt - (long) buf;
          bufpt = buf;

          /* Special case:  Add leading zeros if the flag_zeropad flag is
             ** set and we are not left justified */
          if (flag_zeropad && !flag_leftjustify && length < width)
            {
              int i;
              int nPad = width - length;
              for (i = width; i >= nPad; i--)
                {
                  bufpt[i] = bufpt[i - nPad];
                }
              i = prefix != 0;
              while (nPad--)
                bufpt[i++] = '0';
              length = width;
            }
#endif
          break;
        case SIZE:
          *(va_arg (ap, int *)) = count;
          length = width = 0;
          break;
        case PERCENT:
          buf[0] = '%';
          bufpt = buf;
          length = 1;
          break;
        case CHARLIT:
        case CHARX:
          c = buf[0] = (xtype == CHARX ? va_arg (ap, int) : *++fmt);
          if (precision >= 0)
            {
              for (idx = 1; idx < precision; idx++)
                buf[idx] = c;
              length = precision;
            }
          else
            {
              length = 1;
            }
          bufpt = buf;
          break;
        case STRING:
        case MEM_STRING:
          zMem = bufpt = va_arg (ap, char *);
          if (bufpt == 0)
            bufpt = "(null)";
          length = strlen (bufpt);
          if (precision >= 0 && precision < length)
            length = precision;
          break;
        case TCLESCAPE:
          {
            int i, j, n, c, k;
            char *arg = va_arg (ap, char *);
            if (arg == 0)
              arg = "(NULL)";
            for (i = n = 0; (c = arg[i]) != 0; i++)
              {
                k = NeedEsc[c & 0xff];
                if (k == 0)
                  {
                    n++;
                  }
                else if (k == 1)
                  {
                    n += 4;
                  }
                else
                  {
                    n += 2;
                  }
              }
            n++;
            if (n > BUFSIZE)
              {
                bufpt = zExtra = malloc (n);
              }
            else
              {
                bufpt = buf;
              }
            for (i = j = 0; (c = arg[i]) != 0; i++)
              {
                k = NeedEsc[c & 0xff];
                if (k == 0)
                  {
                    bufpt[j++] = c;
                  }
                else if (k == 1)
                  {
                    bufpt[j++] = '\\';
                    bufpt[j++] = ((c >> 6) & 3) + '0';
                    bufpt[j++] = ((c >> 3) & 7) + '0';
                    bufpt[j++] = (c & 7) + '0';
                  }
                else
                  {
                    bufpt[j++] = '\\';
                    bufpt[j++] = k;
                  }
              }
            bufpt[j] = 0;
            length = j;
            if (precision >= 0 && precision < length)
              length = precision;
          }
          break;
        case ERROR:
          buf[0] = '%';
          buf[1] = (char) c;
          errorflag = 0;
          idx = 1 + (c != 0);
          (*func) ("%", idx, arg);
          count += idx;
          if (c == 0)
            fmt--;
          break;
        }                       /* End switch over the format type */
      /*
         ** The text of the conversion is pointed to by "bufpt" and is
         ** "length" characters long.  The field width is "width".  Do
         ** the output.
       */
      if (!flag_leftjustify)
        {
          register int nspace;
          nspace = width - length;
          if (nspace > 0)
            {
              if (flag_center)
                {
                  nspace = nspace / 2;
                  width -= nspace;
                  flag_leftjustify = 1;
                }
              count += nspace;
              while (nspace >= SPACESIZE)
                {
                  (*func) (spaces, SPACESIZE, arg);
                  nspace -= SPACESIZE;
                }
              if (nspace > 0)
                (*func) (spaces, nspace, arg);
            }
        }
      if (length > 0)
        {
          (*func) (bufpt, length, arg);
          count += length;
        }
      if (xtype == MEM_STRING && zMem)
        {
          free (zMem);
        }
      if (flag_leftjustify)
        {
          register int nspace;
          nspace = width - length;
          if (nspace > 0)
            {
              count += nspace;
              while (nspace >= SPACESIZE)
                {
                  (*func) (spaces, SPACESIZE, arg);
                  nspace -= SPACESIZE;
                }
              if (nspace > 0)
                (*func) (spaces, nspace, arg);
            }
        }
      if (zExtra)
        {
          free (zExtra);
        }
    }                           /* End for loop over the format string */
  return errorflag ? -1 : count;
}                               /* End of function */
/*
   ** The following section of code handles the mprintf routine, that
   ** writes to memory obtained from IDBUG_MALLOC().
 */

/* This structure is used to store state information about the
   ** write in progress
 */
struct sgMprintf
{
  char *zBase;                  /* A base allocation */
  char *zText;                  /* The string collected so far */
  int nChar;                    /* Length of the string so far */
  int nAlloc;                   /* Amount of space allocated in zText */
};

/* The xprintf callback function. */
static void
mout (const char *zNewText, int nNewChar, void *arg)
{
  struct sgMprintf *pM = (struct sgMprintf *) arg;

  if (pM->nChar + nNewChar + 1 > pM->nAlloc)
    {
      pM->nAlloc = pM->nChar + nNewChar * 2 + 1;
      if (pM->zText == pM->zBase)
        {
          pM->zText = malloc (pM->nAlloc);
          if (pM->zText != NULL && pM->nChar)
            memcpy (pM->zText, pM->zBase, pM->nChar);
        }
      else
        {
          /*
           * TODO: WARN: realloc
           */
          pM->zText = realloc (pM->zText, pM->nAlloc);
        }
    }
  if (pM->zText)
    {
      memcpy (&pM->zText[pM->nChar], zNewText, nNewChar);
      pM->nChar += nNewChar;
      pM->zText[pM->nChar] = 0;
    }
}

/*
   ** _db_mprintf_ works like printf, but allocations memory to hold the
   ** resulting string and returns a pointer to the allocated memory.
   **
 */
EXPORT char *
_db_mprintf_ (const char *zFormat,...)
{
  va_list ap;
  struct sgMprintf sMprintf;
  char *zNew;
  char zBuf[200];

  sMprintf.nChar = 0;
  sMprintf.nAlloc = sizeof (zBuf);
  sMprintf.zText = zBuf;
  sMprintf.zBase = zBuf;
  va_start (ap, zFormat);
  vxprintf (mout, &sMprintf, zFormat, ap);
  va_end (ap);
  sMprintf.zText[sMprintf.nChar] = 0;
  if (sMprintf.zText == sMprintf.zBase)
    {
      zNew = malloc (sMprintf.nChar + 1);
      if (zNew)
        strcpy (zNew, zBuf);
    }
  else
    {
      zNew = realloc (sMprintf.zText, sMprintf.nChar + 1);
    }
  return zNew;
}

static const char _dig_vec[] = "0123456789ABCDEFGHIJKLMNOPQRSTUVWXYZ";
static char *
int2str (long int val, char *dst, int radix)
{
  char buffer[65];
  char *p;
  char *NullS = (char *) 0;

  if (radix < 0)
    {
      if (radix < -36 || radix > -2)
        return NullS;
      if (val < 0)
        {
          *dst++ = '-';
          val = -val;
        }
      radix = -radix;
    }
  else
    {
      if (radix > 36 || radix < 2)
        return NullS;
    }
  /*  The slightly contorted code which follows is due to the
     fact that few machines directly support unsigned long / and %.
     Certainly the VAX C compiler generates a subroutine call.  In
     the interests of efficiency (hollow laugh) I let this happen
     for the first digit only; after that "val" will be in range so
     that signed integer division will do.  Sorry 'bout that.
     CHECK THE CODE PRODUCED BY YOUR C COMPILER.  The first % and /
     should be unsigned, the second % and / signed, but C compilers
     tend to be extraordinarily sensitive to minor details of style.
     This works on a VAX, that's all I claim for it.
   */
  p = &buffer[sizeof (buffer) - 1];
  *p = '\0';
  *--p = _dig_vec[(unsigned long) val % (unsigned long) radix];
  val = (unsigned long) val / (unsigned long) radix;

#ifdef HAVE_LDIV
  while (val != 0)
    {
      ldiv_t res;
      res = ldiv (val, radix);
      *--p = _dig_vec[res.rem];
      val = res.quot;
    }

#else

  while (val != 0)
    {
      *--p = _dig_vec[val % radix];
      val /= radix;
    }

#endif
  while ((*dst++ = *p++) != 0);
  return dst - 1;
}
