/*
 * Portable Memory Management Interface for ANSI C,
 * with Extension for Win32 platforms supporting
 * multiple heaps.
 *
 * Copyright (C) 2000 Federico Spinazzi <fspinazzi@hotmail.com>
 *
 * Free Software License:
 *
 * All rights are reserved by the author, with the following exceptions:
 * Permission is granted to freely reproduce and distribute this software,
 * possibly in exchange for a fee, provided that this copyright notice
 * appears intact.
 *
 * Permission is also granted to adapt this software to produce
 * derivative works, as long as the modified versions carry this copyright
 * notice and additional notices stating that the work has been modified.
 *
 * This source code may be translated into executable form and incorporated
 * into proprietary software; there is no requirement for such software to
 * contain a copyright notice related to this source.
 *
 */

#ifndef _IMEM_H_
#define _IMEM_H_

#include <stdlib.h>
#include <string.h>
#include <stdio.h>
#include <time.h>

#include "amb.h"
#include "pmemfunc.h"
#include "imemdefs.h"
#include "except.h"
#include "imemdbug.h"

enum
  {
    THROW =     0x00000002, NO_THROW =     0x00000000,
    KEEP_AMB =  0x00000004, NO_KEEP_AMB =  0x00000000,
    USE_POOL =  0x00000008, NO_USE_POOL =  0x00000000,
    WITH_LOG =  0x00000010, WITH_NO_LOG =  0x00000000,
		WITH_CUSTOM_CLEANUP = 0x00000020,
		WITH_NO_CUSTOM_CLEANUP = 0x00000000,
    /* allow zero blocks when registering blocks*/
    WITH_ZB = 0x00000040, WITH_NO_ZB = 0x00000000,
		LOG_RANDOM_PROB = 0x00000080
  };

enum
  {
    YES = 1, NO = 0, OK = 1,
		AMB_IN_NODE_POOL = 1024
  };
enum
  {
    BY_MALLOC = 1, BY_CALLOC = 2, BY_REALLOC = 3
  };

typedef FILE stream_t;

/*
 * PUBLIC
 */

struct private_imem
{
  size_t flags;

  imem_malloc_t pmalloc;
  imem_calloc_t pcalloc;
  imem_realloc_t prealloc;
  imem_free_t pfree;

	char * malloc_name, * calloc_name, * realloc_name;
	char * free_name, *flags_name;

  void *opaque;

	int (*pdeinit)(struct private_imem *);

	/*
	 * This won't compile under MSVC ...
	 */
	unsigned long int 	seed;
	double failing_probability;
	double hprob;

  amb_list_t _amb_list;
  amb_list_t *amb_list;

	char _zb;
	void * zb;

  unsigned char on_failure;

  size_t size;
  size_t curr_mem_space;
  struct except_params *except_p;

  const char * log_stream_name;
  stream_t * log_stream;

	const char * name;
};

#ifdef IMEM_IMPLEMENTATION

typedef struct private_imem imem_t;

#else /* ! IMEM_IMPLEMENTATION */

typedef struct
  {
    char foo[sizeof (struct private_imem) / sizeof (char)];
  }
imem_t;

#endif /* IMEM_IMPLEMENTATION */

typedef int (*imem_deinit_t)(imem_t *);

typedef struct _imem_funcs_t
{
	size_t flags;
	imem_malloc_t  pmalloc;
	imem_calloc_t  pcalloc;
	imem_realloc_t  prealloc;
	imem_free_t pfree;
	void * opaque;
	imem_deinit_t pdeinit;
	unsigned char prob;
	char * malloc_name, * calloc_name, * realloc_name;
	char * free_name, *flags_name;
} imem_funcs_t;



#ifdef __cplusplus
extern "C"
{
#endif

/*
 * Memory management wrappers
 */

  extern void *a_malloc (imem_t * imem, size_t size, const char *file,
                         const int line, const char *function);

  extern void *a_realloc (imem_t * imem, void *ptr, size_t size,
                    const char *file, const int line, const char *function);

  extern void *a_calloc (imem_t * imem, size_t num, size_t size,
                    const char *file, const int line, const char *function);

  extern void a_free (imem_t * imem, void *ptr, const char *file,
                      const int line, const char *function);

  extern int a_init (imem_t * imem, const imem_funcs_t * imem_funcs,
				struct except_params *, const char *);

  extern int a_deinit (imem_t * imem, int flags);

  extern int a_init_ansi (imem_t * imem);
  extern void *a_ansi_malloc (size_t size, void *op);
  extern void *a_ansi_calloc (size_t num, size_t size, void *op);
  extern void *a_ansi_realloc (void *p, size_t size, void *op);
  extern void a_ansi_free (void *p, void *op);

  extern void a_free_mem_space (imem_t * imem, size_t mem_space);

  extern size_t a_set_mem_space (imem_t * imem, size_t mem_space);
  extern size_t a_get_mem_space (const imem_t * imem);
  extern size_t a_nbytes (imem_t * imem);
  extern size_t a_mem_space_nblocks (imem_t * imem, size_t ms);
  extern size_t a_mem_space_nbytes (imem_t * imem, size_t ms);

  extern void * a_get_opaque (imem_t * imem);
  extern void a_set_random_fail (imem_t * imem, double prob, unsigned long int seed);
  extern void a_random_test (imem_t * imem, size_t n);
	extern void a_init_funcs (imem_funcs_t * mf, int flags, imem_malloc_t fmalloc, imem_calloc_t fcalloc, imem_realloc_t frealloc, imem_free_t ffree, void *opaque, char * nflags, char * nmalloc, char * ncalloc, char * nrealloc, char * nfree);
/*
 * GLOBALS:
 * provided for simplicity in single threaded,
 * ANSI-C run-time based clients programming
 */

  extern imem_t a_global_s;
  extern imem_t *a_global;

#ifdef __cplusplus
}
#endif

#endif                          /* _IMEM_H_ */
