#define NODE_POOL_IMPLEMENTATION

#include "nodepool.h"
/*
 * from Thomas Niemann <niemann@yahoo.com> code
 * http://members.xoom.com
 */
#ifdef WITH_RCSID
static char *rcsid = "$Id: nodepool.c 1.6 2000/06/29 01:13:15 federico Exp $";
#endif

/*
 * TODO:
 * - testing
 * $Log: nodepool.c $
 * Revision 1.6  2000/06/29 01:13:15  federico
 * Uniformed some prototypes (int -> size_t).
 *
 * Revision 1.5  2000/05/22 23:52:21  federico
 * .
 *
 * Revision 1.4  2000/05/11 23:12:24  federico
 * removed a variable from partition realloc set
 *
 * Revision 1.3  2000/05/11 23:08:08  federico
 * Fixed a bug in partition_realloc_set: only the last pool added
 * was kept tracked and so free'd on destroy.
 *
 * Revision 1.2  2000/05/11 22:54:46  federico
 * Added a node_pool interface.
 * partition_handle interface is now removed, used only for internals
 *
 * Revision 1.1  2000/05/10 00:33:49  federico
 * Initial revision
 *
 */

/* revise to indicate maximum alignment requirements for your computer */
#define ALGN 4

/*
 * To make this code thread-safe, you'll need to implement critical regions.
 * These are demarked by the following macros.
 */
#define OS_ENTER_CRITICAL()
#define OS_EXIT_CRITICAL()

#define PLEN sizeof(part_type)  /* length of a pointer */

/*
 * TODO: custom allocator and deallocator
 */

/*
 *  A partition set consists of a pointer to the first free partition.
 *  This is followed by an array of partitions.  Each free partition in the
 *  partition set includes a pointer to the next free partition in the set.
 */

static partition_handle
partition_alloc_set (int n, size_t size, imem_malloc_t alloc, void *opaque)
{
/*
 *  Parameters:
 *    n         number of partitions to create
 *    size      the size, in bytes, of each partition
 *  Returns:
 *    A "handle" to the partition.  This handle is passed to
 *    allocPartition() to allocate a single partition.  If
 *    insufficient memory remains, NULL is returned.
 *  Description:
 *    A set of "n" partitions of size "size" are allocated.
 */

  part_type *p, *p0;
  size_t msize;
  unsigned long psize, hsize, tsize;
  int i;

  /* create "n" partitions of size "size", returning partition handle (ph) */

  /* each partition must hold a pointer for freechain */
  psize = size;
  if (psize < PLEN)
    psize = PLEN;

  /* and be a multiple of ALGN in size */
  psize = (((unsigned long) psize + (ALGN - 1)) / ALGN) * ALGN;

  /* the header must hold a pointer for freechain */
  hsize = PLEN;

  /* and be a multiple of ALGN in size */
  hsize = (((unsigned long) hsize + (ALGN - 1)) / ALGN) * ALGN;

  /* check for integer overflow, and allocate memory */
  tsize = n * (psize) + hsize;
  msize = tsize;
  if (msize != tsize)
    return NULL;

  p0 = IMEM_DEBUG_MALLOC (msize, opaque, alloc (msize, opaque));
  if (p0 == NULL)
    return NULL;

  /* establish free chain */
  OS_ENTER_CRITICAL ();
  p = (part_type *) ((char *) p0 + hsize);
  p0->next = p;                 /* head of freelist */
  for (i = 1; i < n; i++)
    {
      p->next = (part_type *) ((char *) p + psize);
      p = (part_type *) ((char *) p + psize);
    }

  /* end of list */
  p0->another = NULL;
  p->next = NULL;
  OS_EXIT_CRITICAL ();

  return p0;
}


/*
 * TODO:
 * - change h into part_type;
 */
static partition_handle
partition_realloc_set (partition_handle h, int n, size_t size,
                       imem_malloc_t alloc, void *opaque)
{
  partition_handle p0;
  unsigned long hsize;
  part_type *ph, *last;


  /* the header must hold a pointer for freechain */
  hsize = PLEN;

  /* and be a multiple of ALGN in size */
  hsize = (((unsigned long) hsize + (ALGN - 1)) / ALGN) * ALGN;

  p0 = partition_alloc_set (n, size, alloc, opaque);
  if (p0 == NULL)
    {
			DEBUGGING (
				fprintf (stderr, "imem debug: partition_realloc_set: failed\n");
				);
      return NULL;
    }

  ph = ((part_type *) h)->next; /* head of freelist */

  if (ph != NULL)
    {
      while (ph->next != NULL)
        {
          ph = ph->next;
        }
      ph->next = ((char *) p0 + hsize);
    }
  else
    {
      ((part_type *) h)->next = ((char *) p0 + hsize);
    }

  last = (part_type *) h;
  while (last->another != NULL)
    last = last->another;

  last->another = p0;

  return h;
}

static void
partition_free_set (part_type * ph, imem_free_t dealloc, void *opaque)
{
/*
 *  Parameters:
 *    ph        partition handle
 *  Description:
 *    The partitions originally created by allocPartitionSet() are freed.
 */

  assert (ph != NULL);
  assert (dealloc != NULL);

  /*
   * recursively free the list of partitions
   */
  if (ph->another != NULL)
    partition_free_set (ph->another, dealloc, opaque);

  IMEM_DEBUG_FREE (ph, opaque, dealloc (ph, opaque));
}

static void *
partition_alloc (part_type * const ph)
{
/*
 *  Parameters:
 *    ph        partition handle
 *  Returns:
 *    A pointer to a partition.  If no partitions remain, NULL is returned.
 *  Description:
 *    A partition is allocated, and the address of the partition is returned
 *    to caller.
 */

  part_type *p;

  /* allocate partition, returning pointer to memory (mem) */
  OS_ENTER_CRITICAL ();
  p = ph->next;
  if (p != NULL)
    {
      ph->next = p->next;
      p->next = ph;
    }

  OS_EXIT_CRITICAL ();
  return p;
}

static void
partition_free (part_type * const ph, void *const mem)
{
/*
 *  Parameters:
 *    ph        partition handle
 *    mem       pointer to partition
 *  Description:
 *    The partition originally allocated by allocPartition() is freed.
 */

  OS_ENTER_CRITICAL ();
  ((part_type *) (mem))->next = ph->next;
  ph->next = mem;
  OS_EXIT_CRITICAL ();
}


static void *
ansi_malloc (size_t size, void *opaque)
{
  return malloc (size);
}

static void
ansi_free (void *p, void *opaque)
{
  free (p);
}
/*
 * node_pool_init
 * node_pool_create
 */

int
node_pool_init (node_pool_t * np, imem_malloc_t pmalloc,
                imem_free_t pfree, void *opaque)
{
  imem_malloc_t pm;
  imem_free_t pf;
  assert (np != NULL);

  pm = (pmalloc == NULL) ? ansi_malloc : pmalloc;
  pf = (pfree == NULL) ? ansi_free : pfree;

  if (pmalloc == NULL || pfree == NULL)
    assert (pmalloc == NULL && pfree == NULL && opaque == NULL);

  np->pmalloc = pm;
  np->pfree = pf;
  np->opaque = opaque;
  np->ph = NULL;

  return NODE_POOL_OK;
}

int
node_pool_create (node_pool_t * np, int n, size_t size)
{
  assert (np != NULL);
  assert (np->pmalloc != NULL);

  np->ph = partition_alloc_set (n, size, np->pmalloc, np->opaque);
  if (np->ph == NULL)
    {
      return (!NODE_POOL_OK);
    }

  np->n = n;
  np->size = size;

  return NODE_POOL_OK;
}

void
node_pool_destroy (node_pool_t * np)
{
  assert (np != NULL);
  partition_free_set (np->ph, np->pfree, np->opaque);
}

void *
node_pool_get_node (node_pool_t * const np)
{
  void *p;
  assert (np != NULL);
  /*
   * if fails, reallocates !
   * allocator and opaque should be known
   */

  p = partition_alloc (np->ph);

  if (p == NULL)
    {
      /*
       * try to double the pool size ...
       */
      partition_handle new_partition;
      new_partition = partition_realloc_set (np->ph, np->n, np->size,
                                             np->pmalloc, np->opaque);

      if (new_partition == NULL)
        {
          /*
           * TODO:
           * - error: failed to extend the pool
           */
					DEBUGGING (
						fprintf (stderr, "imem debug: node_pool_get_node: "
							"failed to get a node pool\n");
						);
          return NULL;
        }
      else
        {
          np->ph = new_partition;
          p = partition_alloc (np->ph);
          assert (p != NULL);
        }
    }

  return p;
}

void
node_pool_free_node (node_pool_t * np, void *const mem)
{
  assert (np != NULL);
  assert (mem != NULL);
  /*
   * TODO:
   * if a partition_node is empty, frees it. ?
   */
  partition_free (np->ph, mem);

}


#ifdef TEST

#include <stdio.h>
#include <limits.h>

typedef struct
{
  char mem[1000];
}
s1;

typedef struct
  {
    double d;
		int i;
		float f;
		char c;
		s1 s;
		s1 * ps;
		double * pd;
		unsigned long int uli;
  }
s2;

static void *
my_malloc (size_t size, void *opaque)
{
  static int done = 0;

  if (!done)
    {
      done = 1;
      return malloc (size);
    }
  else
    return NULL;
}

static void
my_free (void *p, void *opaque)
{
  free (p);
}

#define SIZE1 500
#define SIZE2 2000
#define N1 20000

int
main (void)
{
  node_pool_t h1, h2;
  s1 *p1;
  s2 *p2, *p2b, *p2c, *p2cl[SIZE2];
  int i = 0, a = 0;
  int ALLOCATED;
	s2 * ps2;
	FILE * outfp;

  srand (time (NULL));
  ALLOCATED = rand () % SIZE2;

  fprintf (stderr, "To be allocated: %d, free: %d\n",
           ALLOCATED, SIZE2 - ALLOCATED);

  memset (&h1, 0, sizeof (h1));
  memset (&h2, 0, sizeof (h2));

  assert (node_pool_init (&h1, NULL, NULL, NULL) == NODE_POOL_OK);
  assert (node_pool_init (&h2, my_malloc, my_free, NULL) == NODE_POOL_OK);

  assert (node_pool_create (&h1, SIZE1, sizeof (s1)) == NODE_POOL_OK);
  assert (node_pool_create (&h2, SIZE2, sizeof (s2)) == NODE_POOL_OK);


  /* here's how you allocate one partition in each set */
  if ((p1 = node_pool_get_node (&h1)) == NULL)
    {
      fprintf (stderr, "unable to allocate p1\n");
      exit (1);
    }

  assert (ALLOCATED <= SIZE2);
  for (i = 0; i < ALLOCATED; ++i)
    {
      assert ((ps2 = node_pool_get_node (&h2)) != NULL);
      fprintf (stderr, "\rAllocated: %d", i);
    }
	fprintf(stderr, "\n");

  i = 0;
  a = 0;
  while (i < SIZE2)
    {
      ps2 = p2cl[i] = (s2 *) node_pool_get_node (&h2);
      if (p2cl[i] != NULL)
				{
	        a++;
					ps2->d = 23.3;
					ps2->i = 3;
					ps2->f = 3.3;
					ps2->c = 'x';
					strncpy (ps2->s.mem,
						"Hello World!",
						sizeof (ps2->s.mem) - 1);
					ps2->ps = &p2cl[i]->s;
					ps2->pd = &p2cl[i]->d;
					ps2->uli = 12;
				}
      fprintf (stderr, "\rAllocated: %d/%d", a, ++i);
    }
  fputs ("\n", stderr);

  i = 0;
	assert((outfp = fopen ("nodepool.out", "wt")) != NULL);
  while (i < SIZE2)
    {
      fprintf (stderr, "\rFreeing previuos: %d", i + 1);
      if (p2cl[i] != NULL)
        {
					fprintf (outfp, "s2.d = %g\n", p2cl[i]->d);
					fprintf (outfp, "s2.i = %d\n", p2cl[i]->i);
					fprintf (outfp, "s2.f = %f\n", p2cl[i]->f);
					fprintf (outfp, "s2.c = %c\n", p2cl[i]->c);
					fprintf (outfp, "s2.ps = %s\n", p2cl[i]->ps);
					fprintf (outfp, "s2.pd = %p\n", p2cl[i]->pd);
					fprintf (outfp, "s2.uli = %lu\n", p2cl[i]->uli);
          node_pool_free_node (&h2, p2cl[i]);
        }
      i++;
    }
  fputs ("\n", stderr);
	fclose (outfp);

  i = 0;
  while ((p2c = node_pool_get_node (&h2)) != NULL && i < N1)
    {
      fprintf (stderr, "\rAllocating: %d", ++i);
      /*
         node_pool_free_node (&h2, p2c);
       */
    }
  fputs ("\n", stderr);
  fprintf (stderr, "Free %d", i);
  assert (i == SIZE2 - ALLOCATED);


  /* here's how you free the allocated partitions */
  node_pool_free_node (&h1, p1);

  /* when done, free the partition sets */
  node_pool_destroy (&h1);
  node_pool_destroy (&h2);

  return 0;
}


#endif /* TEST */
