% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/AFcoxph.R
\name{AFcoxph}
\alias{AFcoxph}
\title{Attributable fraction function based on a Cox Proportional Hazard regression model as a \code{coxph} object (commonly used for cohort sampling designs with time-to-event outcomes).}
\usage{
AFcoxph(object, data, exposure, times, clusterid)
}
\arguments{
\item{object}{a fitted Cox Proportional Hazard regression model object of class "\code{\link[survival]{coxph}}". Method for handling ties must be breslow since this is assumed in the calculation of the standard errors. No special terms such as \code{cluster}, \code{strata} and \code{tt} is allowed in the formula for the fitted object.}

\item{data}{an optional data frame, list or environment (or object coercible by \code{as.data.frame} to a data frame) containing the variables in the model. If not found in data, the variables are taken from environment (\code{formula}), typically the environment from which the function is called.}

\item{exposure}{the name of the exposure variable as a string. The exposure must be binary (0/1) where unexposed is coded as 0.}

\item{times}{a scalar or vector of time points specified by the user for which the attributable fraction function is estimated. If not specified the observed event times will be used.}

\item{clusterid}{the name of the cluster identifier variable as a string, if data are clustered. Cluster robust standard errors will be calculated.}
}
\value{
\item{AF.est}{estimated attributable fraction function for every time point specified by \code{times}.}

\item{AF.var}{estimated variance of \code{AF.est}. The variance is obtained by combining the delta methods with the sandwich formula.}

\item{S.est}{estimated factual survival function; \eqn{S(t)}.}

\item{S.var}{estimated variance of \code{S.est}. The variance is obtained by the sandwich formula.}

\item{S0.est}{estimated counterfactual survival function if exposure would be eliminated; \eqn{S_0(t)}{S0(t)}.}

\item{S0.var}{estimated variance of \code{S0.est}. The variance is obtained by the sandwich formula.}
}
\description{
\code{AFcoxph} estimates the model-based adjusted attributable fraction function from a Cox Proportional Hazard regression model in form of a  \code{\link[survival]{coxph}} object. This model is commonly used for data from cohort sampling designs with time-to-event outcomes.
}
\details{
\code{AFcoxph} estimates the attributable fraction for a time-to-event outcome
under the hypothetical scenario where a binary exposure \code{X} is eliminated from the population. The estimate is adjusted for confounders \code{Z}
by the Cox proportional hazards model (\code{\link[survival]{coxph}}). Let the AF function be defined as
\deqn{AF=1-\frac{\{1-S_0(t)\}}{\{1-S(t)\}}}{AF = 1 - {1 - S0(t)} / {1 - S(t)}}
where \eqn{S_0(t)}{S0(t)} denotes the counterfactual survival function for the event if
the exposure would have been eliminated from the population at baseline and \eqn{S(t)} denotes the factual survival function.
If \code{Z} is sufficient for confounding control, then \eqn{S_0(t)}{S0(t)} can be expressed as \eqn{E_Z\{S(t\mid{X=0,Z })\}}{E_z{S(t|X=0,Z)}}.
The function uses a fitted Cox proportional hazards regression to estimate \eqn{S(t\mid{X=0,Z})}{S(t|X=0,Z)}, and the marginal sample distribution of \code{Z}
to approximate the outer expectation (\enc{Sjölander}{Sjolander} and Vansteelandt, 2014).  If \code{clusterid} is supplied, then a clustered sandwich formula is used in all variance calculations.
}
\examples{
# Simulate a sample from a cohort sampling design with time-to-event outcome
expit <- function(x) 1 / (1 + exp( - x))
n <- 500
time <- c(seq(from = 0.2, to = 1, by = 0.2))
Z <- rnorm(n = n)
X <- rbinom(n = n, size = 1, prob = expit(Z))
Tim <- rexp(n = n, rate = exp(X + Z))
C <- rexp(n = n, rate = exp(X + Z))
Tobs <- pmin(Tim, C)
D <- as.numeric(Tobs < C)
#Ties created by rounding
Tobs <- round(Tobs, digits = 2)

# Example 1: non clustered data from a cohort sampling design with time-to-event outcomes
data <- data.frame(Tobs, D, X,  Z)

# Fit a Cox PH regression model
fit <- coxph(formula = Surv(Tobs, D) ~ X + Z + X * Z, data = data, ties="breslow")

# Estimate the attributable fraction from the fitted Cox PH regression model
AFcoxph_est <- AFcoxph(fit, data=data, exposure ="X", times = time)
summary(AFcoxph_est)

# Example 2: clustered data from a cohort sampling design with time-to-event outcomes
# Duplicate observations in order to create clustered data
id <- rep(1:n, 2)
data <- data.frame(Tobs = c(Tobs, Tobs), D = c(D, D), X = c(X, X), Z = c(Z, Z), id = id)

# Fit a Cox PH regression model
fit <- coxph(formula = Surv(Tobs, D) ~ X + Z + X * Z, data = data, ties="breslow")

# Estimate the attributable fraction from the fitted Cox PH regression model
AFcoxph_clust <- AFcoxph(object = fit, data = data,
                         exposure = "X", times = time, clusterid = "id")
summary(AFcoxph_clust)
plot(AFcoxph_clust, CI = TRUE)

# Estimate the attributable fraction from the fitted Cox PH regression model, time unspecified
AFcoxph_clust_no_time <- AFcoxph(object = fit, data = data,
                         exposure = "X", clusterid = "id")
summary(AFcoxph_clust_no_time)
plot(AFcoxph_clust, CI = TRUE)
}
\references{
Chen, L., Lin, D. Y., and Zeng, D. (2010). Attributable fraction functions for censored event times. \emph{Biometrika} \bold{97}, 713-726.

\enc{Sjölander}{Sjolander}, A. and Vansteelandt, S. (2014). Doubly robust estimation of attributable fractions in survival analysis. \emph{Statistical Methods in Medical Research}. doi: 10.1177/0962280214564003.
}
\seealso{
\code{\link[survival]{coxph}} and \code{\link[survival]{Surv}} used for fitting the Cox proportional hazards model.
}
\author{
Elisabeth Dahlqwist, Arvid \enc{Sjölander}{Sjolander}
}
