% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/first_isolate.R
\name{first_isolate}
\alias{first_isolate}
\title{Determine first (weighted) isolates}
\source{
Methodology of this function is based on: \strong{M39 Analysis and Presentation of Cumulative Antimicrobial Susceptibility Test Data, 4th Edition}, 2014, \emph{Clinical and Laboratory Standards Institute (CLSI)}. \url{https://clsi.org/standards/products/microbiology/documents/m39/}.
}
\usage{
first_isolate(tbl, col_date = NULL, col_patient_id = NULL,
  col_mo = NULL, col_testcode = NULL, col_specimen = NULL,
  col_icu = NULL, col_keyantibiotics = NULL, episode_days = 365,
  testcodes_exclude = NULL, icu_exclude = FALSE,
  filter_specimen = NULL, output_logical = TRUE,
  type = "keyantibiotics", ignore_I = TRUE, points_threshold = 2,
  info = TRUE, col_bactid = NULL, col_genus = NULL,
  col_species = NULL)
}
\arguments{
\item{tbl}{a \code{data.frame} containing isolates.}

\item{col_date}{column name of the result date (or date that is was received on the lab), defaults to the first column of class \code{Date}}

\item{col_patient_id}{column name of the unique IDs of the patients, defaults to the first column that starts with 'patient' (case insensitive)}

\item{col_mo}{column name of the unique IDs of the microorganisms (see \code{\link{mo}}), defaults to the first column of class \code{mo}. Values will be coerced using \code{\link{as.mo}}.}

\item{col_testcode}{column name of the test codes. Use \code{col_testcode = NA} to \strong{not} exclude certain test codes (like test codes for screening). In that case \code{testcodes_exclude} will be ignored. Supports tidyverse-like quotation.}

\item{col_specimen}{column name of the specimen type or group}

\item{col_icu}{column name of the logicals (\code{TRUE}/\code{FALSE}) whether a ward or department is an Intensive Care Unit (ICU)}

\item{col_keyantibiotics}{column name of the key antibiotics to determine first \emph{weighted} isolates, see \code{\link{key_antibiotics}}. Supports tidyverse-like quotation.}

\item{episode_days}{episode in days after which a genus/species combination will be determined as 'first isolate' again}

\item{testcodes_exclude}{character vector with test codes that should be excluded (case-insensitive)}

\item{icu_exclude}{logical whether ICU isolates should be excluded}

\item{filter_specimen}{specimen group or type that should be excluded}

\item{output_logical}{return output as \code{logical} (will else be the values \code{0} or \code{1})}

\item{type}{type to determine weighed isolates; can be \code{"keyantibiotics"} or \code{"points"}, see Details}

\item{ignore_I}{logical to determine whether antibiotic interpretations with \code{"I"} will be ignored when \code{type = "keyantibiotics"}, see Details}

\item{points_threshold}{points until the comparison of key antibiotics will lead to inclusion of an isolate when \code{type = "points"}, see Details}

\item{info}{print progress}

\item{col_bactid}{(deprecated, use \code{col_mo} instead)}

\item{col_genus}{(deprecated, use \code{col_mo} instead) column name of the genus of the microorganisms}

\item{col_species}{(deprecated, use \code{col_mo} instead) column name of the species of the microorganisms}
}
\value{
A vector to add to table, see Examples.
}
\description{
Determine first (weighted) isolates of all microorganisms of every patient per episode and (if needed) per specimen type.
}
\details{
\strong{WHY THIS IS SO IMPORTANT} \cr
    To conduct an analysis of antimicrobial resistance, you should only include the first isolate of every patient per episode \href{https://www.ncbi.nlm.nih.gov/pubmed/17304462}{[1]}. If you would not do this, you could easily get an overestimate or underestimate of the resistance of an antibiotic. Imagine that a patient was admitted with an MRSA and that it was found in 5 different blood cultures the following week. The resistance percentage of oxacillin of all \emph{S. aureus} isolates would be overestimated, because you included this MRSA more than once. It would be \href{https://en.wikipedia.org/wiki/Selection_bias}{selection bias}.
}
\section{Key antibiotics}{

    There are two ways to determine whether isolates can be included as first \emph{weighted} isolates which will give generally the same results: \cr

    \strong{1. Using} \code{type = "keyantibiotics"} \strong{and parameter} \code{ignore_I} \cr
    Any difference from S to R (or vice versa) will (re)select an isolate as a first weighted isolate. With \code{ignore_I = FALSE}, also differences from I to S|R (or vice versa) will lead to this. This is a reliable method and 30-35 times faster than method 2. \cr

    \strong{2. Using} \code{type = "points"} \strong{and parameter} \code{points_threshold} \cr
    A difference from I to S|R (or vice versa) means 0.5 points, a difference from S to R (or vice versa) means 1 point. When the sum of points exceeds \code{points_threshold}, an isolate will be (re)selected as a first weighted isolate.
}

\examples{
# septic_patients is a dataset available in the AMR package. It is true, genuine data.
?septic_patients

library(dplyr)
my_patients <- septic_patients \%>\%
  mutate(first_isolate = first_isolate(.,
                                       col_date = "date",
                                       col_patient_id = "patient_id",
                                       col_mo = "mo"))

# Now let's see if first isolates matter:
A <- my_patients \%>\%
  group_by(hospital_id) \%>\%
  summarise(count = n_rsi(gent),            # gentamicin availability
            resistance = portion_IR(gent))  # gentamicin resistance

B <- my_patients \%>\%
  filter(first_isolate == TRUE) \%>\%         # the 1st isolate filter
  group_by(hospital_id) \%>\%
  summarise(count = n_rsi(gent),            # gentamicin availability
            resistance = portion_IR(gent))  # gentamicin resistance

# Have a look at A and B.
# B is more reliable because every isolate is only counted once.
# Gentamicin resitance in hospital D appears to be 5.4\% higher than
# when you (erroneously) would have used all isolates!

## OTHER EXAMPLES:

\dontrun{

# set key antibiotics to a new variable
tbl$keyab <- key_antibiotics(tbl)

tbl$first_isolate <-
  first_isolate(tbl)

tbl$first_isolate_weighed <-
  first_isolate(tbl,
                col_keyantibiotics = 'keyab')

tbl$first_blood_isolate <-
  first_isolate(tbl,
                filter_specimen = 'Blood')

tbl$first_blood_isolate_weighed <-
  first_isolate(tbl,
                filter_specimen = 'Blood',
                col_keyantibiotics = 'keyab')

tbl$first_urine_isolate <-
  first_isolate(tbl,
                filter_specimen = 'Urine')

tbl$first_urine_isolate_weighed <-
  first_isolate(tbl,
                filter_specimen = 'Urine',
                col_keyantibiotics = 'keyab')

tbl$first_resp_isolate <-
  first_isolate(tbl,
                filter_specimen = 'Respiratory')

tbl$first_resp_isolate_weighed <-
  first_isolate(tbl,
                filter_specimen = 'Respiratory',
                col_keyantibiotics = 'keyab')
}
}
\seealso{
\code{\link{key_antibiotics}}
}
\keyword{first}
\keyword{isolate}
\keyword{isolates}
