% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/bounds_test.R
\name{bounds_t_test}
\alias{bounds_t_test}
\title{Bounds t-test for no cointegration}
\usage{
bounds_t_test(
  object,
  case,
  alpha = NULL,
  pvalue = TRUE,
  exact = FALSE,
  R = 40000,
  vcov_matrix = NULL
)
}
\arguments{
\item{object}{An object of \code{\link[base]{class}} 'ardl' or 'uecm'.}

\item{case}{An integer (1, 3 or 5) or a character string specifying whether
the 'intercept' and/or the 'trend' have to participate in the
short-run relationship (see section 'Cases' below). Note that the t-bounds
test can't be applied for cases 2 and 4.}

\item{alpha}{A numeric value between 0 and 1 indicating the significance
level of the critical value bounds. If \code{NULL} (default), no critical
value bounds for a specific level of significance are provide, only the
p-value. See section 'alpha, bounds and p-value' below for details.}

\item{pvalue}{A logical indicating whether you want the p-value to be
provided. The default is \code{TRUE}. See section 'alpha, bounds and
p-value' below for details.}

\item{exact}{A logical indicating whether you want asymptotic (T = 1000) or
exact sample size critical value bounds and p-value. The default is
\code{FALSE} for asymptotic. See section 'alpha, bounds and p-value' below
for details.}

\item{R}{An integer indicating how many iterations will be used if
\code{exact = TRUE}. Default is 40000.}

\item{vcov_matrix}{The estimated covariance matrix of the random variable
that the test uses to estimate the test statistic. The default is
\code{vcov(object)} (when \code{vcov_matrix = NULL}), but other estimations
of the covariance matrix of the regression's estimated coefficients can
also be used (e.g., using \code{\link[sandwich]{vcovHC}} or
\code{\link[sandwich]{vcovHAC}}). Only applicable if the input object is of
class "uecm".}
}
\value{
A list with class "htest" containing the following components:
  \item{\code{method}}{a character string indicating what type of test was
     performed.}
  \item{\code{alternative}}{a character string describing the alternative
     hypothesis.}
  \item{\code{statistic}}{the value of the test statistic.}
  \item{\code{null.value}}{the value of the population parameters \code{k}
     (the number of independent variables) and \code{T} (the number of
     observations) specified by the null hypothesis.}
  \item{\code{data.name}}{a character string giving the name(s) of the data.}
  \item{\code{parameters}}{numeric vector containing the critical value
     bounds.}
  \item{\code{p.value}}{the p-value of the test.}
  \item{\code{PSS2001parameters}}{numeric vector containing the critical
     value bounds as presented by \cite{Pesaran et al. (2001)}. See section
     alpha, bounds and p-value' below for details.}
  \item{\code{tab}}{data.frame containing the statistic, the critical value
     bounds, the alpha level of significance and the p-value.}
}
\description{
\code{bounds_t_test} performs the t-bounds test for no cointegration
\cite{Pesaran et al. (2001)}. It is a t-test on the parameters of a UECM
(Unrestricted Error Correction Model).
}
\section{Hypothesis testing}{
 \deqn{\Delta y_{t} = c_{0} + c_{1}t +
  \pi_{y}y_{t-1} + \sum_{j=1}^{k}\pi_{j}x_{j,t-1} +
  \sum_{i=1}^{p-1}\psi_{y,i}\Delta y_{t-i} +
  \sum_{j=1}^{k}\sum_{l=1}^{q_{j}-1} \psi_{j,l}\Delta x_{j,t-l} +
  \sum_{j=1}^{k}\omega_{j}\Delta x_{j,t} + \epsilon_{t}}
  \deqn{\mathbf{H_{0}:} \pi_{y} = 0}
  \deqn{\mathbf{H_{1}:} \pi_{y} \neq 0}
}

\section{alpha, bounds and p-value}{
 In this section it is explained how the
  critical value bounds and p-values are obtained.
  \itemize{
     \item If \code{exact = FALSE}, then the asymptotic (T = 1000) critical
         value bounds and p-value are provided.
     \item Only the asymptotic critical value bounds and p-values, and only
         for k <= 10 are precalculated, everything else has to be computed.
     \item Precalculated critical value bounds and p-values were simulated
         using \code{set.seed(2020)} and \code{R = 70000}.
     \item Precalculated critical value bounds exist only for \code{alpha}
         being one of the 0.005, 0.01, 0.025, 0.05, 0.075, 0.1, 0.15 or 0.2,
         everything else has to be computed.
     \item If \code{alpha} is one of the 0.1, 0.05, 0.025 or 0.01 (and
         \code{exact = FALSE} and k <= 10), \code{PSS2001parameters} shows
         the critical value bounds presented in \cite{Pesaran et al. (2001)}
         (less precise).
  }
}

\section{Cases}{
 According to \cite{Pesaran et al. (2001)}, we distinguish the
long-run relationship (cointegrating equation) (and thus the bounds-test and
the Restricted ECMs) between 5 different cases. These differ in terms of
whether the 'intercept' and/or the 'trend' are restricted to participate in
the long-run relationship or they are unrestricted and so they participate in
the short-run relationship.

\describe{
  \item{Case 1:}{\itemize{
    \item No \emph{intercept} and no \emph{trend}.
    \item \code{case} inputs: 1 or "n" where "n" stands for none.}}
  \item{Case 2:}{\itemize{
    \item Restricted \emph{intercept} and no \emph{trend}.
    \item \code{case} inputs: 2 or "rc" where "rc" stands for restricted
     constant.}}
  \item{Case 3:}{\itemize{
    \item Unrestricted \emph{intercept} and no \emph{trend}.
    \item \code{case} inputs: 3 or "uc" where "uc" stands for unrestricted
    constant.}}
  \item{Case 4:}{\itemize{
    \item Unrestricted \emph{intercept} and restricted \emph{trend}.
    \item \code{case} inputs: 4 or "ucrt" where "ucrt" stands for
    unrestricted constant and restricted trend.}}
  \item{Case 5:}{\itemize{
    \item Unrestricted \emph{intercept} and unrestricted \emph{trend}.
    \item \code{case} inputs: 5 or "ucut" where "ucut" stands for
     unrestricted constant and unrestricted trend.}}
}

Note that you can't restrict (or leave unrestricted) a parameter that doesn't
exist in the input model. For example, you can't compute \code{recm(object,
case=3)} if the object is an ARDL (or UECM) model with no intercept. The same
way, you can't compute \code{bounds_f_test(object, case=5)} if the object is
an ARDL (or UECM) model with no linear trend.
}

\section{References}{
 Pesaran, M. H., Shin, Y., & Smith, R. J. (2001). Bounds
  testing approaches to the analysis of level relationships. \emph{Journal of
  Applied Econometrics}, 16(3), 289-326
}

\examples{
data(denmark)

## How to use cases under different models (regarding deterministic terms)

## Construct an ARDL(3,1,3,2) model with different deterministic terms -

# Without constant
ardl_3132_n <- ardl(LRM ~ LRY + IBO + IDE -1, data = denmark, order = c(3,1,3,2))

# With constant
ardl_3132_c <- ardl(LRM ~ LRY + IBO + IDE, data = denmark, order = c(3,1,3,2))

# With constant and trend
ardl_3132_ct <- ardl(LRM ~ LRY + IBO + IDE + trend(LRM), data = denmark, order = c(3,1,3,2))

## t-bounds test for no level relationship (no cointegration) ----------

# For the model without a constant
bounds_t_test(ardl_3132_n, case = 1)
# or
bounds_t_test(ardl_3132_n, case = "n")

# For the model with a constant
# Including the constant term in the short-run relationship (unrestricted constant)
bounds_t_test(ardl_3132_c, case = "uc")
# or
bounds_t_test(ardl_3132_c, case = 3)

# For the model with constant and trend
# Including the constant term and the trend in the short-run relationship
# (unrestricted constant and unrestricted trend)
bounds_t_test(ardl_3132_ct, case = "ucut")
# or
bounds_t_test(ardl_3132_ct, case = 5)

## Note that you can't use bounds t-test for cases 2 and 4, or use a wrong model

# For example, the following tests will produce an error:
\dontrun{
bounds_t_test(ardl_3132_n, case = 2)
bounds_t_test(ardl_3132_c, case = 4)
bounds_t_test(ardl_3132_ct, case = 3)
}

## Asymptotic p-value and critical value bounds (assuming T = 1000) ----

# Include critical value bounds for a certain level of significance

# t-statistic is larger than the I(1) bound (for a=0.05) as expected (p-value < 0.05)
btt <- bounds_t_test(ardl_3132_c, case = 3, alpha = 0.05)
btt
btt$tab

# Traditional but less precise critical value bounds, as presented in Pesaran et al. (2001)
btt$PSS2001parameters

# t-statistic doesn't exceed the I(1) bound (for a=0.005) as p-value is greater than 0.005
bounds_t_test(ardl_3132_c, case = 3, alpha = 0.005)

## Exact sample size p-value and critical value bounds -----------------

# Setting a seed is suggested to allow the replication of results
# 'R' can be increased for more accurate resutls

# t-statistic is smaller than the I(1) bound (for a=0.01) as expected (p-value > 0.01)
# Note that the exact sample p-value (0.009874) is very different than the asymptotic (0.005538)
# It can take more than 90 seconds
\dontrun{
set.seed(2020)
bounds_t_test(ardl_3132_c, case = 3, alpha = 0.01, exact = TRUE)
}
}
\seealso{
\code{\link{bounds_f_test}} \code{\link{ardl}} \code{\link{uecm}}
}
\author{
Kleanthis Natsiopoulos, \email{klnatsio@gmail.com}
}
\keyword{htest}
\keyword{ts}
