% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/workflowfunction.R
\name{RowTest_Fisher}
\alias{RowTest_Fisher}
\title{Apply Fisher Exact Test on Bicluster Rows}
\usage{
RowTest_Fisher(result, matrix, p.adjust = "BH", alpha = 0.05,
  pattern = NULL)
}
\arguments{
\item{result}{A Biclust or BiBitWorkflow Object.}

\item{matrix}{Accompanying binary data matrix which was used to obtain \code{result}.}

\item{p.adjust}{Which method to use when adjusting p-values, see \code{\link[stats]{p.adjust}} (default=\code{"BH"}).}

\item{alpha}{Significance level (adjusted p-values) when constructing the \code{FisherInfo} object (default=0.05).}

\item{pattern}{Numeric vector for which patterns/biclusters the Fisher Exact Test needs to be computed (default = all patterns/biclusters).}
}
\value{
Depending on \code{result}, a \code{FisherResult} and/or \code{FisherInfo} object will be added to the \code{result} and returned (see Details).
}
\description{
Accepts a Biclust or BiBitWorkflow result and applies the Fisher Exact Test for each row (see Details).
}
\details{
Extracts the patterns from either a \code{Biclust} or \code{BiBitWorkflow} object (see below). 
Afterwards for each pattern all rows will be tested using the Fisher Exact Test. This test compares the part of the row inside the pattern (of the bicluster) with the part of the row outside the pattern.
The Fisher Exact Test gives you some information on if the row is uniquely active for this pattern.

Depending on the \code{result} input, different patterns will be extract and different info will be returned:
\describe{
\item{\emph{Biclust S4 Object}}{

Using the column patterns of the Biclust result, all rows are tested using the Fisher Exact Test. 
Afterwards the following 2 objects are added to the \code{info} slot of the Biclust object:
\itemize{
\item \code{FisherResult}: A list object (one element for each pattern) of data frames (Number of Rows \eqn{\times} 6) which contain the names of the rows (\code{Names}), the noise level of the row inside the pattern (\code{Noise}), the signal percentage inside the pattern (\code{InsidePerc1}), the signal percentage outside the pattern (\code{OutsidePerc1}), the p-value of the Fisher Exact Test (\code{Fisher_pvalue}) and the adjusted p-value of the Fisher Exact Test (\code{Fisher_pvalue_adj}).
\item \code{FisherInfo}: Info object which contains a comparison of the current row membership for each pattern with a 'new' row membership based on the significant rows (from the Fisher Exact Test) for each pattern.
It is a list object (one element for each pattern) of lists (6 elements). These list objects per pattern contain the number of new, removed and identical rows (\code{NewRows}, \code{RemovedRows}, \code{SameRows}) when comparing the significant rows with the original row membership (as well as their indices (\code{NewRows_index}, \code{RemovedRows_index})). The \code{MaxNoise} element contains the maximum noise of all Fisher significant rows. 
} 
}
\item{\emph{BiBitWorkflow S3 Object}}{ 

The merged column patterns (after cutting the hierarchical tree) are extracted from the BiBitWorkflow object, namely the \code{$info$MergedColPatterns} slot. 
Afterwards the following object is added to the \code{$info} slot of the BiBitWorkflow object:
\itemize{
\item \code{FisherResult}: Same as above
}
}
}
}
\examples{
\dontrun{
## Prepare some data ##
set.seed(254)
mat <- matrix(sample(c(0,1),5000*50,replace=TRUE,prob=c(1-0.15,0.15)),
              nrow=5000,ncol=50)
mat[1:200,1:10] <- matrix(sample(c(0,1),200*10,replace=TRUE,prob=c(1-0.9,0.9)),
                          nrow=200,ncol=10)
mat[300:399,6:15] <- matrix(sample(c(0,1),100*10,replace=TRUE,prob=c(1-0.9,0.9)),
                            nrow=100,ncol=10)
mat[400:599,21:30] <- matrix(sample(c(0,1),200*10,replace=TRUE,prob=c(1-0.9,0.9)),
                             nrow=200,ncol=10)
mat[700:799,29:38] <- matrix(sample(c(0,1),100*10,replace=TRUE,prob=c(1-0.9,0.9)),
                             nrow=100,ncol=10)
mat <- mat[sample(1:5000,5000,replace=FALSE),sample(1:50,50,replace=FALSE)]

## Apply BiBitWorkflow ##
out <- BiBitWorkflow(matrix=mat,minr=50,minc=5,noise=0.2,cut_type="number",cut_pm=4)

## Apply RowTest_Fisher on Biclust Object -> returns Biclust Object ##
out_new <- RowTest_Fisher(result=out$Biclust,matrix=mat)
# FisherResult output in info slot
str(out_new@info$FisherResult)
# FisherInfo output in info slot (comparison with original BC's)
str(out_new@info$FisherInfo)


## Apply RowTest_Fisher on BiBitWorkflow Object -> returns BiBitWorkflow Object ##
out_new2 <- RowTest_Fisher(result=out,matrix=mat)
# FisherResult output in BiBitWorkflow info element
str(out_new2$info$FisherResult)
# Fisher output is added to "NoiseScree" plot
NoiseScree(result=out_new2,matrix=mat,type="Added")
}
}
\author{
Ewoud De Troyer
}

