% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/knndm.R
\name{knndm}
\alias{knndm}
\title{K-fold Nearest Neighbour Distance Matching}
\usage{
knndm(
  tpoints,
  modeldomain = NULL,
  ppoints = NULL,
  space = "geographical",
  k = 10,
  maxp = 0.5,
  clustering = "hierarchical",
  linkf = "ward.D2",
  samplesize = 1000,
  sampling = "regular"
)
}
\arguments{
\item{tpoints}{sf or sfc point object. Contains the training points samples.}

\item{modeldomain}{sf polygon object defining the prediction area. Optional; alternative to ppoints (see Details).}

\item{ppoints}{sf or sfc point object. Contains the target prediction points. Optional; alternative to modeldomain (see Details).}

\item{space}{character. Only "geographical" knndm, i.e. kNNDM in the geographical space, is currently implemented.}

\item{k}{integer. Number of folds desired for CV. Defaults to 10.}

\item{maxp}{numeric. Maximum fold size allowed, defaults to 0.5, i.e. a single fold can hold a maximum of half of the training points.}

\item{clustering}{character. Possible values include "hierarchical" and "kmeans". See details.}

\item{linkf}{character. Only relevant if clustering = "hierarchical". Link function for agglomerative hierarchical clustering.
Defaults to "ward.D2". Check `stats::hclust` for other options.}

\item{samplesize}{numeric. How many points in the modeldomain should be sampled as prediction points?
Only required if modeldomain is used instead of ppoints.}

\item{sampling}{character. How to draw prediction points from the modeldomain? See `sf::st_sample`.
Only required if modeldomain is used instead of ppoints.}
}
\value{
An object of class \emph{knndm} consisting of a list of eight elements:
indx_train, indx_test (indices of the observations to use as
training/test data in each kNNDM CV iteration), Gij (distances for
G function construction between prediction and target points), Gj
(distances for G function construction during LOO CV), Gjstar (distances
for modified G function during kNNDM CV), clusters (list of cluster IDs),
W (Wasserstein statistic), and space (stated by the user in the function call).
}
\description{
This function implements the kNNDM algorithm and returns the necessary
indices to perform a k-fold NNDM CV for map validation.
}
\details{
knndm is a k-fold version of NNDM LOO CV for medium and large datasets. Brielfy, the algorithm tries to
find a k-fold configuration such that the integral of the absolute differences (Wasserstein W statistic)
between the empirical nearest neighbour distance distribution function between the test and training data during CV (Gj*),
and the empirical nearest neighbour distance distribution function between the prediction and training points (Gij),
is minimised. It does so by performing clustering of the training points' coordinates for different numbers of
clusters that range from k to N (number of observations), merging them into k final folds,
and selecting the configuration with the lowest W.

Using a projected CRS in `knndm` has large computational advantages since fast nearest neighbour search can be
done via the `FNN` package, while working with geographic coordinates requires computing the full
spherical distance matrices. As a clustering algorithm, `kmeans` can only be used for
projected CRS while `hierarchical` can work with both projected and geographical coordinates, though it requires
calculating the full distance matrix of the training points even for a projected CRS.

In order to select between clustering algorithms and number of folds `k`, different `knndm` configurations can be run
and compared, being the one with a lower W statistic the one that offers a better match. W statistics between `knndm`
runs are comparable as long as `tpoints` and `ppoints` or `modeldomain` stay the same.

Map validation using knndm should be used using `CAST::global_validation`, i.e. by stacking all out-of-sample
predictions and evaluating them all at once. The reasons behind this are 1) The resulting folds can be
unbalanced and 2) nearest neighbour functions are constructed and matched using all CV folds simultaneously.

If training data points are very clustered with respect to the prediction area and the presented knndm
configuration still show signs of Gj* > Gij, there are several things that can be tried. First, increase
the `maxp` parameter; this may help to control for strong clustering (at the cost of having unbalanced folds).
Secondly, decrease the number of final folds `k`, which may help to have larger clusters.

The `modeldomain` is a sf polygon that defines the prediction area. The function takes a regular point sample
(amount defined by `samplesize`) from the spatial extent. As an alternative use `ppoints` instead of
`modeldomain`, if you have already defined the prediction locations (e.g. raster pixel centroids).
When using either `modeldomain` or `ppoints`, we advise to plot the study area polygon and the
training/prediction points as a previous step to ensure they are aligned.
}
\note{
Experimental cycle. Article describing and testing the algorithm in preparation.
}
\examples{
########################################################################
# Example 1: Simulated data - Randomly-distributed training points
########################################################################

library(sf)
library(ggplot2)

# Simulate 1000 random training points in a 100x100 square
set.seed(1234)
simarea <- list(matrix(c(0,0,0,100,100,100,100,0,0,0), ncol=2, byrow=TRUE))
simarea <- sf::st_polygon(simarea)
train_points <- sf::st_sample(simarea, 1000, type = "random")
pred_points <- sf::st_sample(simarea, 1000, type = "regular")
plot(simarea)
plot(pred_points, add = TRUE, col = "blue")
plot(train_points, add = TRUE, col = "red")

# Run kNNDM for the whole domain, here the prediction points are known.
knndm_folds <- knndm(train_points, ppoints = pred_points, k = 5)
knndm_folds
plot(knndm_folds)
folds <- as.character(knndm_folds$clusters)
ggplot() +
  geom_sf(data = simarea, alpha = 0) +
  geom_sf(data = train_points, aes(col = folds))

########################################################################
# Example 2: Simulated data - Clustered training points
########################################################################
\dontrun{
library(sf)
library(ggplot2)

# Simulate 1000 clustered training points in a 100x100 square
set.seed(1234)
simarea <- list(matrix(c(0,0,0,100,100,100,100,0,0,0), ncol=2, byrow=TRUE))
simarea <- sf::st_polygon(simarea)
train_points <- clustered_sample(simarea, 1000, 50, 5)
pred_points <- sf::st_sample(simarea, 1000, type = "regular")
plot(simarea)
plot(pred_points, add = TRUE, col = "blue")
plot(train_points, add = TRUE, col = "red")

# Run kNNDM for the whole domain, here the prediction points are known.
knndm_folds <- knndm(train_points, ppoints = pred_points, k = 5)
knndm_folds
plot(knndm_folds)
folds <- as.character(knndm_folds$clusters)
ggplot() +
  geom_sf(data = simarea, alpha = 0) +
  geom_sf(data = train_points, aes(col = folds))
}
########################################################################
# Example 3: Real- world example; using a modeldomain instead of previously
# sampled prediction locations
########################################################################
\dontrun{
library(sf)
library(terra)
library(ggplot2)

### prepare sample data:
dat <- readRDS(system.file("extdata","Cookfarm.RDS",package="CAST"))
dat <- aggregate(dat[,c("DEM","TWI", "NDRE.M", "Easting", "Northing","VW")],
   by=list(as.character(dat$SOURCEID)),mean)
pts <- dat[,-1]
pts <- st_as_sf(pts,coords=c("Easting","Northing"))
st_crs(pts) <- 26911
studyArea <- rast(system.file("extdata","predictors_2012-03-25.tif",package="CAST"))
studyArea[!is.na(studyArea)] <- 1
studyArea <- as.polygons(studyArea, values = FALSE, na.all = TRUE) |>
    st_as_sf() |>
    st_union()
pts <- st_transform(pts, crs = st_crs(studyArea))
plot(studyArea)
plot(st_geometry(pts), add = TRUE, col = "red")

knndm_folds <- knndm(pts, modeldomain=studyArea, k = 5)
knndm_folds
plot(knndm_folds)
folds <- as.character(knndm_folds$clusters)
ggplot() +
  geom_sf(data = pts, aes(col = folds))

#use for cross-validation:
library(caret)
ctrl <- trainControl(method="cv",
   index=knndm_folds$indx_train,
   savePredictions='final')
model_knndm <- train(dat[,c("DEM","TWI", "NDRE.M")],
   dat$VW,
   method="rf",
   trControl = ctrl)
global_validation(model_knndm)
}
}
\references{
\itemize{
\item Linnenbrink, J., Milà, C., Ludwig, M., and Meyer, H.: kNNDM: k-fold Nearest Neighbour Distance Matching Cross-Validation for map accuracy estimation, EGUsphere [preprint], https://doi.org/10.5194/egusphere-2023-1308, 2023.
\item Milà, C., Mateu, J., Pebesma, E., Meyer, H. (2022): Nearest Neighbour Distance Matching Leave-One-Out Cross-Validation for map validation. Methods in Ecology and Evolution 00, 1– 13.
}
}
\seealso{
\code{\link{geodist}}, \code{\link{nndm}}
}
\author{
Carles Milà and Jan Linnenbrink
}
