% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/set_priors.R
\name{set_priors}
\alias{set_priors}
\title{Set prior distribution}
\usage{
set_priors(
  model,
  priors = NULL,
  distribution = NA,
  alphas = NA,
  node = NA,
  label = NA,
  statement = NA,
  confound = NA,
  nodal_type = NA,
  param_names = NA,
  param_set = NA
)
}
\arguments{
\item{model}{A model created with \code{make_model}}

\item{priors}{A optional vector of positive reals indicating priors over all parameters. These are interpreted as arguments for Dirichlet distributions---one for each parameter set. To see the structure of parameter sets examine model$parameters_df}

\item{distribution}{String (or list of strings) indicating a common prior distribution (uniform, jeffreys or certainty)}

\item{alphas}{Real positive numbers giving hyperparameters of the Dirichlet distribution}

\item{node}{A string (or list of strings) indicating nodes for which priors are to be altered}

\item{label}{String. Label for nodal type indicating nodal types for which priors are to be altered}

\item{statement}{A causal query (or list of queries) that determines nodal types for which priors are to be altered}

\item{confound}{A confound statement (or list of statements) that restricts nodal types for which priors are to be altered}

\item{nodal_type}{String. Label for nodal type indicating nodal types for which priors are to be altered}

\item{param_names}{String. The name of specific parameter in the form of, for example, 'X.1', 'Y.01'}

\item{param_set}{String. Indicates the name of the set of parameters to be modified (useful when setting confounds)}
}
\value{
An object of class \code{causal_model}. It essentially returns a list containing the elements comprising
a model (e.g. 'statement', 'nodal_types' and 'DAG') with the `priors` attached to it.
}
\description{
A flexible function to add priors to a model.
}
\details{
Four arguments govern *which* parameters should be altered. The default is 'all' but this can be reduced by specifying

* \code{label} The label of a particular nodal type, written either in the form Y0000 or Y.Y0000

* \code{node}, which restricts for example to parameters associated with node 'X'

* \code{statement}, which restricts for example to nodal types that satisfy the statement 'Y[X=1] > Y[X=0]'

* \code{confound}, which restricts for example to nodal types that satisfy the statement 'Y[X=1] > Y[X=0]'

Two arguments govern what values to apply:

* \code{alphas} is one or more non negative numbers and

* \code{distribution} indicates one of a common class: uniform, jeffreys, or 'certain'

Any arguments entered as lists or vectors of size > 1 should be of the same length as each other.

For more examples and details see \code{make_priors}
}
\examples{

library(dplyr)
# Set priors to the model
model <- make_model('X -> Y') \%>\%
  set_priors(alphas = 3)
get_priors(model)
model <- make_model('X -> Y') \%>\%
  set_priors(distribution = 'jeffreys')
get_priors(model)

# Pass all nodal types
model <- make_model("Y <- X") \%>\%
  set_priors(.4)
get_priors(model)
model <- make_model("Y <- X") \%>\%
  set_priors(.7)
get_priors(model)
model <- make_model("Y <- X") \%>\%
  set_priors(distribution = "jeffreys")
get_priors(model)

# Passing by names of node, parameter set or label
model <- make_model('X -> M -> Y')
model_new_priors <- set_priors(model, param_name = "X.1", alphas = 2)
get_priors(model_new_priors)
model_new_priors <- set_priors(model, node = 'X', alphas = 3)
get_priors(model_new_priors)
model_new_priors <- set_priors(model, param_set = 'Y', alphas = 5)
get_priors(model_new_priors)
model_new_priors <- set_priors(model, node = c('X', 'Y'), alphas = 3)
get_priors(model_new_priors)
model_new_priors <- set_priors(model, param_set = c('X', 'Y'), alphas = 5)
get_priors(model_new_priors)
model_new_priors <- set_priors(model, node = list('X', 'Y'), alphas = list(3, 6))
get_priors(model_new_priors)
model_new_priors <- set_priors(model, param_set = list('X', 'Y'), alphas = list(4, 6))
get_priors(model_new_priors)
model_new_priors <- set_priors(model,
                               node = c('X', 'Y'),
                               distribution = c('certainty', 'jeffreys'))
get_priors(model_new_priors)
model_new_priors <- set_priors(model,
                               param_set = c('X', 'Y'),
                               distribution = c('jeffreys', 'certainty'))
get_priors(model_new_priors)
model_new_priors <- set_priors(model, label = '01', alphas = 5)
get_priors(model_new_priors)
model_new_priors <- set_priors(model, node = 'Y', label = '00', alphas = 2)
get_priors(model_new_priors)
model_new_priors <- set_priors(model, node =c('M', 'Y'), label = '11', alphas = 4)
get_priors(model_new_priors)

# Passing a causal statement
model_new_priors <- set_priors(model, statement = 'Y[M=1] > Y[M=0]', alphas = 3)
get_priors(model_new_priors)
model_new_priors <- set_priors(model,
                               statement = c('Y[M=1] > Y[M=0]', 'M[X=1]== M[X=0]'),
                               alphas = c(3, 2))
get_priors(model_new_priors)

# Passing a confound statement
model <- make_model('X->Y') \%>\%
 set_confound(list(X = 'Y[X=1] > Y[X=0]', X = 'Y[X=1] < Y[X=0]'))

model_new_priors <- set_priors(model,
            confound = list(X='Y[X=1] > Y[X=0]',
                            X='Y[X=1] < Y[X=0]'),
            alphas = c(3, 6))
get_priors(model_new_priors)
model_new_priors <- set_priors(model, confound= list(X='Y[X=1] > Y[X=0]'), alphas = 4)
get_priors(model_new_priors)
model_new_priors <- set_priors(model, param_set='X_1', alphas = 5)
get_priors(model_new_priors)
model_new_priors <-  set_priors(model, param_names='X_2.1', alphas = .75)
get_priors(model_new_priors)

# A more complex example
model <- make_model('X -> Y') \%>\%
  set_confound(list(X = 'Y[X=1]>Y[X=0]'))\%>\%
  set_priors(statement = 'X[]==1',
              confound = list(X = 'Y[X=1]>Y[X=0]', X = 'Y[X=1]<Y[X=0]'),
              alphas = c(2, .5))
get_priors(model)
}
\seealso{
Other priors: 
\code{\link{get_priors}()},
\code{\link{make_par_values_multiple}()},
\code{\link{make_par_values}()},
\code{\link{make_priors}()},
\code{\link{make_values_task_list}()}
}
\concept{priors}
