#' Check model function
#'
#' Generates a synthetic time series with the given properties and compares its sample statistics with the theoretically expected values.
#'
#' @inheritParams generateTS
#' @param plot logical - plot the result? (if TRUE, saved as an attribute 'plot')
#' @param returnTS logical - return timeseries generated by the check function? (if TRUE, saved as an attribute 'TS')
#'
#' @import moments reshape2
#' @export
#'
#' @keywords internal
#'
#' @examples
#'
#' library(CoSMoS)
#'
#' \donttest{
#' ## check your model
#' checkmodel(margdist = 'ggamma',
#'            margarg = list(shape1 = 3,
#'                           shape2 = .5,
#'                           scale = 10),
#'            n = 100000,
#'            p = 30,
#'            p0 = 0,
#'            acsvalue = acs(id = 'weibull',
#'                           t = 0:30,
#'                           shape = .5,
#'                           scale = 10))
#' }
#'
#' \dontshow{
#' checkmodel(margdist = 'ggamma',
#'            margarg = list(shape1 = 3,
#'                           shape2 = .5,
#'                           scale = 10),
#'            n = 1000,
#'            p = 30,
#'            p0 = 0,
#'            acsvalue = acs(id = 'weibull',
#'                           t = 0:30,
#'                           shape = .5,
#'                           scale = 10))
#' }
#'
checkmodel <- function(margdist, margarg, n, p = NULL, p0 = .6, acsvalue, TSn = 1, distbounds = c(-Inf, Inf), plot = FALSE, returnTS = FALSE) {

  suppressWarnings({x <- generateTS(margdist = margdist,
                                    margarg = margarg,
                                    n = n, p = p, p0 = p0, TSn = TSn,
                                    acsvalue = acsvalue)

  a <- sapply(x, function(y) acf(y, plot = F)$acf)

  out <- data.frame(mean = c(popmean(margdist,
                                     margarg,
                                     distbounds = distbounds,
                                     p0 = p0),
                             sapply(x, mean)),
                    sd = c(popsd(margdist,
                                 margarg,
                                 distbounds = distbounds,
                                 p0 = p0),
                           sapply(x, sd)),
                    skew = c(popskew(margdist,
                                     margarg,
                                     distbounds = distbounds,
                                     p0 = p0),
                             sapply(x, moments::skewness)),
                    p0 = c(p0,
                           sapply(x, function(y) length(which(y == 0))/length(y))),
                    acf_t1 = c(acsvalue[2],
                               a[2,]),
                    acf_t2 = c(acsvalue[3],
                               a[3,]),
                    acf_t3 = c(acsvalue[4],
                               a[4,]))

  row.names(out) <- c('expected', paste0('simulation', 1:TSn))})

  out <- round(out, 2)

  if(plot) {

    p <- checkplot(x = out,
                   margdist = margdist,
                   margarg = margarg,
                   p0 = p0)
  } else {

    p <- NA
  }

  if(!returnTS) {

    x <- NA
  }

  structure(.Data = out,
            plot = p,
            TS = x)
}

#' Check generated timeseries
#'
#' Compares generated time series sample statistics with the theoretically expected values.
#'
#' @param TS generated timeseries
#' @inheritParams checkmodel
#'
#' @export
#'
#' @examples
#'
#' library(CoSMoS)
#'
#' ## check your generated timeseries
#' x <- generateTS(margdist = 'burrXII',
#'                 margarg = list(scale = 1,
#'                                shape1 = .75,
#'                                shape2 = .25),
#'                 acsvalue = acs(id = 'weibull',
#'                                t = 0:30,
#'                                scale = 10,
#'                                shape = .75),
#'                 n = 1000, p = 30, p0 = .5, TSn = 5)
#'
#' checksimulation(x)
#'
checksimulation <- function(TS, distbounds = c(-Inf, Inf), plot = FALSE) {

  if (!is.list(TS)) {

    TS <- list(TS)
  }

  att <- attributes(TS[[1]])

  margdist <- att$margdist
  margarg <- att$margarg
  p0 <- att$p0
  acsvalue <- att$acsvalue

  ac <- sapply(TS, function(x) acf(x, plot = F)$acf)

  out <- data.frame(mean = c(popmean(margdist,
                                     margarg,
                                     distbounds = distbounds,
                                     p0 = p0),
                             sapply(TS, mean)),
                    sd = c(popsd(margdist,
                                 margarg,
                                 distbounds = distbounds,
                                 p0 = p0),
                           sapply(TS, sd)),
                    skew = c(popskew(margdist,
                                     margarg,
                                     distbounds = distbounds,
                                     p0 = p0),
                             sapply(TS, moments::skewness)),
                    p0 = c(p0,
                           sapply(TS, function(x) length(which(x == 0))/length(x))),
                    acf_t1 = c(acsvalue[2],
                               ac[2,]),
                    acf_t2 = c(acsvalue[3],
                               ac[3,]),
                    acf_t3 = c(acsvalue[4],
                               ac[4,]))

  row.names(out) <- c('expected', paste0('simulation', seq_along(TS)))

  out <- round(out, 2)

  if(plot) {

    p <- checkplot(x = out,
                   margdist = margdist,
                   margarg = margarg,
                   p0 = p0)
  } else {

    p <- NA
  }

  structure(.Data = out,
            plot = p)
}


#' Plot function for check results
#'
#' @param x check result
#'
#' @keywords internal
#' @export
#'
checkplot <- function(x, margdist, margarg, p0) {

  dta <- melt(as.matrix(x))
  dta.e <- dta[which(dta$Var1 == 'expected'),]
  dta.s <- dta[which(dta$Var1 != 'expected'),]

  p <- ggplot() +
    geom_boxplot(data = dta.s,
                 aes_string(x = 'Var2',
                            y = 'value',
                            group = 'Var2')) +
    geom_point(data = dta.e,
               aes_string(x = 'Var2',
                          y = 'value',
                          group = 'Var2'),
               size = 2,
               colour = 'red1') +
    facet_wrap('Var2',
               scales = 'free',
               nrow = 1) +
    labs(x = '',
         y = '',
         title = paste('Marginal =', margdist),
         subtitle = paste(paste(names(margarg),
                                '=',
                                margarg,
                                collapse = '; '),
                          paste('\np0 =',
                                p0),
                          collapse = ' ')) +
    theme_bw() +
    theme(axis.title.x = element_blank(),
          axis.text.x = element_blank(),
          axis.ticks.x = element_blank())

  return(p)
}
