% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/estimationCondCopulas.R
\name{estimateParCondCopula}
\alias{estimateParCondCopula}
\alias{estimateParCondCopula_ZIJ}
\title{Estimation of parametric conditional copulas}
\usage{
estimateParCondCopula(
  observedX1,
  observedX2,
  observedX3,
  newX3,
  family,
  method = "mle",
  h
)

estimateParCondCopula_ZIJ(Z1_J, Z2_J, observedX3, newX3, family, method, h)
}
\arguments{
\item{observedX1}{a vector of \code{n} observations
of the first conditioned variable}

\item{observedX2}{a vector of \code{n} observations
of the second conditioned variable}

\item{observedX3}{a vector of \code{n} observations
of the conditioning variable}

\item{newX3}{a vector of new observations of \eqn{X3}}

\item{family}{an integer indicating the parametric family of copulas to be used,
following the conventions of the package \code{\link{VineCopula}}.}

\item{method}{the method of estimation of the conditional parameters.
Can be \code{"mle"} for maximum likelihood estimation
or \code{"itau"} for estimation by inversion of Kendall's tau.}

\item{h}{bandwidth to be chosen}

\item{Z1_J}{the conditional pseudos-observations of the first variable,
i.e. \eqn{\hat F_{1|J}( x_{i,1} | x_J = x_{i,J})} for \eqn{i=1,\dots, n}.}

\item{Z2_J}{the conditional pseudos-observations of the second variable,
i.e. \eqn{\hat F_{2|J}( x_{i,2} | x_J = x_{i,J})} for \eqn{i=1,\dots, n}.}
}
\value{
a vector of size \code{length(newX3)} containing
the estimated conditional copula parameters for each value of \code{newX3}.
}
\description{
The function \code{estimateParCondCopula} computes an estimate
of the conditional parameters in a conditional parametric copula model, i.e.
\deqn{C_{X_1, X_2 | X_3 = x_3} = C_{\theta(x_3)},}
for some parametric family \eqn{(C_\theta)}, some conditional
parameter \eqn{\theta(x_3)}, and a three-dimensional random
vector \eqn{(X_1, X_2, X_3)}. Remember that \eqn{C_{X_1,X_2 | X_3 = x_3}}
denotes the conditional copula of \eqn{X_1} and \eqn{X_2} given \eqn{X_3 = x_3}.

The function \code{estimateParCondCopula_ZIJ} is an auxiliary function
that is called when conditional pseudos-observations are already
available when one wants to estimate a parametric conditional copula.
}
\examples{

# We simulate from a conditional copula
N = 500

X3 = rnorm(n = N, mean = 5, sd = 2)
conditionalTau = 0.9 * pnorm(X3, mean = 5, sd = 2)
simCopula = VineCopula::BiCopSim(
    N=N , family = 1, par = VineCopula::BiCopTau2Par(1 , conditionalTau ))
X1 = qnorm(simCopula[,1])
X2 = qnorm(simCopula[,2])

gridnewX3 = seq(2, 8, by = 1)
conditionalTauNewX3 = 0.9 * pnorm(gridnewX3, mean = 5, sd = 2)

vecEstimatedThetas = estimateParCondCopula(
  observedX1 = X1, observedX2 = X2, observedX3 = X3,
  newX3 = gridnewX3, family = 1, h = 0.1)

# Estimated conditional parameters
vecEstimatedThetas
# True conditional parameters
VineCopula::BiCopTau2Par(1 , conditionalTauNewX3 )

# Estimated conditional Kendall's tau
VineCopula::BiCopPar2Tau(1 , vecEstimatedThetas )
# True conditional Kendall's tau
conditionalTauNewX3


}
\references{
Derumigny, A., & Fermanian, J. D. (2017).
About tests of the “simplifying” assumption for conditional copulas.
Dependence Modeling, 5(1), 154-197.
\doi{10.1515/demo-2017-0011}
}
\seealso{
\code{\link{estimateNPCondCopula}} for estimating a conditional
copula in a nonparametric setting ( = without parametric assumption on the
conditional copula).
\code{\link{simpA.param}} for a test that this conditional copula is
constant with respect to the value \eqn{x_3} of the conditioning variable.
}
