% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/lyapunov.R
\name{lyapunov}
\alias{lyapunov}
\title{Estimates the Lyapunov exponent through several methods}
\usage{
lyapunov(
  data,
  m = 1:4,
  lag = 1:1,
  timelapse = c("FIXED", "VARIABLE"),
  h = 2:10,
  w0maxit = 100,
  wtsmaxit = 1e+06,
  pre.white = TRUE,
  lyapmethod = c("SLE", "LLE", "ALL"),
  blocking = c("BOOT", "NOVER", "EQS", "FULL", "ALL"),
  B = 1000,
  trace = 1,
  seed.t = TRUE,
  seed = 56666459,
  doplot = TRUE,
  ...
)
}
\arguments{
\item{data}{a \code{vector}, a time-series object \code{ts} or \code{xts}, a \code{data.frame}, a \code{data.table} or a \code{matrix} depending on the method selected in \code{timelapse}.}

\item{m}{a non-negative integer denoting a lower and upper bound for the embedding dimension (Default 1:4).}

\item{lag}{a non-negative integer denoting a lower and upper bound for the the reconstruction delay (Default 1:1).}

\item{timelapse}{a character denoting if the time-series data are sampled at uniform time-frequency e.g., 1-month, 1-day, 1-hour, 30-min, 5-min, 1-min and so on \code{FIXED} or non-uniform time-frequency which are not equally spaced in time \code{VARIABLE} (Default \code{FIXED}).}

\item{h}{a non-negative integer denoting a lower and upper bound for the number of neurones (or nodes) in the single hidden layer (Default 2:10).}

\item{w0maxit}{a non-negative integer denoting the maximum iterations to estimate the initial parameter vector of the neural net models (Default 100).}

\item{wtsmaxit}{a non-negative integer denoting the maximum iterations to estimate the weights parameter vector of the neural net models (Default 1e6).}

\item{pre.white}{a logical value denoting if the user wants to use as points to evaluate the partial derivatives the delayed vectors filtered by the neural net model chosen \code{TRUE} or not \code{FALSE} (Default \code{TRUE}).}

\item{lyapmethod}{a character denoting the procedure chosen to estimate the Lyapunov exponent. If \code{LLE} has been selected the function will estimate only the largest Lyapunov exponent through the Norma-2 method. If \code{SLE} has been selected the function will estimate the Lyapunov exponent spectrum through the QR decomposition. Otherwise \code{ALL} has to be specified. In this case the function will estimate the Lyapunov exponent considering both procedures (Default \code{SLE}).}

\item{blocking}{a character denoting the blocking method chosen for figuring out the Lyapunov exponent. Available options are \code{FULL} if the user considers the full sample, \code{NOVER} if the user considers the non-overlapping sample, \code{EQS} if the user considers the equally spaced sample, \code{BOOT} if the user considers the bootstrap sample or \code{ALL} if the user considers all of them (Default \code{BOOT}).}

\item{B}{a non-negative integer denoting the number of bootstrap iterations (Default 1000).}

\item{trace}{a binary value denoting if the user wants to print the output on the console \code{1} or not \code{0} (Default 1).}

\item{seed.t}{a logical value denoting if the user wants to fix the seed \code{TRUE} or not \code{FALSE} (Default TRUE).}

\item{seed}{a non-negative integer denoting the value of the seed selected if \code{seed.t = TRUE} (Default 56666459).}

\item{doplot}{a logical value denoting if the user wants to draw a plot \code{TRUE} or not \code{FALSE}. If it is \code{TRUE} the evolution of the Lyapunov exponent values are represented for the whole period considering the blocking method chosen by the user. It shows as many graphs as embedding dimensions have been considered (Default \code{TRUE}).}

\item{...}{further arguments passed to or from \code{nnet} function.}
}
\value{
This function returns several objects considering the parameter set selected by the user. The largest Lyapunov exponent (Norma-2 procedure) and the Lyapunov exponent spectrum (QR decomposition procedure) by each blocking method are estimated. It also contains some useful information about the estimated jacobian, the best-fitted feed-forward single hidden layer neural net model, the best set of weights found, the fitted values, the residuals obtained, the best embedding parameters set chosen, the sample size or the block length considered by each blocking method. This function provides the standard error, the z test value and the p-value for testing the null hypothesis \eqn{H0: \lambda_k > 0 for k = 1,2,3, \ldots, m}. Reject the null hypothesis ${H_0}$ means lack of chaotic behaviour. That is, the data-generating process does not have a chaotic attractor because of it does not show the property of sensitivity to initial conditions.
}
\description{
This is an all-in-one function. It provides, at the same time, the delayed-coordinate embedding vector (\code{embedding}), estimates the best neural net model (\code{netfit}), calculates the partial derivatives directly from the chosen neural network model (\code{jacobian.net}). Finally, this function estimates both the largest Lyapunov exponent through the Norma-2 procedure (\code{lyapunov.max}) and the Lyapunov exponent spectrum through the QR decomposition procedure (\code{lyapunov.spec}) taking into account the full sample and three different methods of subsampling by blocks.
}
\note{
We have considered it appropriate to incorporate a function that unifies the whole process to make it easier and more intuitive for the R users. The DChaos package provides several ways to figure out robustly the neural net estimator of the k-th Lyapunov exponent. Particularly, there are 8 functions (one for each procedure and blocking method) which estimate the Lyapunov exponents consistently. Hence the DChaos package allows the R users to choose between two different procedures to obtain the neural net estimator of the k-th Lyapunov exponent and four ways of subsampling by blocks: full sample, non-overlapping sample, equally spaced sample and bootstrap sample. The blocking methods what they do is to split the time-series data into several blocks by estimating a Lyapunov exponent for each subsample. If the R users choose the non-overlapping sample (\code{blocking = "NOVER"}), the equally spaced sample (\code{blocking = "EQS"}) or the bootstrap sample (\code{blocking = "BOOT"}) the mean and median values of the Lyapunov exponent for each block or subsample are saved. By default we recommend using the median value as it is more robust to the presence of outliers. Notice that the parameter \code{B} will only be considered if the R users choose the bootstrap blocking method.
}
\examples{
## set.seed(34)
## Simulates time-series data from the Logistic map with chaos
## ts        <- DChaos::logistic.sim(n=1000, a=4)
## show(head(ts, 5))

## Provides the Lyapunov exponent spectrum by the QR decomposition procedure considering the
## bootstrap blocking method directly from the Logistic map with chaos simulated.
## exponent <- DChaos::lyapunov(ts, m=3:3, lag=1:1, timelapse="FIXED", h=2:10, w0maxit=100,
##                     wtsmaxit=1e6, pre.white=TRUE, lyapmethod="SLE", blocking="ALL",
##                     B=100, trace=1, seed.t=TRUE, seed=56666459, doplot=FALSE))
## summmary(exponent)
}
\references{
Ellner, S., Gallant, A., McCaffrey, D., Nychka, D. 1991 Convergence rates and data requirements for jacobian-based estimates of lyapunov exponents from data. Physics Letters A 153(6):357-363.

McCaffrey, D.F., Ellner, S., Gallant, A.R., Nychka, D.W. 1992 Estimating the lyapunov exponent of a chaotic system with nonparametric regression. Journal of the American Statistical Association 87(419):682-695.

Nychka, D., Ellner, S., Gallant, A.R., McCaffrey, D. 1992 Finding chaos in noisy systems. Journal of the Royal Statistical Society 54(2):399-426.

Whang, Y.J., Linton, O. 1999 The asymptotic distribution of nonparametric estimates of the lyapunov exponent for stochastic time series. Journal of Econometrics 91(1):1-42.

Shintani, M., Linton, O. 2004 Nonparametric neural network estimation of Lyapunov exponents and a direct test for chaos. Journal of Econometrics 120(1):1-33.
}
\seealso{
\code{\link{lyapunov.max}}, \code{\link{lyapunov.spec}}
}
\author{
Julio E. Sandubete, Lorenzo Escot
}
