% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/mutate_edge_attrs.R
\name{mutate_edge_attrs}
\alias{mutate_edge_attrs}
\title{Mutate a set of edge attribute values}
\usage{
mutate_edge_attrs(graph, edge_attr_from, expressions, edge_attr_to = NULL)
}
\arguments{
\item{graph}{a graph object of class
\code{dgr_graph}.}

\item{edge_attr_from}{the name of the edge attribute
column from which values will be mutated. An
\code{NA} value can be provided here if edge
attribute names are used in \code{expressions}
statements. Note that if \code{NA} is used, a value
must be provided for \code{edge_attr_to}.}

\item{expressions}{one or more expressions for
mutation of all edge attribute values specified by
\code{edge_attr_from}. To reference the edge
attribute given in \code{edge_attr_from}, use the
\code{~} character. Otherwise, all edge attributes
can be referenced by using the names of those edge
attributes directly in the expressions. Expressions
are evaluated in the order provided.}

\item{edge_attr_to}{an optionally supplied name of
a new edge attribute to which the mutated values
will be applied. This will retain the original edge
attribute and its values. If \code{NA} is used
with \code{edge_attr_from}, a value must be provided
here (since mutated values must be placed
somewhere).}
}
\value{
a graph object of class \code{dgr_graph}.
}
\description{
Within a graph's internal edge data
frame (edf), mutate numeric edge attribute values
using one or more expressions. Optionally, one can
specify a different edge attribute name and create
a new edge attribute while retaining the original
edge attribute and its values.
}
\examples{
# Create a graph with 3 edges
graph <-
  create_graph() \%>\%
  add_path(4) \%>\%
  set_edge_attrs(
    "width", c(3.4, 2.3, 7.2))

# Get the graph's internal edf to show which
# edge attributes are available
get_edge_df(graph)
#>   id from to  rel width
#> 1  1    1  2 <NA>   3.4
#> 2  2    2  3 <NA>   2.3
#> 3  3    3  4 <NA>   7.2

# Mutate the `width` edge attribute, dividing
# each value by 2
graph <-
  graph \%>\%
  mutate_edge_attrs("width", "~ / 2")

# Get the graph's internal edf to show that the
# edge attribute `width` had its values changed
get_edge_df(graph)
#>   id from to  rel width
#> 1  1    1  2 <NA>  1.70
#> 2  2    2  3 <NA>  1.15
#> 3  3    3  4 <NA>  3.60

# Create a new edge attribute, called `length`,
# that is the log of values in `width` plus 2
# (and round all values to 2 decimal places)
graph <-
  graph \%>\%
  mutate_edge_attrs(
    "width", "round(log(~) + 2, 2)", "length")

# Get the graph's internal edf to show that
# the edge attribute values had been mutated
# and used as the new edge attribute `length`
get_edge_df(graph)
#>   id from to  rel width length
#> 1  1    1  2 <NA>  1.70   2.53
#> 2  2    2  3 <NA>  1.15   2.14
#> 3  3    3  4 <NA>  3.60   3.28

# Create a new edge attribute called `area`,
# which is the product of the `width` and
# `length` attributes; note that we can provide
# NA to `edge_attr_from` since we are naming
# at least one of the edge attributes in the
# `expressions` vector (and providing a new
# edge attribute name: `area`)
graph <-
  graph \%>\%
  mutate_edge_attrs(
    NA, "width * length", "area")

# Get the graph's internal edf to show that
# the edge attribute values had been multiplied
# together, creating a new edge attribute `area`
get_edge_df(graph)
#>   id from to  rel width length   area
#> 1  1    1  2 <NA>  1.70   2.53  4.301
#> 2  2    2  3 <NA>  1.15   2.14  2.461
#> 3  3    3  4 <NA>  3.60   3.28 11.808
}

