#' Classify points as above or below the treeline
#'
#' @description Calculates if the points (from the input data frame \code{coords}) are above the treeline (\code{TRUE}) or not (\code{FALSE}).
#' This is achieved by using climate layers for growing season length and growing season temperature. For each coordinate a value from both
#' rasters is extracted and added to the input data frame. Then points are classified, the default thresholds and the treeline definition
#' is based on Paulsen and Körner, Alp. Bot. 124: 1-12 (2014). Classification (as boolean) is also added to the output.
#' @usage classify_above_treeline(coords, gstRaster, gslRaster, gstTreshold = 6.4,
#'                               gslTreshold = 94)
#' @param coords Data frame representing coordinates (WGS 84) to be classified.
#' The first column must contain the longitude, the second the latitude, both in decimal degrees.
#' The values must be of the data type "numeric" and finite.
#' A data frame can be generated by using the function \code{generate_grid}.
#' @param gstRaster Climatic raster that contains the growing season temperature. Data type "SpatRaster".
#' @param gslRaster Climatic raster that contains the growing season length. Data type "SpatRaster".
#' @param gstTreshold Growing season temperature threshold for tree growth (in degree Celsius). One value, data type "numeric" and finite.
#' @param gslTreshold Growing season length threshold for tree growth (days). One value, data type "integer" and finite.
#' @return A data frame containing: longitude, latitude,
#' growing season temperature, growing season length, and a boolean.
#' The boolean indicates if the point is above the treeline.
#' @author Livio Bätscher, Jurriaan M. de Vos
#' @examples
#' #Get raster layer from CHELSA
#' gstURL <- paste0("https://os.zhdk.cloud.switch.ch/envicloud/chelsa/chelsa_V2/",
#'                  "GLOBAL/climatologies/1981-2010/bio/CHELSA_gst_1981-2010_V.2.1.tif")
#' gslURL <- paste0("https://os.zhdk.cloud.switch.ch/envicloud/chelsa/chelsa_V2/",
#'                  "GLOBAL/climatologies/1981-2010/bio/CHELSA_gsl_1981-2010_V.2.1.tif")
#' \donttest{
#' gst <- terra::rast(gstURL, vsi = TRUE)
#' gsl <- terra::rast(gslURL, vsi = TRUE)
#'
#' #Classify a single point
#' point <- data.frame("lon" = 8.65, "lat" = 46.87)
#' classify_above_treeline(coords = point, gstRaster = gst, gslRaster = gsl,
#'                         gstTreshold = 6.4, gslTreshold = 94)
#'
#' #Classify a dummy data frame
#' longitude <- rep(8.53, 11)
#' latitude <- seq(46.8, 46.9, 0.01)
#' temp <- data.frame(longitude, latitude)
#' classify_above_treeline(coords = temp, gstRaster = gst, gslRaster = gsl,
#'                         gstTreshold = 6.4, gslTreshold = 94)
#' }
#' @export

classify_above_treeline <- function(coords, gstRaster, gslRaster, gstTreshold = 6.4, gslTreshold = 94) {
  #Error handling
  if (!is.data.frame(coords)) {stop("coords must be a data frame")} else if (ncol(coords) < 2) {stop("coords needs to have at least two columns")} else if (sum(!is.finite(as.matrix(coords))) != 0) {stop("coords must be numeric and finite")}
  if (class(gstRaster)[1] != "SpatRaster") {stop("gstRaster must be from class SpatRaster")}
  if (class(gslRaster)[1] != "SpatRaster") {stop("gslRaster must be from class SpatRaster")}
  if (length(gstTreshold) != 1) {stop("gstTreshold must be of length 1")} else if (!is.finite(gstTreshold)) {stop("gstTreshold must be numeric and finite")}
  if (length(gslTreshold) != 1) {stop("gslTreshold must be of length 1")} else if (!is.finite(gslTreshold) || gslTreshold %% 1 != 0) {stop("gslTreshold must be a integer and finite")}

  #Now get the raster value that matches the coordinate from the input data frame
  coords <- cbind(coords, "growingSeasonTemperature" = terra::extract(gstRaster, coords[, 1:2])[,2])
  coords <- cbind(coords, "growingSeasonLength" = terra::extract(gslRaster, coords[, 1:2])[,2])

  #Decide if a point is above or below the treeline
  aboveTreeline <- rep(FALSE, nrow(coords))

  #Definition according to Paulsen and Körner 2014
  aboveTreeline[coords$growingSeasonTemperature < gstTreshold | coords$growingSeasonLength < gslTreshold] <- TRUE
  aboveTreeline[is.na(coords$growingSeasonTemperature)] <- NA

  coords <- cbind(coords, aboveTreeline) #Add the vector to the data frame

  return(coords) #Return the new data frame
}
