##----------------------------------------------------------------------------##
##----------------------------------------------------------------------------##
##-----------------------Infectious Disease Forecasting-----------------------##
##----------------------------------------------------------------------------##
##----------------------------------------------------------------------------##
#' @include DataContainers.R

##----------------------------------------------------------------------------##
##-----------------------------------Forecasts--------------------------------##
##----------------------------------------------------------------------------##

####################################Forecast####################################
#' @title Forecast
#' @description An abstract class for storing the results of forecasting.  These
#'   classes do not contain any data directly, but instead contain a data object.
#'   Extend this class when you want to store the results of a model, and none
#'   of the convenience classes are applicable.
#' @docType class
#' @importFrom R6 R6Class
#' @export Forecast
#' @keywords forecast core
#' @family Forecast
#' @example Forecast.R

Forecast <- R6Class(
  classname = "Forecast",
  inherit = Generic,
  private = list(
    .forecastMadeTime = NA,
    .forecastTimes = NA,
    .model = Generic$new(),
    .data = MatrixData$new()
  ),
  public = list(
    #' @method mean This \bold{must} be extended.  This method extracts the elementwise mean of the forecast.  This function will not change the number of rows or columns in the data, but will convert probabilistic estimates into deterministic ones.
    #' @return a MatrixData.
    mean = function(){
      private$defaultAbstract()
    },
    #' @method median This \bold{must} be extended.  This method extracts the elementwise median of the forecast.  This function will not change the number of rows or columns in the data, but will convert probabilistic estimates into deterministic ones.
    #' @return a MatrixData.
    median = function(){
      private$defaultAbstract()
    },
    #' @method quantile This \bold{must} be extended.  Get the cutoffs for each percentile in alphas.
    #' @param alphas A numeric vector with elements between \code{0} and \code{1} of percentiles to find cutoffs for.
    #' @return an ArrayData.
    quantile = function(alphas){
      private$defaultAbstract()
    },
    #' @method binDist This \bold{must} be extended.  Get the distribution of simulations of the data within fixed bins.
    #' @param cutoffs A numeric vector with elements to use as the dividing values for the bins.
    #' @return a FrameData.
    binDist = function(cutoffs){
      private$defaultAbstract()
    }
  ),
  active = list(
    #' @field forecastMadeTime When the forecast was created.
    forecastMadeTime = function(value){
      if(missing(value)){
        return(private$.forecastMadeTime)
      }
      stop("Do not write directly to the creation time.")
    },
    #' @field forecastTimes The times the forecast is about.
    forecastTimes = function(value){
      if(missing(value)){
        return(private$.forecastTimes)
      }
      stop("If you want different times, do another forecast instead.")
    },
    #' @field model The model used to create the forecast.
    model = function(value){
      if(missing(value)){
        return(private$.model)
      }
      stop("If you want a different model, do another forecast instead.")
    },
    #' @field data  The data used to create the forecast.
    data = function(value){
      if(missing(value)){
        return(private$.data)
      }
      stop("If you want different data, do another forecast instead.")
    }
  )
)

###############################Simulated Forecast###############################
#' @title SimulatedForecast
#' @description This class is a forecast where the data is many simulated trials.
#' @docType class
#' @importFrom R6 R6Class
#' @export SimulatedForecast
#' @keywords forecast
#' @family Forecast
#' @example SimulatedForecast.R
SimulatedForecast <- R6Class(
  classname = "SimulatedForecast",
  inherit = Forecast,
  private = list(
    .nsim = 0,
    .data = AbstractSimulatedIncidenceMatrix$new()
  ),
  public = list(
    #' @method mean This method extracts the elementwise mean of the forecast.  This function will not change the number of rows or columns in the data, but will convert probabilistic estimates into deterministic ones.
    mean = function(){
      self$data$summarize(mean)
    },
    #' @method median This method extracts the elementwise median of the forecast.  This function will not change the number of rows or columns in the data, but will convert probabilistic estimates into deterministic ones.
    #' @return a MatrixData.
    median = function(){
      self$data$summarize(median)
    },
    #' @method quantile Get the cutoffs for each percentile in alphas.
    #' @param alphas A numeric vector with elements between \code{0} and \code{1} of percentiles to find cutoffs for.
    #' @return an ArrayData.
    quantile = function(alphas){
      SimulatedIncidenceMatrix$new(
        lapply(
          alphas,
          function(alpha){
            self$data$summarize(function(x){quantile(x , alpha)})
          }
        )
      )
    },
    #' @method binDist Get the distribution of simulations of the data within fixed bins.
    #' @param cutoffs A numeric vector with elements to use as the dividing values for the bins.
    #' @return a FrameData.
    binDist = function(cutoffs){
      SimulatedIncidenceMatrix$new(
        lapply(
          cutoffs,
          function(cutoff){
            self$data$summarize(function(x){ecdf(x)(cutoff)})
          }
        )
      )
    }
  ),
  active = list(
    #' @field nsim The number of simulations.
    nsim = function(value){
      if('nsim' %in% private$.debug){
        browser()
      }
      if(missing(value)){
        return(private$.nsim)
      }
      stop("Do not modify the number of simulations directly")
    },
    #' @field sample Draw a random sample from the possible model predictions.  Please see implementation of the data for the properties of the sampling.
    sample = function (value){
      if(!missing(value)){
        stop("Do not try to write to a random sample")
      }
      return(self$data$sample)
    }
  )
)
