% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/epoch.R
\name{epoch}
\alias{epoch}
\alias{apply.epoch}
\alias{mean.epoch}
\alias{sd.epoch}
\alias{median.epoch}
\alias{mad.epoch}
\alias{autocor.epoch}
\alias{quantile.epoch}
\title{Compute epochal summary statistics.}
\usage{
apply.epoch(x, epoch.size = 10, incl.date = FALSE, FUN)
}
\arguments{
\item{x}{The object to compute statistics for. Can be an "AccData" object, a matrix, or a vector.}

\item{epoch.size}{Numeric giving intervals to consider and aggregate. For "AccData" x taken as seconds. Otherwise, considered as rows, or as individual readings.}

\item{incl.date}{logical. If TRUE, include a column of times or original indices with the results.}

\item{FUN}{A function to be applied to each epoch.}
}
\value{
A vector or array giving the computed epochal summaries. With incl.date = TRUE,
the result is given as a data.frame suitable for plotting.
}
\description{
Computes epochal summary statistics for an "AccData" object, matrix, or vector, and collates into a matrix or vector.
}
\details{
These functions compute epochal summary statistics for "AccData" objects, matrices and vectors.

apply.epoch is the general function - according to the size of epoch.size, it splits up the x into collections
of consecutive rows, each with the same size. These are then successively supplied to FUN as its first argument.
If the result of FUN is a single value, then the results are concatenated into a vector output.
Otherwise, an array is formed with each row corresponding to a single epochal group. For AccData, the sampling
frequency of the dataset is used to interpret the epoch size in seconds. Otherwise, the raw record indices are used.
If incl.date is set, the original timestamp vector of the data, or the original indices, are downsampled and
included as the first column of the output.

The remaining functions are wrappers that compute various commonly useful statistics -- in particular, applied to
"AccData" objects and arrays, they by default compute the epochal SVM mean, standard deviation, median, median
absolute deviation, and autocorrelation, and sample quantiles respectively. (Arrays are treated as each column
representing the x, y, and z components respectively.) Applied to vector input, processing will occur without the
SVM calculation. This behaviour may be overridden by the sqrt setting, which will force the function to use the
squared (default for arrays and "AccData") or original unit (default for vectors) values in the statistical analysis.
}
\examples{
\dontrun{
dat <- read.bin(system.file("binfile/TESTfile.bin", package = "GENEAread")[1]
, calibrate = TRUE)

#look for the epochs that exceed a certain threshold 50\% of the time
plot(apply.epoch( dat, epoch.size = 3 ,
                  FUN = function(t) mean(abs(svm(t) -1)>0.2)> 0.5 ), type = "l")

plot(dat[,1], svm(dat), log = "y", pch = ".")
lines(mean.epoch(dat, incl.date = TRUE), lwd = 2)
lines(mean.epoch(dat, epoch.size = 30, incl.date = TRUE), col = 2, lwd = 2)
# This should give all the same results, but by a different way
lines(apply.epoch(dat, epoch.size = 30,
                  FUN = function(A) mean(svm(A, FALSE)), incl.date = TRUE), col = 3)
epsize = 30
lines(apply.epoch(dat, epoch.size = epsize,
                  FUN = function(t) median(t[,1])),
                  apply.epoch(dat, epoch.size = epsize,
                  FUN = function(A) mean(svm(A, FALSE))), col = 4)
#note this is different
lines(apply.epoch(dat, epoch.size = epsize,
                  FUN = function(t) median(t[,1])),
                  apply.epoch(dat, epoch.size = epsize,
                              FUN = function(A) mean(svm(A, sqrt = TRUE)))^2,
                              col = 5)

#plot some statistics
par(mfrow = c(5,1), mar = c(1,4.5,1,1))
plot(sd.epoch(dat), type="l")
plot(median.epoch(dat), type= "l")
plot(mad.epoch(dat), type= "l")
plot(autocor.epoch(dat), type= "l")
tmp = quantile.epoch(dat, quantiles= c(0.1, 0.25, 0.5, 0.75, 0.9)); matplot(tmp, type = "l")
}
}
