
#' Number of contributors suppression rule
#' 
#' The number of contributors is the number unique contributing 'charVar' codes. 
#' 
#'
#' @param data  Input data as a data frame
#' @param freq  Vector of aggregate frequencies 
#' @param numVar Numerical variables. When several variables, only first is used. 
#' @param x Model matrix generated by parent function
#' @param maxN Primary suppression when number of contributors `<= maxN`.
#' @param protectZeros Suppression parameter. Only TRUE (default) is used implemented. 
#' @param charVar Variable with contributor codes. 
#'                When empty, unique contributor in each row is assumed.
#' @param removeCodes Vector of codes to be omitted when counting contributors.
#'                With empty `charVar` row indices are assumed
#'                and conversion to integer is performed.
#' @param remove0 When `FALSE` (default), data rows where `numVar` (if any) is zero are omitted when counting contributors.
#' @param ... unused parameters
#'
#' @return List where first element is logical vector defining primary suppressions.
#'         The second element is data frame where `nRule` is number contributors used 
#'         in rule and where `nAll` is similar, but without omitting codes in `removeCodes`. 
#' @export
#'
NContributorsRule <- function(data, freq, numVar, x, 
                              maxN = 3, 
                              protectZeros = FALSE, 
                              charVar = NULL, 
                              removeCodes = character(0), 
                              remove0 = TRUE, 
                              ...) {
  if (length(charVar)>1) {
    stop("Only single charVar implemented in suppression rule")
  }
  if (length(numVar) > 1){
    warning("Multiple numVar were supplied, only the first is used in suppression rule.")
    numVar <- numVar[1]
  }
  if (protectZeros) {
    stop("TRUE protectZeros not implemented")
  }
  if (length(charVar)) {
    y <- data[[charVar]]
  } else {
    y <- seq_len(nrow(data))
    removeCodes <- as.integer(removeCodes)
  }
  if (remove0 & length(numVar) > 0) {
    y[data[[numVar]] == 0] <- NA
  }
  
  nAll <- Ncontributors(x, y)
  y[y %in% removeCodes] <- NA
  nRule <- Ncontributors(x, y)
  primary <- (nRule <= maxN) & (nRule > 0)
  list(primary = primary, numExtra = data.frame(nRule = nRule, nAll = nAll))
}


NContributorsRule_identical <- NContributorsRule


#' Identical to ´NContributorsRule´
#' 
#' The function is included for compatibility after changing the name to 
#' \code{\link{NContributorsRule}}
#' 
#' @rdname NcontributorsRule_identical
#' 
#' @inheritParams NContributorsRule
#' 
#' @export
#' @keywords internal
#'
NcontributorsRule <- NContributorsRule_identical


# Without @rdname NcontributorsRule_identical:
# Default NcontributorsRule overwrites NContributorsRule.Rd  

# With 
# NcontributorsRule <- NContributorsRule 
# it is impossible to hide  function in separate rd-file with keywords internal

# With 
# NcontributorsRule <- function(...) NContributorsRule(...)   
# test fails since default values cannot be found in function

