#' Generate an empty chat history list for a Generative AI model
#'
#' This function establishes a connection to a Generative AI model by providing essential
#' parameters and generates an empty chat history list for the model.
#'
#' @param model.parameter A character vector containing the Generative AI service provider,
#' corresponding model, version, API key, and proxy status.
#'
#' @return If successful, a chat history list is generated for the Generative AI model.
#'
#' @details Providing accurate and valid information for each parameter is crucial
#' to ensure successful text generation by the Generative AI model. The chat history will be
#' modified by this function. If any of the provided parameters is incorrect, the function
#' will respond with an error message based on the information received from the API.
#' Use the function \code{\link{available.models}} to see all supported Generative AI models.
#'
#' @seealso
#' \href{https://genai.gd.edu.kg/r/documentation/}{GenAI - R Documentation}
#'
#' @examples
#' \dontrun{
#'  # Get available models
#'  models = available.models()
#'
#'  # Connect to the model, replace API_KEY with your api key
#'  google.model = connect.genai("google",
#'                               models$google$model[1],
#'                               models$google$version[1],
#'                               "API_KEY",
#'                               FALSE)
#'
#'  # Connect to the model, replace API_KEY with your api key
#'  openai.model = connect.genai("openai",
#'                               models$openai$model[1],
#'                               models$openai$version[1],
#'                               "API_KEY",
#'                               FALSE)
#'  # Setup an empty chat history
#'  google.history = chat.setup(google.model)
#'  openai.history = chat.setup(openai.model)
#'
#'  # Start chat
#'  temperature = 0.9
#'  prompt = "Write a story about a magic backpack in about 100 words."
#'  google.history = chat(google.model,
#'                        temperature,
#'                        google.history,
#'                        prompt)
#'
#'  # Get the most recent message generated by the model
#'  cat(chat.recent(google.model, google.history))
#'
#'  openai.history = chat(openai.model,
#'                        temperature,
#'                        openai.history,
#'                        prompt)
#'
#'  # Get the most recent message generated by the model
#'  cat(chat.recent(openai.model, openai.history))
#'
#'  prompt = "What is the word count of the story you just wrote?"
#'  google.history = chat(google.model,
#'                        temperature,
#'                        google.history,
#'                        prompt)
#'
#'  # Get the most recent message generated by the model
#'  cat(chat.recent(google.model, google.history))
#'
#'  openai.history = chat(openai.model,
#'                        temperature,
#'                        openai.history,
#'                        prompt)
#'
#'  # Get the most recent message generated by the model
#'  cat(chat.recent(openai.model, openai.history))
#' }
#'
#' @export
chat.setup = function(model.parameter) {
  switch (model.parameter["provider"],
          google = {
            requestBody = list(contents = list())
            return (requestBody)
          },
          openai = {
            requestBody = list(model = model.parameter["model"],
                               messages = list(list(role = "system",
                                                    content = "You are a helpful assistant.")))
            return (requestBody)
          })
}
