% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/GermaParl.R
\docType{package}
\name{GermaParl-package}
\alias{GermaParl-package}
\alias{_PACKAGE}
\alias{GermaParl}
\title{GermaParl R Data Package.}
\description{
\emph{GermaParl} is  a corpus of parliamentary debates in the German
Bundestag. The package offers a convenient dissemination mechanism for the
\emph{GermaParl} corpus.  The corpus has been linguistically annotated and
indexed using the data format of the \emph{Corpus Workbench} (CWB). To make
full use if this data format, working with \emph{GermaParl} in combination
with the \emph{polmineR} package is recommended. After installation, only a
small sample corpus will be included in the package ("GERMAPARLMINI"). To
download the full corpus from the open science repository \emph{Zenodo}, use
the \code{germaparl_download_corpus} function. The package offers further
functionality to augment the corpus (see the the examples section below).
}
\details{
The \emph{GermaParl} R package and the \emph{GermaParl} corpus are two
different pieces of research data: The package offers a mechanism to ship,
easily install and augment the data. The indexed corpus is the actual data.
Package and corpus have different version numbers and should be quoted in
combination in publications. We recommend to follow the instructions you see
when calling \code{citation(package = "GermaParl")}. To ensure that the
recommended citation fits the corpus you use, the citation for the corpus is
available only when a version of \emph{GermaParl} has been downloaded and
installed.
}
\examples{
# This example uses the GERMAPARLSAMPLE corpus rather than the full GERMAPARL 
# so that the time consumed for testing the code is not excessive. To apply
# everything on GERMAPARL rather than GERMAPARLSAMPLE, set variable 'samplemode' 
# to FALSE, or simply omit argument 'sample'.

samplemode <- TRUE
corpus_id <- "GERMAPARLSAMPLE" # to get/amend full corpus: corpus_id <- "GERMAPARL"

# This example assumes that the directories used by the CWB do not yet exist, so
# temporary directories are created.
cwb_dirs <- cwbtools::create_cwb_directories(prefix = tempdir(), ask = interactive())
registry_tmp <- cwb_dirs[["registry_dir"]]

# Download corpus from Zenodo
germaparl_download_corpus(
  registry_dir = registry_tmp,
  corpus_dir = cwb_dirs[["corpus_dir"]],
  verbose = FALSE,
  sample = samplemode
)

# Check availability of the corpus
library(polmineR)
corpus() # should include downloaded GERMAPARL(SAMPLE) corpus
count("GERMAPARLSAMPLE", "Daten") # an arbitrary test
germaparl_is_installed(sample = samplemode) # TRUE now
germaparl_get_version(sample = samplemode) # get version of indexed corpus
germaparl_get_doi(sample = samplemode) # get 'document object identifier' (DOI) of GERMAPARL corpus

# Encode structural attribute 'speech'
germaparl_encode_speeches(
  registry_dir = cwb_dirs[["registry_dir"]],
  corpus_dir = cwb_dirs[["corpus_dir"]],
  sample = samplemode
)

# Check whether the new attribute is available 
s_attributes(corpus_id)
values <- s_attributes(corpus_id, "speech")
sizes <- size(corpus_id, s_attribute = "speech")
dtm <- as.DocumentTermMatrix(corpus_id, p_attribute = "word", s_attribute = "speech")

# Download topic model (k = 250)
germaparl_download_lda(
  k = 30, # use k = 250 for full GERMAPARL corpus
  data_dir = file.path(cwb_dirs[["corpus_dir"]], tolower(corpus_id)),
  sample = samplemode
)
lda <- germaparl_load_topicmodel(k = 30L, registry_dir = registry_tmp, sample = samplemode)
lda_terms <- topicmodels::terms(lda, 10)

# Encode topic model classification of speeches
germaparl_encode_lda_topics(
  k = 30, # use k = 250 for full GERMAPARL corpus
  n = 3,
  registry_dir = cwb_dirs[["registry_dir"]],
  data_dir = file.path(cwb_dirs[["corpus_dir"]], tolower(corpus_id)),
  sample = samplemode
)

# Check whether the newly encoded attribute 'topics' is available
s_attributes(corpus_id)
sc <- corpus(corpus_id) \%>\% subset(grep("\\\\|18\\\\|", topics))
b <- as.speeches(sc, s_attribute_name = "speaker")
length(b)
}
\references{
Blaette, Andreas (2018): "Using Data Packages to Ship Annotated
  Corpora of Parliamentary Protocols: The GermaParl R Package". ISBN
  979-10-95546-02-3. Available online at
  \url{http://lrec-conf.org/workshops/lrec2018/W2/pdf/15_W2.pdf}.
}
\seealso{
Useful links:
\itemize{
  \item \url{https://github.com/polmine/GermaParl}
  \item Report bugs at \url{https://github.com/polmine/GermaParl/issues}
}

}
\author{
Andreas Blaette \email{andreas.blaette@uni-due.de}
}
\keyword{package}
