% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/function_AnnualRegime.R
\name{AnnualRegime}
\alias{AnnualRegime}
\title{Calculate annual regimes}
\usage{
AnnualRegime(
  x,
  stat = c("mean", "sum"),
  ts.in = NULL,
  ts.out = NULL,
  start.mon = 1,
  incl.leap = FALSE,
  na.rm = TRUE,
  format = c("list", "long")
)
}
\arguments{
\item{x}{Data frame, with column-wise equally-spaced time series. Date-times in \code{\link{POSIXct}} format in first column.
Typically an imported basin or time output file from HYPE. See details.}

\item{stat}{Character string, either \code{"mean"} or \code{"sum"}. Defines the type of aggregation to be computed for output
time periods, see Details. Defaults to \code{"mean"}.}

\item{ts.in}{Character string, timestep of \code{x}, attribute \code{timestep} in \code{x} per default.
Otherwise one of \code{"month"}, \code{"week"}, \code{"day"}, or \code{"nhour"} (n = number of hours).}

\item{ts.out}{Character string, timestep for results, defaults to \code{ts.in}. This timestep must be equal to or longer than
\code{ts.in}.}

\item{start.mon}{Integer between 1 and 12, starting month of the hydrological year, used to order the output.}

\item{incl.leap}{Logical, leap days (Feb 29) are removed from results per default, set to \code{TRUE} to keep them. Applies
to daily and shorter time steps only.}

\item{na.rm}{Logical, indicating if \code{NA} values should be stripped before averages are calculated.}

\item{format}{Character string. Output format, \code{list} (default) or \code{long}. See Value.}
}
\value{
If argument \code{format} is \code{list}, \code{AnnualRegime} returns a list with 8 elements and two additional \code{\link[=attributes]{attributes()}}. Each list element contains a
named data frame with aggregated annual regime data: arithmetic means, medians, minima, maxima, and 5\\%, 25\\%, 75\\%, and 95\\%
percentiles.

Each data frames contains, in column-wise order: reference dates in \code{POSIXct} format, date information as string, and aggregated
variables found in \code{x}.

Reference dates are given as dates in either 1911, 1912, or 1913 (just because a leap day and outer weeks '00'/'53' occur during
these years) and can be used for plots starting at the beginning of the hydrological year (with axis annotations set to months only).
Daily and hourly time steps are given as is, weekly time steps are given as mid-week dates (Wednesday), monthly time steps as
mid month dates (15th).

If argument \code{format} is \code{long}, \code{AnnualRegime} returns a four-column data frame with one value per row, and all variable information aligned
with the values. Columns in the data frame: \code{id} with SUBIDs or HYPE variable IDs, \code{month/week/day} with aggregation time steps, \code{name} with
short names of regime data (means, medians, minima, maxima, percentiles), and \code{value} with the variable value.

Attribute \code{period} contains a two-element POSIXct vector containing start and end dates of the
source data. Attribute \code{timestep} contains a timestep keyword corresponding to function argument \code{ts.out}.
}
\description{
Calculate annual regimes based on long-term time series, typically imported HYPE basin output and time output result files.
}
\details{
\code{AnnualRegime} uses \code{\link{aggregate}} to calculate long-term average regimes for all data columns provided in \code{x},
including long-term arithmetic means, medians, minima and maxima, and 5\\%, 25\\%, 75\\%, and 95\\% percentiles. With HYPE result files,
\code{AnnualRegime} is particularly applicable to basin and time output files imported using \code{\link{ReadBasinOutput}} and
\code{\link{ReadTimeOutput}}. The function does not check if equally spaced time steps are provided in \code{x} or if the
overall time period in \code{x} covers full years so that the calculated averages are based on the same number of values.

Values within each output time period can be aggregated either by arithmetic means or by sums within each period, e.g. typically
means for temperatures and sums for precipitation. Long-term aggregated values are always computed as arithmetic means.
}
\note{
If weekly data are provided in \code{x}, \code{AnnualRegime} will inflate \code{x} to daily time steps before computing
results. Values in \code{x} will be assigned to the preceeding week days, corresponding to HYPE file output, where weekly
values are conventionally printed on the last day of the week. If \code{NA} values are present in the original weekly data,
these will be filled with the next available value as a side effect of the inflation.

If weekly output time steps are computed in combination with a user-defined start month, the function will round up weeks to
determine the first week of the hydrological year. Weeks are identified using Monday as first day of the week and the first Monday
of the year as day 1 of week 1 (see conversion code \code{\%W} in \code{\link{strptime}}). Boundary weeks \code{'00'} and
\code{'53'} are merged to week \code{'00'} prior to average computations.
}
\examples{
# Source data, HYPE basin output with a number of result variables
te <- ReadBasinOutput(filename = system.file("demo_model", "results", "0003587.txt", 
package = "HYPEtools"))
# Daily discharge regime, computed and observed, hydrologigical year from October 
AnnualRegime(te[, c("DATE", "COUT", "ROUT")], ts.in = "day", start.mon = 10)
# Id., aggregated to weekly means
AnnualRegime(te[, c("DATE", "COUT", "ROUT")], ts.in = "day", ts.out = "week", start.mon = 10)
# Long format, e.g. for subsequent plotting with ggplot
AnnualRegime(te[, c("DATE", "COUT", "ROUT")], ts.in = "day", ts.out = "week", format = "long", 
start.mon = 10)
# Precipitation regime, monthly sums
AnnualRegime(te[, c("DATE", "UPCPRC")], ts.in = "day", ts.out = "month", stat = "sum")

}
\seealso{
\code{\link{PlotAnnualRegime}}
}
