\name{dhyperb}
\alias{dhyperb}
\alias{phyperb}
\alias{qhyperb}
\alias{rhyperb}

\title{Hyperbolic Distribution}

\description{
  Density function, distribution function, quantiles and 
  random number generation for the hyperbolic distribution 
  with parameter vector \code{theta}.
}

\usage{
dhyperb(x, theta, K.nu=NULL, log.pars=FALSE)
phyperb(q, theta, tol=10^(-5), subdivisions=100)
qhyperb(p, theta, tol=10^(-5), n.interpol=100, subdivisions=100, ...)
rhyperb(n, theta)
}

\arguments{
  \item{x,q}{vector of quantiles}
  \item{p}{vector of probabilities}
  \item{n}{number of observations to be generated}
  \item{theta}{parameter vector taking the form \code{c(pi,zeta,delta,mu)}}
  \item{K.nu}{sets the value of the Bessel function in the density. 
    See \bold{Details}}
  \item{log.pars}{logical; if \code{TRUE} the second and third components 
    of theta are taken to be log(zeta) and log(delta) respectively}
  \item{tol}{an upper bound on the error in the distribution function. 
    See \bold{Details}}
  \item{subdivisions}{the maximum number of subdivisions used to integrate 
    the density returning the distribution function}
  \item{n.interpol}{the number of points used in qhyperb for cubic spline 
    interpolation (see \code{splinefun}) of the distribution function}
  \item{\dots}{passes arguments to \code{uniroot}.  See \bold{Details}}
}

\details{
  The hyperbolic distribution has density

  \deqn{f(x)=\frac{1}{2\sqrt{1+\pi^2}K_1(\zeta)} %
    e^{-\zeta[\sqrt{1+\pi^2}\sqrt{1+(\frac{x-\mu}{\delta})^2}-%
    \pi\frac{x-\mu}{\delta}]}}{ %
  f(x)=1/(2 sqrt(1+pi^2) K_1(zeta)) %
  exp(-zeta(sqrt(1+pi^2) sqrt(1+((x-mu)/delta)^2)-pi (x-mu)/delta))}

  where \deqn{K_1()}{K_1()} is the modified Bessel function of the 
  third kind with order 1.

  A succinct description of the hyperbolic distribution is given in
  Barndorff-Nielsen and Blaesild (1983). Three different possible
  parameterisations are described in that paper. All use location and
  scale parameters \eqn{\mu}{mu} and \eqn{\delta}{delta}. There are two
  other parameters in each case.

  Use \code{hyperb.change.pars} to convert from the alpha/beta or phi/gamma 
  parameterisations to the parameterisation (pi/zeta) used above.

  \code{phyperb} uses a numerical integration routine (\code{integrate()}) 
  to integrate the density \code{dhyperb}.  See the notes on \code{integrate} 
  for the shortfalls of this procedure.  In particular note the weakness of 
  this function when integrating over a region that is approximately zero.

  \code{qhyperb} uses \code{splinefun} on values generated by \code{pyhyperb} 
  to approximate the distribution function.  The quantiles are then found 
  using the \code{uniroot} function.

  \code{rhyperb} is based on the HYP algorithm given by Atkinson (1982).
}

\value{
  \code{dhyperb} gives the density, \code{phyperb} gives the distribution 
  function, \code{qhyperb} gives the quantile function and \code{rhyperb} 
  generates random variates.
}

\references{
  Atkinson, A.C. (1982). 
  The simulation of generalized inverse Gaussian and hyperbolic random
  variables. 
  \emph{SIAM J. Sci. Stat. Comput.}, 
  \bold{3}, 502--515.

  Barndorff-Nielsen, O. and Blaesild, P (1983).
  Hyperbolic distributions. 
  In \emph{Encyclopedia of Statistical Sciences}, 
  eds., Johnson, N. L., Kotz, S. and Read, C. B., Vol. 3,
  pp. 700--707. New York: Wiley.
}

\author{David Scott \email{d.scott@auckland.ac.nz}, 
Ai-Wei Lee, Jennifer Tso, Richard Trendall}

\seealso{
  \code{\link{integrate}} for its shortfalls, \code{\link{splinefun}}, 
  \code{\link{uniroot}} and \code{\link{hyperb.change.pars}} for 
  changing parameters to the pi/zeta parameterisation
}

\examples{
theta <- c(0,1,1,0)
data.vector <- rhyperb(500,theta)
par(mfrow=c(1,2))
hist(data.vector,freq=FALSE)
curve(dhyperb(x,theta),add=TRUE,
      range(data.vector)[1],range(data.vector)[2])
log.hist(data.vector)
curve(log(dhyperb(x,theta)),add=TRUE,
      range(data.vector)[1],range(data.vector)[2])
hyperb.range <- calculate.range(c(0,1,1,0),tol=10^(-2))
curve(dhyperb(x,theta),
      hyperb.range[1],hyperb.range[2],log="")
title("Density of the\n Hyperbolic Distribution")
curve(phyperb(x,theta),
      hyperb.range[1],hyperb.range[2],log="")
title("Distribution Function of the\n Hyperbolic Distribution")
}

\keyword{distribution} 